/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: BUSL-1.1
 */

function parseRelativeDate(relativeDate) {
  const match = relativeDate.match(/^(-?\d+)h$/);
  if (!match) {
    throw new Error(`Invalid relative date format: ${relativeDate}`);
  }
  const hours = parseInt(match[1], 10);
  const date = new Date();
  date.setHours(date.getHours() + hours);
  if (hours > 0) {
    date.setMinutes(59, 59, 999); // Set to end of the hour for future dates
  } else {
    date.setMinutes(0, 0, 0); // Set to start of the hour for past dates
  }
  return date;
}

export function getFilteredSecrets(schema, request) {
  const url = new URL(request.url);
  const params = url.searchParams;
  // Add support for one basic filter just to be able to see data change
  let { data } = this.serialize(schema['vaultReporting.secrets'].all());

  const paramKeys = Array.from(new Set(params.keys()));

  const keySwaps = {
    secret_names: 'secret_name',
    secret_types: 'secret_type',
    namespaces: 'namespace',
    mount_paths: 'mount_path',
    created_by_ids: 'created_by_id',
    secret_last_accessed_by_ids: 'secret_last_accessed_by_id',
    secret_last_modified_by_ids: 'secret_last_modified_by_id',
    secret_deleted_by_ids: 'secret_deleted_by_id',
  };

  paramKeys.forEach((key) => {
    // handle basic non-date equality filters (dates will be filtered out due to naming convention)
    if (
      key.startsWith('filter.') &&
      !key.endsWith('include_deleted') &&
      !key.endsWith('.operator') &&
      !key.endsWith('.absolute') &&
      !key.endsWith('.relative_to_now')
    ) {
      let field = key.replace('filter.', '');
      if (keySwaps[field]) {
        field = keySwaps[field];
      }
      const value = params.getAll(key).filter((v) => !!v);
      if (value) {
        data = data.filter((item) => {
          return value.includes(item[field]);
        });
      }
      // handle include_deleted
    } else if (key.startsWith('filter.') && key.endsWith('.operator')) {
      const fieldKey = key.replace('.operator', '').replace('filter.', '');
      const operator = params.get(key);
      const filterValue = parseRelativeDate(
        params.get(`filter.${fieldKey}.relative_to_now`),
      );

      if (operator && filterValue) {
        data = data.filter((item) => {
          const itemValue = new Date(item[fieldKey]);
          switch (operator) {
            case 'GREATER_THAN':
              return itemValue > filterValue;
            case 'GREATER_THAN_OR_EQUAL':
              return itemValue >= filterValue;
            case 'LESS_THAN':
              return itemValue < filterValue;
            case 'LESS_THAN_OR_EQUAL':
              return itemValue <= filterValue;
            default:
              console.warn('unknown operator', operator);
              return false;
          }
        });
      }
    }
  });

  // filter out any deleted secrets unless include_deleted is true
  const includeDeleted = params.get('filter.include_deleted') === 'true';
  if (!includeDeleted) {
    data = data.filter((item) => !item.secret_deleted_at);
  }

  if (params.has('sorting.order_by')) {
    const orderBy = params.get('sorting.order_by');
    const [field, direction] = orderBy.split(' ');

    data = data.sort((a, b) => {
      if (direction === 'asc') {
        return a[field] > b[field] ? 1 : -1;
      } else {
        return a[field] < b[field] ? 1 : -1;
      }
    });
  }
  return { data };
}

export function getSecretsInventory(schema, request) {
  const url = new URL(request.url);
  const params = url.searchParams;
  const { data } = getFilteredSecrets.call(this, schema, request);
  const pageSize = parseInt(params.get('pagination.page_size'), 10) || 10;
  const nextPageToken = parseInt(params.get('pagination.next_page_token'), 10);
  const previousPageToken = parseInt(
    params.get('pagination.previous_page_token'),
    10,
  );

  // Apply pagination
  if (isNaN(pageSize) || pageSize <= 0) {
    throw new Error(`Invalid page size: ${pageSize}`);
  }
  if (nextPageToken < 0 || previousPageToken < 0) {
    throw new Error(
      `Invalid page tokens: nextPageToken=${nextPageToken}, previousPageToken=${previousPageToken}`,
    );
  }

  let startIndex;
  let pageData = data;

  if (nextPageToken) {
    startIndex = nextPageToken;
  } else if (previousPageToken) {
    startIndex = previousPageToken - pageSize - 1;
  } else {
    startIndex = 0;
  }

  pageData = data.slice(startIndex, startIndex + pageSize);

  return {
    data: pageData,
    report_as_of: new Date().toISOString(),
    pagination: {
      next_page_token:
        startIndex + pageSize < data.length
          ? String(startIndex + pageSize)
          : null,
      previous_page_token: startIndex > 0 ? String(startIndex + 1) : null,
    },
  };
}

export function getSecretsInventoryMetadata(schema, request) {
  const { data } = getFilteredSecrets.call(this, schema, request);
  return {
    count: data.length,
  };
}

function getDistinctFields(
  schema,
  request,
  search,
  fieldKey,
  responseKey,
  collectionKey,
) {
  const { data } = getFilteredSecrets.call(this, schema, request);

  let fields = data.map((item) => item[fieldKey]);
  if (search) {
    fields = fields.filter((item) => item.includes(search.trim()));
  }
  // Return first 10 results to approximate getting pagination and only getting the first page of data
  const uniqueFields = Array.from(new Set(fields)).slice(0, 10);

  return {
    [collectionKey || `${fieldKey}s`]: uniqueFields.map((field) => ({
      [responseKey || fieldKey]: field,
    })),
  };
}

export function getDistinctNamespaces(schema, request) {
  const url = new URL(request.url);
  const params = url.searchParams;
  const namespacePathPrefix = params.get('namespace_path_prefix');
  return getDistinctFields.call(
    this,
    schema,
    request,
    namespacePathPrefix,
    'namespace',
    'path',
    'namespaces',
  );
}

export function getDistinctMounts(schema, request) {
  const url = new URL(request.url);
  const params = url.searchParams;
  const mountPathPrefix = params.get('mount_path_prefix');
  return getDistinctFields.call(
    this,
    schema,
    request,
    mountPathPrefix,
    'mount_path',
    'mount_path',
    'mounts',
  );
}
