
import { HdsPaginationCompact, HdsTextBody, HdsIcon, HdsAdvancedTable, HdsLinkStandalone, HdsTextDisplay, HdsCardContainer, HdsAccordion } from '@hashicorp/design-system-components/components';
import { or } from '../../utils/helpers.js';
import { hash, fn } from '@ember/helper';
import { on } from '@ember/modifier';
import FilterBar from './filter-bar.js';
import Component from '@glimmer/component';
import { service } from '@ember/service';
import { precompileTemplate } from '@ember/template-compilation';
import { setComponentTemplate } from '@ember/component';
import { g, i } from 'decorator-transforms/runtime';

/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: BUSL-1.1
 */
class InventoryTable extends Component {
  static {
    g(this.prototype, "router", [service]);
  }
  #router = (i(this, "router"), void 0);
  static {
    g(this.prototype, "reportingAnalytics", [service]);
  }
  #reportingAnalytics = (i(this, "reportingAnalytics"), void 0);
  trackAnalyticsEvent = (eventName, properties) => {
    const name = `${this.args.analyticsPrefix || ''}${eventName}`;
    this.reportingAnalytics.trackEvent(name, properties);
  };
  handleApplyFilters = filters => {
    const existingFilterKeys = (this.args.appliedFilters || []).map(f => f.field);
    const newFilterKeys = filters.map(f => f.field);
    const addedFilters = newFilterKeys.filter(key => !existingFilterKeys.includes(key));
    const removedFilters = existingFilterKeys.filter(key => !newFilterKeys.includes(key));
    this.args.onFilterApplied(filters);
    this.trackAnalyticsEvent('inventory_table_filtered', {
      addedFilters,
      removedFilters
    });
  };
  handleSort = (sortBy, sortOrder) => {
    if (this.args.onSortApplied) {
      this.args.onSortApplied([`${sortBy} ${sortOrder}`]);
      this.trackAnalyticsEvent('inventory_table_sorted', {
        sortBy,
        sortOrder
      });
    }
  };
  handlePageChange = direction => {
    // construct the pagination object based on the direction
    let pagination;
    if (direction === 'next') {
      pagination = {
        next_page_token: this.args.nextPageToken,
        page_size: this.args.pageSize || 10
      };
    } else {
      pagination = {
        previous_page_token: this.args.previousPageToken,
        page_size: this.args.pageSize || 10
      };
    }
    this.args.onPageChange(pagination);
  };
  handlePageSizeChange = pageSize => {
    if (this.args.onPageSizeChange) {
      this.args.onPageSizeChange(pageSize);
    } else {
      const pagination = {
        page_size: pageSize
      };
      this.args.onPageChange(pagination);
    }
  };
  handleColumnsChanged = visibleColumnKeys => {
    if (this.args.onColumnsChanged) {
      this.args.onColumnsChanged(visibleColumnKeys);
    }
    this.trackAnalyticsEvent('inventory_table_columns_changed', {
      visibleColumnKeys
    });
  };
  get isDisabledNext() {
    return !this.args.nextPageToken || this.args.isLoading;
  }
  get isDisabledPrev() {
    return !this.args.previousPageToken || this.args.isLoading;
  }
  get routeName() {
    return this.router.currentRouteName || '';
  }
  getCellContent = (row = {}, key, formatter) => {
    let value = row?.[key || ''];
    if (value === undefined) return '';
    if (formatter) {
      try {
        value = formatter(row);
      } catch (error) {
        console.error('Error formatting cell content:', error);
      }
    }
    return value ?? '';
  };
  getParams = quickFilter => {
    const params = quickFilter?.getParams();
    return params;
  };
  hasCount = quickFilter => {
    return quickFilter?.count !== undefined && quickFilter?.count !== null && !isNaN(quickFilter.count);
  };
  get sortBy() {
    const sort = this.args?.appliedSort?.[0];
    if (!sort) return {
      key: '',
      order: 'asc'
    };
    const [key, order] = sort.split(' ');
    if (!order || !['asc', 'desc'].includes(order)) {
      return {
        key,
        order: 'asc'
      };
    }
    return {
      key,
      order
    };
  }
  get rows() {
    if (this.args.isLoading || this.args.isError || this.args.isEmpty) {
      return [{}];
    }
    return this.args.rows || [];
  }
  get hasMessage() {
    return this.args.isLoading || this.args.isError || this.args.isEmpty;
  }
  get columnsForFilterBar() {
    const visibleKeys = this.args.visibleColumnKeys || this.args.columns.map(col => col.key);
    const result = this.args.columns.map(col => ({
      label: col.label,
      key: col.key,
      visible: visibleKeys.includes(col.key)
    }));
    return result;
  }
  get visibleColumns() {
    const visibleKeys = this.args.visibleColumnKeys || this.args.columns.map(col => col.key);
    return this.args.columns.filter(col => visibleKeys.includes(col.key));
  }
  static {
    setComponentTemplate(precompileTemplate("\n    <div class=\"inventory\" ...attributes>\n\n      {{#if @quickFilters}}\n        <HdsAccordion class=\"inventory__quick-filters-accordion\" as |A|>\n          <A.Item @containsInteractive={{true}} @isOpen={{true}}>\n            <:toggle>\n              Quick filters\n            </:toggle>\n\n            <:content>\n              <div>Click a quick filter below to apply multiple filters at once\n                and identify secrets needing attention.</div>\n\n              <div class=\"inventory__quick-filters\">\n                {{#each @quickFilters as |quickFilter|}}\n                  <HdsCardContainer class=\"inventory__quick-filter\" @hasBorder={{true}} {{!-- @glint-expect-error --}} @level=\"mid\" data-test-vault-reporting-inventory-table-quick-filter={{quickFilter.label}}>\n                    <div class=\"inventory__quick-filter-header\">\n                      <HdsTextDisplay @tag=\"h2\" @size=\"300\" @weight=\"medium\" @color=\"medium\" data-test-vault-reporting-inventory-table-quick-filter-title>\n                        {{quickFilter.label}}\n                      </HdsTextDisplay>\n\n                    </div>\n\n                    <div class=\"inventory__quick-filter-description\">{{quickFilter.description}}</div>\n\n                    <HdsTextDisplay @tag=\"p\" @size=\"300\" data-test-vault-reporting-inventory-table-quick-filter-count class=\"inventory__quick-filter-text\">\n                      {{#if (this.hasCount quickFilter)}}\n                        <span class=\"inventory__quick-filter-value\">{{quickFilter.count}}</span>\n                        /\n                        {{@totalUnfilteredCount}}\n                      {{else}}\n                        - / -\n                      {{/if}}\n                    </HdsTextDisplay>\n\n                    <HdsLinkStandalone @icon=\"filter\" @iconPosition=\"leading\" @text=\"Apply filter\" @color=\"secondary\" @route={{this.routeName}} @query={{this.getParams quickFilter}} class=\"inventory__quick-filter-link\" {{on \"click\" (fn this.trackAnalyticsEvent \"quick_filter_applied\" (hash filter=quickFilter.label))}} data-test-vault-reporting-inventory-table-quick-filter-apply />\n                  </HdsCardContainer>\n                {{/each}}\n              </div>\n\n            </:content>\n          </A.Item>\n        </HdsAccordion>\n      {{/if}}\n\n      <FilterBar @onFiltersApplied={{this.handleApplyFilters}} @appliedFilters={{@appliedFilters}} @filterFieldDefinitions={{@filterFieldDefinitions}} @columns={{this.columnsForFilterBar}} @onColumnsChanged={{this.handleColumnsChanged}} />\n\n      <HdsAdvancedTable @columns={{this.visibleColumns}} @model={{this.rows}} @onSort={{this.handleSort}} @sortBy={{this.sortBy.key}} @sortOrder={{this.sortBy.order}} @hasResizableColumns={{true}} class={{if this.hasMessage \"inventory__table--has-message\"}} data-test-vault-reporting-inventory-table>\n        <:body as |B|>\n          {{!-- @glint-expect-error --}}\n          <B.Tr data-test-vault-reporting-inventory-table-row>\n            {{#if @isLoading}}\n              <B.Td>\n                <div class=\"inventory__loading\">\n                  <HdsIcon @name=\"loading\" @size=\"24\" @color=\"primary\" />\n                  <HdsTextDisplay>Loading...</HdsTextDisplay>\n                </div>\n              </B.Td>\n            {{else if @isError}}\n              <B.Td>\n                <div class=\"inventory__error\">\n                  <HdsIcon @name=\"alert-circle\" @size=\"24\" @color=\"primary\" />\n                  <HdsTextDisplay>Error fetching data!</HdsTextDisplay>\n                </div>\n              </B.Td>\n            {{else if @isEmpty}}\n              <B.Td>\n                <div class=\"inventory__empty\">\n                  <HdsIcon @name=\"file-minus\" @size=\"24\" @color=\"primary\" />\n                  <HdsTextDisplay>No results.</HdsTextDisplay>\n                </div>\n              </B.Td>\n            {{else}}\n              {{#each this.visibleColumns as |column|}}\n                <B.Td class=\"inventory__cell\" data-test-vault-reporting-inventory-table-cell={{column.key}}>\n                  <HdsTextBody class=\"inventory__cell__primary-text\" @size=\"200\">\n                    {{or (this.getCellContent B.data column.key column.cellFormatter) \"-\"}}\n                  </HdsTextBody>\n                  {{#if column.secondaryKey}}\n                    <HdsTextBody class=\"inventory__cell__secondary-text\" @size=\"100\" @color=\"faint\">\n                      {{this.getCellContent B.data column.secondaryKey column.secondaryCellFormatter}}\n                    </HdsTextBody>\n                  {{/if}}\n                </B.Td>\n              {{/each}}\n            {{/if}}\n          </B.Tr>\n        </:body>\n      </HdsAdvancedTable>\n      <HdsPaginationCompact class=\"vault-reporting-inventory-table__pagination-bar\" @onPageChange={{this.handlePageChange}} @isDisabledPrev={{this.isDisabledPrev}} @isDisabledNext={{this.isDisabledNext}} @showSizeSelector={{true}} @onPageSizeChange={{this.handlePageSizeChange}} />\n    </div>\n  ", {
      strictMode: true,
      scope: () => ({
        HdsAccordion,
        HdsCardContainer,
        HdsTextDisplay,
        HdsLinkStandalone,
        on,
        fn,
        hash,
        FilterBar,
        HdsAdvancedTable,
        HdsIcon,
        HdsTextBody,
        or,
        HdsPaginationCompact
      })
    }), this);
  }
}

export { InventoryTable as default };
//# sourceMappingURL=inventory-table.js.map
