/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: BUSL-1.1
 */

import { Factory } from 'miragejs';
import { faker } from '@faker-js/faker';

// Seed the faker library to ensure consistent data generation
// Add preceding numbers to secret names to help with testing
faker.seed(12345);
let secretCounter = 0;

const allUsers = new Array(15).fill(null).map(() => crypto.randomUUID());

const getPath = () => {
  return `/${faker.hacker.noun()}/${faker.hacker.noun()}`;
};
export default {
  'vaultReporting.secret': Factory.extend({
    secret_type: () =>
      faker.helpers.arrayElement(['static', 'dynamic', 'auto-rotating']),
    namespace: () => getPath(),
    mount_path: () => getPath(),
    created_at: () => faker.date.recent({ days: 120 }),
    created_by_id: () => faker.helpers.arrayElement(allUsers),
    config_last_modified_by_id: () => faker.helpers.arrayElement(allUsers),
    config_last_accessed_by_id: () => faker.helpers.arrayElement(allUsers),
    secret_last_modified_by_id: () => faker.helpers.arrayElement(allUsers),
    secret_last_accessed_by_id: () => faker.helpers.arrayElement(allUsers),
    version_count: () => faker.number.int({ min: 1, max: 5 }),
    afterCreate(secret) {
      const updates = {
        secret_name:
          `${secretCounter++}-${faker.hacker.adjective()}-${faker.hacker.noun()}`.replace(
            /\s+/g,
            '-',
          ),
      };

      if (secret.secret_type === 'dynamic') {
        updates.plugin_name = faker.helpers.arrayElement(['database', 'aws']);
      } else if (secret.secret_type === 'static') {
        updates.plugin_name = faker.helpers.arrayElement(['kv-v1', 'kv-v2']);
      } else if (secret.secret_type === 'auto-rotating') {
        updates.plugin_name = 'kv-v2';
        updates.next_rotation_at = faker.date.soon({ days: 120 });
      }
      updates.secret_last_accessed_at = faker.date.between({
        from: secret.created_at,
        to: new Date(),
      });

      updates.secret_last_modified_at = faker.date.between({
        from: secret.created_at,
        to: new Date(),
      });

      if (!updates.ttl) {
        updates.ttl = faker.helpers.arrayElement([
          '3600s', // 1 hour
          '86400s', // 1 day
          '604800s', // 1 week
          '2592000s', // 30 days
          null,
        ]);
      }

      if (faker.datatype.boolean({ probability: 0.5 })) {
        updates.secret_deleted_at = faker.date.between({
          from: secret.created_at,
          to: new Date(),
        });
        updates.secret_deleted_by_id = faker.helpers.arrayElement(allUsers);
      }

      updates.config_last_modified_at = faker.date.between({
        from: secret.created_at,
        to: new Date(),
      });

      updates.config_last_accessed_at = faker.date.between({
        from: secret.created_at,
        to: new Date(),
      });

      secret.update(updates);
    },
  }),
};
