/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: BUSL-1.1
 */

import { module, test } from 'qunit';
import { setupRenderingTest } from 'vault/tests/helpers';
import { render } from '@ember/test-helpers';
import { hbs } from 'ember-cli-htmlbars';

import { setupMirage } from 'ember-cli-mirage/test-support';

import { GENERAL } from 'vault/tests/helpers/general-selectors';

module('Integration | Component | recovery/snapshots', function (hooks) {
  setupRenderingTest(hooks);
  setupMirage(hooks);

  hooks.beforeEach(function () {
    this.model = {
      snapshots: [],
      canLoadSnapshot: false,
    };
    this.renderComponent = () => render(hbs`<Recovery::Page::Snapshots @model={{this.model}}/>`);
  });

  test('it displays empty state in CE', async function (assert) {
    this.model = { snapshots: [], showCommunityMessage: true };

    await render(hbs`<Recovery::Page::Snapshots @model={{this.model}}/>`);
    assert
      .dom(GENERAL.emptyStateTitle)
      .hasText('Secrets Recovery is an enterprise feature', 'CE empty state title renders');
    assert
      .dom(GENERAL.emptyStateMessage)
      .hasText(
        'Secrets Recovery allows you to restore accidentally deleted or lost secrets from a snapshot. The snapshots can be provided via upload or loaded from external storage.',
        'CE empty state message renders'
      );
    assert
      .dom(GENERAL.emptyStateActions)
      .hasText('Learn more about upgrading', 'CE empty state action renders');

    assert.dom(GENERAL.badge('enterprise')).exists();
  });

  test('it displays empty state in non root namespace', async function (assert) {
    const nsService = this.owner.lookup('service:namespace');
    nsService.setNamespace('test-ns');

    await this.renderComponent();

    assert
      .dom(GENERAL.emptyStateTitle)
      .hasText('Snapshot upload is restricted', 'non root namespace empty state title renders');
    assert
      .dom(GENERAL.emptyStateMessage)
      .hasText(
        'Snapshot uploading is only available in root namespace. Please navigate to root and upload your snapshot. ',
        'non root empty state message renders'
      );
    assert
      .dom(GENERAL.emptyStateActions)
      .hasText('Take me to root namespace', 'non root empty state action renders');
  });

  test('it displays empty state when user cannot load snapshot', async function (assert) {
    await this.renderComponent();
    assert.dom(GENERAL.emptyStateTitle).hasText('No snapshot available', 'empty state title renders');
    assert
      .dom(GENERAL.emptyStateMessage)
      .hasText(
        'Ready to restore secrets? Please contact your admin to either upload a snapshot or grant you uploading permissions to get started.',
        'empty state message renders'
      );
    assert
      .dom(GENERAL.emptyStateActions)
      .hasText('Learn more about Secrets Recovery', 'empty state action renders');
  });

  test('it displays empty state when user can load snapshot', async function (assert) {
    this.model.canLoadSnapshot = true;
    await this.renderComponent();
    assert
      .dom(GENERAL.emptyStateTitle)
      .hasText('Upload a snapshot to get started', 'empty state title renders');
    assert
      .dom(GENERAL.emptyStateMessage)
      .hasText(
        'Secrets Recovery allows you to restore accidentally deleted or lost secrets from a snapshot. The snapshots can be provided via upload or loaded from external storage.',
        'empty state message renders'
      );
    assert.dom(GENERAL.emptyStateActions).hasText('Upload snapshot', 'empty state action renders');
  });
});
