import type { Application } from "../application.js";
import type { Theme } from "./theme.js";
import { IndexEvent, type MarkdownEvent, PageEvent, RendererEvent } from "./events.js";
import type { ProjectReflection } from "../models/ProjectReflection.js";
import { AbstractComponent } from "../utils/index.js";
import type { Comment, Reflection } from "../models/index.js";
import type { DefaultThemeRenderContext } from "./themes/default/DefaultThemeRenderContext.js";
import { EventHooks, JSX } from "#utils";
import { MarkedPlugin } from "./plugins/index.js";
import { type Router } from "./router.js";
/**
 * Describes the hooks available to inject output in the default theme.
 * If the available hooks don't let you put something where you'd like, please open an issue!
 */
export interface RendererHooks {
    /**
     * Applied immediately after the opening `<head>` tag.
     */
    "head.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately before the closing `</head>` tag.
     */
    "head.end": [DefaultThemeRenderContext];
    /**
     * Applied immediately after the opening `<body>` tag.
     */
    "body.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately before the closing `</body>` tag.
     */
    "body.end": [DefaultThemeRenderContext];
    /**
     * Applied immediately before the main template.
     */
    "content.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately after the main template.
     */
    "content.end": [DefaultThemeRenderContext];
    /**
     * Applied immediately before calling `context.sidebar`.
     */
    "sidebar.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately after calling `context.sidebar`.
     */
    "sidebar.end": [DefaultThemeRenderContext];
    /**
     * Applied immediately before calling `context.pageSidebar`.
     */
    "pageSidebar.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately after calling `context.pageSidebar`.
     */
    "pageSidebar.end": [DefaultThemeRenderContext];
    /**
     * Applied immediately before the "Generated by TypeDoc" link in the footer.
     */
    "footer.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately after the "Generated by TypeDoc" link in the footer.
     */
    "footer.end": [DefaultThemeRenderContext];
    /**
     * Applied immediately before a comment's tags are rendered.
     *
     * This may be used to set {@link Models.CommentTag.skipRendering} on any tags which
     * should not be rendered.
     */
    "comment.beforeTags": [DefaultThemeRenderContext, Comment, Reflection];
    /**
     * Applied immediately after a comment's tags are rendered.
     *
     * This may be used to set {@link Models.CommentTag.skipRendering} on any tags which
     * should not be rendered as this hook is called before the tags are actually
     * rendered.
     */
    "comment.afterTags": [DefaultThemeRenderContext, Comment, Reflection];
}
export interface RendererEvents {
    beginRender: [RendererEvent];
    beginPage: [PageEvent];
    endPage: [PageEvent];
    endRender: [RendererEvent];
    parseMarkdown: [MarkdownEvent];
    prepareIndex: [IndexEvent];
}
/**
 * The renderer processes a {@link ProjectReflection} using a {@link Theme} instance and writes
 * the emitted html documents to a output directory. You can specify which theme should be used
 * using the `--theme <name>` command line argument.
 *
 * {@link Renderer} is a subclass of {@link EventDispatcher} and triggers a series of events while
 * a project is being processed. You can listen to these events to control the flow or manipulate
 * the output.
 *
 *  * {@link Renderer.EVENT_BEGIN}<br>
 *    Triggered before the renderer starts rendering a project. The listener receives
 *    an instance of {@link RendererEvent}.
 *
 *    * {@link Renderer.EVENT_BEGIN_PAGE}<br>
 *      Triggered before a document will be rendered. The listener receives an instance of
 *      {@link PageEvent}.
 *
 *    * {@link Renderer.EVENT_END_PAGE}<br>
 *      Triggered after a document has been rendered, just before it is written to disc. The
 *      listener receives an instance of {@link PageEvent}.
 *
 *  * {@link Renderer.EVENT_END}<br>
 *    Triggered after the renderer has written all documents. The listener receives
 *    an instance of {@link RendererEvent}.
 *
 * * {@link Renderer.EVENT_PREPARE_INDEX}<br>
 *    Triggered when the JavascriptIndexPlugin is preparing the search index. Listeners receive
 *    an instance of {@link IndexEvent}.
 *
 * @summary Writes HTML output from TypeDoc's models
 * @group None
 */
export declare class Renderer extends AbstractComponent<Application, RendererEvents> {
    private routers;
    private themes;
    /** @event */
    static readonly EVENT_BEGIN_PAGE = "beginPage";
    /** @event */
    static readonly EVENT_END_PAGE = "endPage";
    /** @event */
    static readonly EVENT_BEGIN = "beginRender";
    /** @event */
    static readonly EVENT_END = "endRender";
    /** @event */
    static readonly EVENT_PREPARE_INDEX = "prepareIndex";
    /**
     * A list of async jobs which must be completed *before* rendering output.
     * They will be called after {@link RendererEvent.BEGIN} has fired, but before any files have been written.
     *
     * This may be used by plugins to register work that must be done to prepare output files. For example: asynchronously
     * transform markdown to HTML.
     *
     * Note: This array is cleared after calling the contained functions on each {@link Renderer.render} call.
     */
    preRenderAsyncJobs: Array<(output: RendererEvent) => Promise<void>>;
    /**
     * A list of async jobs which must be completed after rendering output files but before generation is considered successful.
     * These functions will be called after all documents have been written to the filesystem.
     *
     * This may be used by plugins to register work that must be done to finalize output files. For example: asynchronously
     * generating an image referenced in a render hook.
     *
     * Note: This array is cleared after calling the contained functions on each {@link Renderer.render} call.
     */
    postRenderAsyncJobs: Array<(output: RendererEvent) => Promise<void>>;
    /**
     * The theme that is used to render the documentation.
     */
    theme?: Theme;
    /**
     * The router which is used to determine the pages to render and
     * how to link between pages.
     */
    router?: Router;
    /**
     * Hooks which will be called when rendering pages.
     * Note:
     * - Hooks added during output will be discarded at the end of rendering.
     * - Hooks added during a page render will be discarded at the end of that page's render.
     *
     * See {@link RendererHooks} for a description of each available hook, and when it will be called.
     */
    hooks: EventHooks<RendererHooks, JSX.Element>;
    /** @internal */
    private accessor themeName;
    /** @internal */
    private accessor routerName;
    private accessor cleanOutputDir;
    private accessor cname;
    private accessor githubPages;
    /** @internal */
    accessor cacheBust: boolean;
    private accessor pretty;
    renderStartTime: number;
    markedPlugin: MarkedPlugin;
    constructor(owner: Application);
    /**
     * Define a new theme that can be used to render output.
     * This API will likely be changing at some point, to allow more easily overriding parts of the theme without
     * requiring additional boilerplate.
     * @param name
     * @param theme
     */
    defineTheme(name: string, theme: new (renderer: Renderer) => Theme): void;
    /** @internal intended for test usage only */
    removeTheme(name: string): void;
    /**
     * Define a new router that can be used to determine the output structure.
     * @param name
     * @param router
     */
    defineRouter(name: string, router: new (app: Application) => Router): void;
    /** @internal intended for test usage only */
    removeRouter(name: string): void;
    /**
     * Render the given project reflection to the specified output directory.
     *
     * @param project  The project that should be rendered.
     * @param outputDirectory  The path of the directory the documentation should be rendered to.
     */
    render(project: ProjectReflection, outputDirectory: string): Promise<void>;
    private runPreRenderJobs;
    /**
     * Render a single page.
     *
     * @param page An event describing the current page.
     * @return TRUE if the page has been saved to disc, otherwise FALSE.
     */
    private renderDocument;
    private prepareRouter;
    private prepareTheme;
    /**
     * Prepare the output directory. If the directory does not exist, it will be
     * created. If the directory exists, it will be emptied.
     *
     * @param directory  The path to the directory that should be prepared.
     * @returns TRUE if the directory could be prepared, otherwise FALSE.
     */
    private prepareOutputDirectory;
}
