---
title: "Traefik Kubernetes Services Documentation"
description: "Learn how to configure routing and load balancing in Traefik Proxy to reach Services, which handle incoming requests. Read the technical documentation."
--- 

A `TraefikService` is a custom resource that sits on top of the Kubernetes Services. It enables advanced load-balancing features such as a [Weighted Round Robin](#weighted-round-robin) load balancing, a [Highest Random Weight](#highest-random-weight) load balancing, or a [Mirroring](#mirroring) between your Kubernetes Services.

Services configure how to reach the actual endpoints that will eventually handle incoming requests. In Traefik, the target service can be either a standard [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/)—which exposes a pod—or a TraefikService. The latter allows you to combine advanced load-balancing options like:

- [Weighted Round Robin load balancing](#weighted-round-robin).
- [Highest Random Weight load balancing](#highest-random-weight).
- [Mirroring](#mirroring). 

## Weighted Round Robin

The WRR is able to load balance the requests between multiple services based on weights. The WRR `TraefikService` allows you to load balance the traffic between Kubernetes Services and other instances of `TraefikService` (another WRR service -, or a mirroring service).

### Configuration Example

```yaml tab="IngressRoute"
apiVersion: traefik.io/v1alpha1
kind: IngressRoute
metadata:
  name: test-name
  namespace: apps

spec:
  entryPoints:
  - websecure
  routes:
  - match: Host(`example.com`) && PathPrefix(`/foo`)
    kind: Rule
    services:
    # Set a WRR TraefikService
    - name: wrr1
      namespace: apps
      kind: TraefikService
  tls:
    # Add a TLS certificate from a Kubernetes Secret
    secretName: supersecret
```

```yaml tab="TraefikService WRR Level#1"
apiVersion: traefik.io/v1alpha1
kind: TraefikService
metadata:
  name: wrr1
  namespace: apps

spec:
  weighted:
    services:
        # Kubernetes Service
      - name: svc1
        namespace: apps
        port: 80
        weight: 1
        # Second level WRR service
      - name: wrr2
        namespace: apps
        kind: TraefikService
        weight: 1
        # Mirroring service
        # The service is described in the Mirroring example
      - name: mirror1
        namespace: apps
        kind: TraefikService
        weight: 1
```

```yaml tab="TraefikService WRR Level#2"
apiVersion: traefik.io/v1alpha1
kind: TraefikService
metadata:
  name: wrr2
  namespace: apps

spec:
  weighted:
    services:
      # Kubernetes Service
      - name: svc2
        namespace: apps
        port: 80
        weight: 1
      # Kubernetes Service
      - name: svc3
        namespace: apps
        port: 80
        weight: 1

```

```yaml tab="Kubernetes Services"
apiVersion: v1
kind: Service
metadata:
  name: svc1
  namespace: apps

spec:
  ports:
  - name: http
      port: 80
  selector:
    app: traefiklabs
    task: app1
---
apiVersion: v1
kind: Service
metadata:
  name: svc2
  namespace: apps

spec:
  ports:
  - name: http
    port: 80
  selector:
    app: traefiklabs
    task: app2
---
apiVersion: v1
kind: Service
metadata:
  name: svc3
  namespace: apps

spec:
  ports:
  - name: http
    port: 80
  selector:
    app: traefiklabs
    task: app3
```

```yaml tab="Secret"
apiVersion: v1
kind: Secret
metadata:
  name: supersecret
  namespace: apps

data:
  tls.crt: LS0tLS1CRUdJTiBDRVJUSUZJQ0FURS0tLS0tCi0tLS0tRU5EIENFUlRJRklDQVRFLS0tLS0=
  tls.key: LS0tLS1CRUdJTiBQUklWQVRFIEtFWS0tLS0tCi0tLS0tRU5EIFBSSVZBVEUgS0VZLS0tLS0=
```

### Configuration Options

| Field                                                          | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | Default                                                              | Required |
|:---------------------------------------------------------------|:---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|:---------------------------------------------------------------------|:---------|
| <a id="services" href="#services" title="#services">`services`</a> | List of any combination of TraefikService and [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/). <br />.                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |                                                                      | No       |
| <a id="servicesm-kind" href="#servicesm-kind" title="#servicesm-kind">`services[m].`<br />`kind`</a> | Kind of the service targeted.<br />Two values allowed:<br />- **Service**: Kubernetes Service<br /> - **TraefikService**: Traefik Service.                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | ""                                                                   | No       |
| <a id="servicesm-name" href="#servicesm-name" title="#servicesm-name">`services[m].`<br />`name`</a> | Service name.<br />The character `@` is not authorized.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | ""                                                                   | Yes      |
| <a id="servicesm-namespace" href="#servicesm-namespace" title="#servicesm-namespace">`services[m].`<br />`namespace`</a> | Service namespace.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | ""                                                                   | No       |
| <a id="servicesm-port" href="#servicesm-port" title="#servicesm-port">`services[m].`<br />`port`</a> | Service port (number or port name).<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | ""                                                                   | No       |
| <a id="servicesm-responseForwarding-flushInterval" href="#servicesm-responseForwarding-flushInterval" title="#servicesm-responseForwarding-flushInterval">`services[m].`<br />`responseForwarding.`<br />`flushInterval`</a> | Interval, in milliseconds, in between flushes to the client while copying the response body.<br />A negative value means to flush immediately after each write to the client.<br />This configuration is ignored when a response is a streaming response; for such responses, writes are flushed to the client immediately.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                          | 100ms                                                                | No       |
| <a id="servicesm-scheme" href="#servicesm-scheme" title="#servicesm-scheme">`services[m].`<br />`scheme`</a> | Scheme to use for the request to the upstream Kubernetes Service.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | "http"<br />"https" if `port` is 443 or contains the string *https*. | No       |
| <a id="servicesm-serversTransport" href="#servicesm-serversTransport" title="#servicesm-serversTransport">`services[m].`<br />`serversTransport`</a> | Name of ServersTransport resource to use to configure the transport between Traefik and your servers.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                | ""                                                                   | No       |
| <a id="servicesm-passHostHeader" href="#servicesm-passHostHeader" title="#servicesm-passHostHeader">`services[m].`<br />`passHostHeader`</a> | Forward client Host header to server.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | true                                                                 | No       |
| <a id="servicesm-healthCheck-scheme" href="#servicesm-healthCheck-scheme" title="#servicesm-healthCheck-scheme">`services[m].`<br />`healthCheck.scheme`</a> | Server URL scheme for the health check endpoint.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                        | ""                                                                   | No       |
| <a id="servicesm-healthCheck-mode" href="#servicesm-healthCheck-mode" title="#servicesm-healthCheck-mode">`services[m].`<br />`healthCheck.mode`</a> | Health check mode.<br /> If defined to grpc, will use the gRPC health check protocol to probe the server.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                               | "http"                                                               | No       |
| <a id="servicesm-healthCheck-path" href="#servicesm-healthCheck-path" title="#servicesm-healthCheck-path">`services[m].`<br />`healthCheck.path`</a> | Server URL path for the health check endpoint.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                          | ""                                                                   | No       |
| <a id="servicesm-healthCheck-interval" href="#servicesm-healthCheck-interval" title="#servicesm-healthCheck-interval">`services[m].`<br />`healthCheck.interval`</a> | Frequency of the health check calls for healthy targets.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName]`ExternalName`.                                                                                                                                                                                                                                                                                                                                                                  | "100ms"                                                              | No       |
| <a id="servicesm-healthCheck-unhealthyInterval" href="#servicesm-healthCheck-unhealthyInterval" title="#servicesm-healthCheck-unhealthyInterval">`services[m].`<br />`healthCheck.unhealthyInterval`</a> | Frequency of the health check calls for unhealthy targets.<br />When not defined, it defaults to the `interval` value.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName]`ExternalName`.                                                                                                                                                                                                                                                                                                    | "100ms"                                                              | No       |
| <a id="servicesm-healthCheck-method" href="#servicesm-healthCheck-method" title="#servicesm-healthCheck-method">`services[m].`<br />`healthCheck.method`</a> | HTTP method for the health check endpoint.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                              | "GET"                                                                | No       |
| <a id="servicesm-healthCheck-status" href="#servicesm-healthCheck-status" title="#servicesm-healthCheck-status">`services[m].`<br />`healthCheck.status`</a> | Expected HTTP status code of the response to the health check request.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type ExternalName.<br />If not set, expect a status between 200 and 399.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                              |                                                                      | No       |
| <a id="servicesm-healthCheck-port" href="#servicesm-healthCheck-port" title="#servicesm-healthCheck-port">`services[m].`<br />`healthCheck.port`</a> | URL port for the health check endpoint.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                                 |                                                                      | No       |
| <a id="servicesm-healthCheck-timeout" href="#servicesm-healthCheck-timeout" title="#servicesm-healthCheck-timeout">`services[m].`<br />`healthCheck.timeout`</a> | Maximum duration to wait before considering the server unhealthy.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                       | "5s"                                                                 | No       |
| <a id="servicesm-healthCheck-hostname" href="#servicesm-healthCheck-hostname" title="#servicesm-healthCheck-hostname">`services[m].`<br />`healthCheck.hostname`</a> | Value in the Host header of the health check request.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                   | ""                                                                   | No       |
| <a id="servicesm-healthCheck-followRedirect" href="#servicesm-healthCheck-followRedirect" title="#servicesm-healthCheck-followRedirect">`services[m].`<br />`healthCheck.`<br />`followRedirect`</a> | Follow the redirections during the healtchcheck.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                        | true                                                                 | No       |
| <a id="servicesm-healthCheck-headers" href="#servicesm-healthCheck-headers" title="#servicesm-healthCheck-headers">`services[m].`<br />`healthCheck.headers`</a> | Map of header to send to the health check endpoint<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                      |                                                                      | No       |
| <a id="servicesm-sticky-cookie-name" href="#servicesm-sticky-cookie-name" title="#servicesm-sticky-cookie-name">`services[m].`<br />`sticky.`<br />`cookie.name`</a> | Name of the cookie used for the stickiness.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | Abbreviation of a sha1<br />(ex: `_1d52e`).                          | No       |
| <a id="servicesm-sticky-cookie-httpOnly" href="#servicesm-sticky-cookie-httpOnly" title="#servicesm-sticky-cookie-httpOnly">`services[m].`<br />`sticky.`<br />`cookie.httpOnly`</a> | Allow the cookie can be accessed by client-side APIs, such as JavaScript.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | false                                                                | No       |
| <a id="servicesm-sticky-cookie-secure" href="#servicesm-sticky-cookie-secure" title="#servicesm-sticky-cookie-secure">`services[m].`<br />`sticky.`<br />`cookie.secure`</a> | Allow the cookie can only be transmitted over an encrypted connection (i.e. HTTPS).<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | false                                                                | No       |
| <a id="servicesm-sticky-cookie-sameSite" href="#servicesm-sticky-cookie-sameSite" title="#servicesm-sticky-cookie-sameSite">`services[m].`<br />`sticky.`<br />`cookie.sameSite`</a> | [SameSite](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Set-Cookie/SameSite) policy.<br />Allowed values:<br />-`none`<br />-`lax`<br />`strict`<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                        | ""                                                                   | No       |
| <a id="servicesm-sticky-cookie-maxAge" href="#servicesm-sticky-cookie-maxAge" title="#servicesm-sticky-cookie-maxAge">`services[m].`<br />`sticky.`<br />`cookie.maxAge`</a> | Number of seconds until the cookie expires.<br />Negative number, the cookie expires immediately.<br />0, the cookie never expires.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                  | 0                                                                    | No       |
| <a id="servicesm-strategy" href="#servicesm-strategy" title="#servicesm-strategy">`services[m].`<br />`strategy`</a> | Load balancing strategy between the servers.<br />RoundRobin is the only supported value yet.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | "RoundRobin"                                                         | No       |
| <a id="servicesm-weight" href="#servicesm-weight" title="#servicesm-weight">`services[m].`<br />`weight`</a> | Service weight.<br />To use only to refer to WRR TraefikService                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | ""                                                                   | No       |
| <a id="servicesm-nativeLB" href="#servicesm-nativeLB" title="#servicesm-nativeLB">`services[m].`<br />`nativeLB`</a> | Allow using the Kubernetes Service load balancing between the pods instead of the one provided by Traefik.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                           | false                                                                | No       |
| <a id="servicesm-nodePortLB" href="#servicesm-nodePortLB" title="#servicesm-nodePortLB">`services[m].`<br />`nodePortLB`</a> | Use the nodePort IP address when the service type is NodePort.<br />It allows services to be reachable when Traefik runs externally from the Kubernetes cluster but within the same network of the nodes.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                            | false                                                                | No       |
| <a id="sticky-cookie-name" href="#sticky-cookie-name" title="#sticky-cookie-name">`sticky.`<br />`cookie.name`</a> | Name of the cookie used for the stickiness at the WRR service level.<br />When sticky sessions are enabled, a `Set-Cookie` header is set on the initial response to let the client know which server handles the first response.<br />On subsequent requests, to keep the session alive with the same server, the client should send the cookie with the value set.<br />If the server pecified in the cookie becomes unhealthy, the request will be forwarded to a new server (and the cookie will keep track of the new server).<br />More information about WRR stickiness [here](#stickiness-on-multiple-levels) | Abbreviation of a sha1<br />(ex: `_1d52e`).                          | No       |
| <a id="sticky-cookie-httpOnly" href="#sticky-cookie-httpOnly" title="#sticky-cookie-httpOnly">`sticky.`<br />`cookie.httpOnly`</a> | Allow the cookie used for the stickiness at the WRR service level to be accessed by client-side APIs, such as JavaScript.<br />More information about WRR stickiness [here](#stickiness-on-multiple-levels)                                                                                                                                                                                                                                                                                                                                                                                                          | false                                                                | No       |
| <a id="sticky-cookie-secure" href="#sticky-cookie-secure" title="#sticky-cookie-secure">`sticky.`<br />`cookie.secure`</a> | Allow the cookie used for the stickiness at the WRR service level to be only transmitted over an encrypted connection (i.e. HTTPS).<br />More information about WRR stickiness [here](#stickiness-on-multiple-levels)                                                                                                                                                                                                                                                                                                                                                                                                | false                                                                | No       |
| <a id="sticky-cookie-sameSite" href="#sticky-cookie-sameSite" title="#sticky-cookie-sameSite">`sticky.`<br />`cookie.sameSite`</a> | [SameSite](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Set-Cookie/SameSite) policy for the cookie used for the stickiness at the WRR service level.<br />Allowed values:<br />-`none`<br />-`lax`<br />`strict`<br />More information about WRR stickiness [here](#stickiness-on-multiple-levels)                                                                                                                                                                                                                                                                                                      | ""                                                                   | No       |
| <a id="sticky-cookie-maxAge" href="#sticky-cookie-maxAge" title="#sticky-cookie-maxAge">`sticky.`<br />`cookie.maxAge`</a> | Number of seconds until the cookie used for the stickiness at the WRR service level expires.<br />Negative number, the cookie expires immediately.<br />0, the cookie never expires.                                                                                                                                                                                                                                                                                                                                                                                                                                 | 0                                                                    | No       |

#### Stickiness on multiple levels

When chaining or mixing load-balancers (e.g. a load-balancer of servers is one of the "children" of a load-balancer of services),
for stickiness to work all the way, the option needs to be specified at all required levels.
Which means the client needs to send a cookie with as many key/value pairs as there are sticky levels.

Sticky sessions, for stickiness to work all the way, must be specified at each load-balancing level.

For instance, in the example below, there is a first level of load-balancing because there is a (Weighted Round Robin) load-balancing of the two `whoami` services,
and there is a second level because each whoami service is a `replicaset` and is thus handled as a load-balancer of servers.

```yaml tab="IngressRoute"
apiVersion: traefik.io/v1alpha1
kind: IngressRoute
metadata:
  name: ingressroutebar
  namespace: apps

spec:
  entryPoints:
  - web
  routes:
  - match: Host(`example.com`) && PathPrefix(`/foo`)
    kind: Rule
    services:
    - name: wrr1
      namespace: apps
      kind: TraefikService
```

```yaml tab="TraefikService WRR with 2 level of stickiness"
apiVersion: traefik.io/v1alpha1
kind: TraefikService
metadata:
  name: wrr1
  namespace: apps
  
spec:
  weighted:
    services:
    - name: whoami1
      kind: Service
      port: 80
      weight: 1
      # Stickiness level2 (on the Kubernetes service)
      sticky:
        cookie:
        name: lvl2
    - name: whoami2
      kind: Service
      weight: 1
      port: 80
      # Stickiness level2 (on the Kubernetes service)
      sticky:
        cookie:
        name: lvl2
  # Stickiness level2 (on the WRR service)
  sticky:
    cookie:
    name: lvl1
```

In the example above, to keep a session open with the same server, the client would then need to specify the two levels within the cookie for each request, e.g. with curl:

```bash
# Assuming `10.42.0.6` is the IP address of one of the replicas (a pod then) of the `whoami1` service.
curl -H Host:example.com -b "lvl1=default-whoami1-80; lvl2=http://10.42.0.6:80" http://localhost:8000/foo
```

## Highest Random Weight

The HRW (Highest Random Weight) load balancer uses consistent hashing to ensure that requests from the same client IP are always routed to the same service. Unlike weighted round-robin which distributes requests based on weights, HRW provides consistent routing based on the client's remote address.

This is particularly useful for maintaining session affinity without requiring sticky cookies, as clients will consistently reach the same backend service based on their IP address.

### Configuration Example

```yaml tab="IngressRoute"
apiVersion: traefik.io/v1alpha1
kind: IngressRoute
metadata:
  name: test-hrw
  namespace: apps

spec:
  entryPoints:
  - websecure
  routes:
  - match: Host(`example.com`) && PathPrefix(`/app`)
    kind: Rule
    services:
    # Set an HRW TraefikService
    - name: hrw1
      namespace: apps
      kind: TraefikService
  tls:
    secretName: supersecret
```

```yaml tab="TraefikService HRW"
apiVersion: traefik.io/v1alpha1
kind: TraefikService
metadata:
  name: hrw1
  namespace: apps

spec:
  highestRandomWeight:
    services:
      # Kubernetes Service with weight 10
      - name: svc1
        namespace: apps
        port: 80
        weight: 10
      # Kubernetes Service with weight 20
      - name: svc2
        namespace: apps
        port: 80
        weight: 20
      # Another TraefikService
      - name: wrr1
        namespace: apps
        kind: TraefikService
        weight: 15
```

```yaml tab="Kubernetes Services"
apiVersion: v1
kind: Service
metadata:
  name: svc1
  namespace: apps

spec:
  ports:
  - name: http
    port: 80
  selector:
    app: traefiklabs
    task: app1
---
apiVersion: v1
kind: Service
metadata:
  name: svc2
  namespace: apps

spec:
  ports:
  - name: http
    port: 80
  selector:
    app: traefiklabs
    task: app2
```

### Configuration Options

| Field                                                          | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | Default                                                              | Required |
|:---------------------------------------------------------------|:---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|:---------------------------------------------------------------------|:---------|
| <a id="hrw-services" href="#hrw-services" title="#hrw-services">`services`</a> | List of any combination of TraefikService and [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/). Each service must have a weight assigned.                                                                                                                                                                                                                                                                                                                                                                                                                                     |                                                                      | Yes      |
| <a id="hrw-servicesm-kind" href="#hrw-servicesm-kind" title="#hrw-servicesm-kind">`services[m].`<br />`kind`</a> | Kind of the service targeted.<br />Two values allowed:<br />- **Service**: Kubernetes Service<br /> - **TraefikService**: Traefik Service.                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | ""                                                                   | No       |
| <a id="hrw-servicesm-name" href="#hrw-servicesm-name" title="#hrw-servicesm-name">`services[m].`<br />`name`</a> | Service name.<br />The character `@` is not authorized.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | ""                                                                   | Yes      |
| <a id="hrw-servicesm-namespace" href="#hrw-servicesm-namespace" title="#hrw-servicesm-namespace">`services[m].`<br />`namespace`</a> | Service namespace.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | ""                                                                   | No       |
| <a id="hrw-servicesm-port" href="#hrw-servicesm-port" title="#hrw-servicesm-port">`services[m].`<br />`port`</a> | Service port (number or port name).<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | ""                                                                   | No       |
| <a id="hrw-servicesm-weight" href="#hrw-servicesm-weight" title="#hrw-servicesm-weight">`services[m].`<br />`weight`</a> | Service weight used in the HRW algorithm. Higher weights increase the probability of selection for a given client IP.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 1                                                                    | No       |
| <a id="hrw-servicesm-responseForwarding-flushInterval" href="#hrw-servicesm-responseForwarding-flushInterval" title="#hrw-servicesm-responseForwarding-flushInterval">`services[m].`<br />`responseForwarding.`<br />`flushInterval`</a> | Interval, in milliseconds, in between flushes to the client while copying the response body.<br />A negative value means to flush immediately after each write to the client.<br />This configuration is ignored when a response is a streaming response; for such responses, writes are flushed to the client immediately.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                          | 100ms                                                                | No       |
| <a id="hrw-servicesm-scheme" href="#hrw-servicesm-scheme" title="#hrw-servicesm-scheme">`services[m].`<br />`scheme`</a> | Scheme to use for the request to the upstream Kubernetes Service.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | "http"<br />"https" if `port` is 443 or contains the string *https*. | No       |
| <a id="hrw-servicesm-serversTransport" href="#hrw-servicesm-serversTransport" title="#hrw-servicesm-serversTransport">`services[m].`<br />`serversTransport`</a> | Name of ServersTransport resource to use to configure the transport between Traefik and your servers.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                | ""                                                                   | No       |
| <a id="hrw-servicesm-passHostHeader" href="#hrw-servicesm-passHostHeader" title="#hrw-servicesm-passHostHeader">`services[m].`<br />`passHostHeader`</a> | Forward client Host header to server.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | true                                                                 | No       |
| <a id="hrw-servicesm-healthCheck-scheme" href="#hrw-servicesm-healthCheck-scheme" title="#hrw-servicesm-healthCheck-scheme">`services[m].`<br />`healthCheck.scheme`</a> | Server URL scheme for the health check endpoint.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                        | ""                                                                   | No       |
| <a id="hrw-servicesm-healthCheck-mode" href="#hrw-servicesm-healthCheck-mode" title="#hrw-servicesm-healthCheck-mode">`services[m].`<br />`healthCheck.mode`</a> | Health check mode.<br /> If defined to grpc, will use the gRPC health check protocol to probe the server.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                               | "http"                                                               | No       |
| <a id="hrw-servicesm-healthCheck-path" href="#hrw-servicesm-healthCheck-path" title="#hrw-servicesm-healthCheck-path">`services[m].`<br />`healthCheck.path`</a> | Server URL path for the health check endpoint.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                          | ""                                                                   | No       |
| <a id="hrw-servicesm-healthCheck-interval" href="#hrw-servicesm-healthCheck-interval" title="#hrw-servicesm-healthCheck-interval">`services[m].`<br />`healthCheck.interval`</a> | Frequency of the health check calls for healthy targets.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                  | "100ms"                                                              | No       |
| <a id="hrw-servicesm-healthCheck-unhealthyInterval" href="#hrw-servicesm-healthCheck-unhealthyInterval" title="#hrw-servicesm-healthCheck-unhealthyInterval">`services[m].`<br />`healthCheck.unhealthyInterval`</a> | Frequency of the health check calls for unhealthy targets.<br />When not defined, it defaults to the `interval` value.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                    | "100ms"                                                              | No       |
| <a id="hrw-servicesm-healthCheck-method" href="#hrw-servicesm-healthCheck-method" title="#hrw-servicesm-healthCheck-method">`services[m].`<br />`healthCheck.method`</a> | HTTP method for the health check endpoint.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                              | "GET"                                                                | No       |
| <a id="hrw-servicesm-healthCheck-status" href="#hrw-servicesm-healthCheck-status" title="#hrw-servicesm-healthCheck-status">`services[m].`<br />`healthCheck.status`</a> | Expected HTTP status code of the response to the health check request.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type ExternalName.<br />If not set, expect a status between 200 and 399.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                              |                                                                      | No       |
| <a id="hrw-servicesm-healthCheck-port" href="#hrw-servicesm-healthCheck-port" title="#hrw-servicesm-healthCheck-port">`services[m].`<br />`healthCheck.port`</a> | URL port for the health check endpoint.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                                 |                                                                      | No       |
| <a id="hrw-servicesm-healthCheck-timeout" href="#hrw-servicesm-healthCheck-timeout" title="#hrw-servicesm-healthCheck-timeout">`services[m].`<br />`healthCheck.timeout`</a> | Maximum duration to wait before considering the server unhealthy.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                       | "5s"                                                                 | No       |
| <a id="hrw-servicesm-healthCheck-hostname" href="#hrw-servicesm-healthCheck-hostname" title="#hrw-servicesm-healthCheck-hostname">`services[m].`<br />`healthCheck.hostname`</a> | Value in the Host header of the health check request.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                   | ""                                                                   | No       |
| <a id="hrw-servicesm-healthCheck-followRedirect" href="#hrw-servicesm-healthCheck-followRedirect" title="#hrw-servicesm-healthCheck-followRedirect">`services[m].`<br />`healthCheck.`<br />`followRedirect`</a> | Follow the redirections during the healtchcheck.<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                        | true                                                                 | No       |
| <a id="hrw-servicesm-healthCheck-headers" href="#hrw-servicesm-healthCheck-headers" title="#hrw-servicesm-healthCheck-headers">`services[m].`<br />`healthCheck.headers`</a> | Map of header to send to the health check endpoint<br />Evaluated only if the kind is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type `ExternalName`.                                                                                                                                                                                                                                                                                                                                                                                      |                                                                      | No       |
| <a id="hrw-servicesm-sticky-cookie-name" href="#hrw-servicesm-sticky-cookie-name" title="#hrw-servicesm-sticky-cookie-name">`services[m].`<br />`sticky.`<br />`cookie.name`</a> | Name of the cookie used for the stickiness.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | Abbreviation of a sha1<br />(ex: `_1d52e`).                          | No       |
| <a id="hrw-servicesm-sticky-cookie-httpOnly" href="#hrw-servicesm-sticky-cookie-httpOnly" title="#hrw-servicesm-sticky-cookie-httpOnly">`services[m].`<br />`sticky.`<br />`cookie.httpOnly`</a> | Allow the cookie can be accessed by client-side APIs, such as JavaScript.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | false                                                                | No       |
| <a id="hrw-servicesm-sticky-cookie-secure" href="#hrw-servicesm-sticky-cookie-secure" title="#hrw-servicesm-sticky-cookie-secure">`services[m].`<br />`sticky.`<br />`cookie.secure`</a> | Allow the cookie can only be transmitted over an encrypted connection (i.e. HTTPS).<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | false                                                                | No       |
| <a id="hrw-servicesm-sticky-cookie-sameSite" href="#hrw-servicesm-sticky-cookie-sameSite" title="#hrw-servicesm-sticky-cookie-sameSite">`services[m].`<br />`sticky.`<br />`cookie.sameSite`</a> | [SameSite](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Set-Cookie/SameSite) policy.<br />Allowed values:<br />-`none`<br />-`lax`<br />`strict`<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                        | ""                                                                   | No       |
| <a id="hrw-servicesm-sticky-cookie-maxAge" href="#hrw-servicesm-sticky-cookie-maxAge" title="#hrw-servicesm-sticky-cookie-maxAge">`services[m].`<br />`sticky.`<br />`cookie.maxAge`</a> | Number of seconds until the cookie expires.<br />Negative number, the cookie expires immediately.<br />0, the cookie never expires.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                  | 0                                                                    | No       |
| <a id="hrw-servicesm-strategy" href="#hrw-servicesm-strategy" title="#hrw-servicesm-strategy">`services[m].`<br />`strategy`</a> | Load balancing strategy between the servers.<br />RoundRobin is the only supported value yet.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | "RoundRobin"                                                         | No       |
| <a id="hrw-servicesm-nativeLB" href="#hrw-servicesm-nativeLB" title="#hrw-servicesm-nativeLB">`services[m].`<br />`nativeLB`</a> | Allow using the Kubernetes Service load balancing between the pods instead of the one provided by Traefik.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                           | false                                                                | No       |
| <a id="hrw-servicesm-nodePortLB" href="#hrw-servicesm-nodePortLB" title="#hrw-servicesm-nodePortLB">`services[m].`<br />`nodePortLB`</a> | Use the nodePort IP address when the service type is NodePort.<br />It allows services to be reachable when Traefik runs externally from the Kubernetes cluster but within the same network of the nodes.<br />Evaluated only if the kind is **Service**.                                                                                                                                                                                                                                                                                                                                                            | false                                                                | No       |

### How HRW Works

The Highest Random Weight algorithm combines consistent hashing with weighted load balancing:

1. **Consistent Hashing**: For each incoming request, the client's remote address is hashed to ensure consistent routing.
2. **Weighted Selection**: Each service is assigned a random value based on both the hash of the client IP and the service's weight.
3. **Highest Selection**: The service with the highest calculated value receives the request.

This approach provides several benefits:

- **Session Affinity**: Clients consistently reach the same backend service
- **Weighted Distribution**: Services with higher weights are more likely to be selected
- **No State Required**: Unlike sticky cookies, no client-side or server-side state is needed
- **Fault Tolerance**: If a service becomes unavailable, requests are redistributed consistently among remaining services

## Mirroring

The mirroring is able to mirror requests sent to a service to other services.

A mirroring service allows you to send the trafiic to many services together:

- The **main** service receives 100% of the traffic,
- The **mirror** services receive a percentage of the traffic.

For example, to upgrade the version of your application. You can set the service that targets current version as the **main** service, and the service of the new version a **mirror** service.
Thus you can start testing the behavior of the new version keeping the current version reachable.

The mirroring `TraefikService` allows you to reference Kubernetes Services and other instances of `TraefikService` (another WRR service -, or a mirroring service).

Please note that by default the whole request is buffered in memory while it is being mirrored.
See the `maxBodySize` option in the example below for how to modify this behavior.

### Configuration Example

```yaml tab="IngressRoute"
apiVersion: traefik.io/v1alpha1
kind: IngressRoute
metadata:
  name: ingressroutebar
  namespace: default

spec:
  entryPoints:
    - web
  routes:
  - match: Host(`example.com`) && PathPrefix(`/foo`)
    kind: Rule
    services:
    - name: mirror1
      namespace: default
      kind: TraefikService
```

```yaml tab="Mirroring from a Kubernetes Service"
# Mirroring from a k8s Service
apiVersion: traefik.io/v1alpha1
kind: TraefikService
metadata:
  name: mirror1
  namespace: default

spec:
  mirroring:
    name: svc1                      # svc1 receives 100% of the traffic
    port: 80
    mirrors:
      - name: svc2                  # svc2 receives a copy of 20% of this traffic
        port: 80
        percent: 20
      - name: svc3                  # svc3 receives a copy of 15% of this traffic
        kind: TraefikService
        percent: 15
```

```yaml tab="Mirroring from a TraefikService (WRR)"
# Mirroring from a Traefik Service
apiVersion: traefik.io/v1alpha1
kind: TraefikService
metadata:
  name: mirror1
  namespace: default

spec:
  mirroring:
    name: wrr1                      # wrr1 receives 100% of the traffic
    kind: TraefikService
    mirrors:
      - name: svc2                  # svc2 receives a copy of 20% of this traffic
        port: 80
        percent: 20
      - name: svc3                  # svc3 receives a copy of 10% of this traffic
        kind: TraefikService
        percent: 10
```

```yaml tab="Kubernetes Services"
apiVersion: v1
kind: Service
metadata:
  name: svc1
  namespace: default

spec:
  ports:
    - name: http
      port: 80
  selector:
    app: traefiklabs
    task: app1
---
apiVersion: v1
kind: Service
metadata:
  name: svc2
  namespace: default

spec:
  ports:
    - name: http
      port: 80
  selector:
    app: traefiklabs
    task: app2
```

### Configuration Options

!!!note "Main and mirrored services"

    The main service properties are set as the option root level.

    The mirrored services properties are set in the `mirrors` list.

| Field                                                         | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | Default                                                              | Required |
|:--------------------------------------------------------------|:----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|:---------------------------------------------------------------------|:---------|
| <a id="kind" href="#kind" title="#kind">`kind`</a> | Kind of the main service.<br />Two values allowed:<br />- **Service**: Kubernetes Service<br />- **TraefikService**: Traefik Service.<br />More information [here](#services)                                                                                                                                                                                                                                                                                                                                                                                                     | ""                                                                   | No       |
| <a id="name" href="#name" title="#name">`name`</a> | Main service name.<br />The character `@` is not authorized.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | ""                                                                   | Yes      |
| <a id="namespace" href="#namespace" title="#namespace">`namespace`</a> | Main service namespace.<br />More information [here](#services).                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | ""                                                                   | No       |
| <a id="port" href="#port" title="#port">`port`</a> | Main service port (number or port name).<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | ""                                                                   | No       |
| <a id="responseForwarding-flushInterval" href="#responseForwarding-flushInterval" title="#responseForwarding-flushInterval">`responseForwarding.`<br />`flushInterval`</a> | Interval, in milliseconds, in between flushes to the client while copying the response body.<br />A negative value means to flush immediately after each write to the client.<br />This configuration is ignored when a response is a streaming response; for such responses, writes are flushed to the client immediately.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                   | 100ms                                                                | No       |
| <a id="scheme" href="#scheme" title="#scheme">`scheme`</a> | Scheme to use for the request to the upstream Kubernetes Service.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                             | "http"<br />"https" if `port` is 443 or contains the string *https*. | No       |
| <a id="serversTransport" href="#serversTransport" title="#serversTransport">`serversTransport`</a> | Name of ServersTransport resource to use to configure the transport between Traefik and the main service's servers.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                           | ""                                                                   | No       |
| <a id="passHostHeader" href="#passHostHeader" title="#passHostHeader">`passHostHeader`</a> | Forward client Host header to main service's server.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                          | true                                                                 | No       |
| <a id="healthCheck-scheme" href="#healthCheck-scheme" title="#healthCheck-scheme">`healthCheck.scheme`</a> | Server URL scheme for the health check endpoint.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                      | ""                                                                   | No       |
| <a id="healthCheck-mode" href="#healthCheck-mode" title="#healthCheck-mode">`healthCheck.mode`</a> | Health check mode.<br /> If defined to grpc, will use the gRPC health check protocol to probe the server.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                             | "http"                                                               | No       |
| <a id="healthCheck-path" href="#healthCheck-path" title="#healthCheck-path">`healthCheck.path`</a> | Server URL path for the health check endpoint.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                        | ""                                                                   | No       |
| <a id="healthCheck-interval" href="#healthCheck-interval" title="#healthCheck-interval">`healthCheck.interval`</a> | Frequency of the health check calls for healthy targets.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                              | "100ms"                                                              | No       |
| <a id="healthCheck-unhealthyInterval" href="#healthCheck-unhealthyInterval" title="#healthCheck-unhealthyInterval">`healthCheck.unhealthyInterval`</a> | Frequency of the health check calls for unhealthy targets.<br />When not defined, it defaults to the `interval` value.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                | "100ms"                                                              | No       |
| <a id="healthCheck-method" href="#healthCheck-method" title="#healthCheck-method">`healthCheck.method`</a> | HTTP method for the health check endpoint.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                            | "GET"                                                                | No       |
| <a id="healthCheck-status" href="#healthCheck-status" title="#healthCheck-status">`healthCheck.status`</a> | Expected HTTP status code of the response to the health check request.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type ExternalName.<br />If not set, expect a status between 200 and 399.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                       |                                                                      | No       |
| <a id="healthCheck-port" href="#healthCheck-port" title="#healthCheck-port">`healthCheck.port`</a> | URL port for the health check endpoint.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                               |                                                                      | No       |
| <a id="healthCheck-timeout" href="#healthCheck-timeout" title="#healthCheck-timeout">`healthCheck.timeout`</a> | Maximum duration to wait before considering the server unhealthy.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                     | "5s"                                                                 | No       |
| <a id="healthCheck-hostname" href="#healthCheck-hostname" title="#healthCheck-hostname">`healthCheck.hostname`</a> | Value in the Host header of the health check request.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                 | ""                                                                   | No       |
| <a id="healthCheck-followRedirect" href="#healthCheck-followRedirect" title="#healthCheck-followRedirect">`healthCheck.`<br />`followRedirect`</a> | Follow the redirections during the healtchcheck.<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                      | true                                                                 | No       |
| <a id="healthCheck-headers" href="#healthCheck-headers" title="#healthCheck-headers">`healthCheck.headers`</a> | Map of header to send to the health check endpoint<br />Evaluated only if the kind of the main service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                    |                                                                      | No       |
| <a id="sticky-cookie-name-2" href="#sticky-cookie-name-2" title="#sticky-cookie-name-2">`sticky.`<br />`cookie.name`</a> | Name of the cookie used for the stickiness on the main service.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                               | Abbreviation of a sha1<br />(ex: `_1d52e`).                          | No       |
| <a id="sticky-cookie-httpOnly-2" href="#sticky-cookie-httpOnly-2" title="#sticky-cookie-httpOnly-2">`sticky.`<br />`cookie.httpOnly`</a> | Allow the cookie can be accessed by client-side APIs, such as JavaScript.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                     | false                                                                | No       |
| <a id="sticky-cookie-secure-2" href="#sticky-cookie-secure-2" title="#sticky-cookie-secure-2">`sticky.`<br />`cookie.secure`</a> | Allow the cookie can only be transmitted over an encrypted connection (i.e. HTTPS).<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                           | false                                                                | No       |
| <a id="sticky-cookie-sameSite-2" href="#sticky-cookie-sameSite-2" title="#sticky-cookie-sameSite-2">`sticky.`<br />`cookie.sameSite`</a> | [SameSite](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Set-Cookie/SameSite) policy.<br />Allowed values:<br />-`none`<br />-`lax`<br />`strict`<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                 | ""                                                                   | No       |
| <a id="sticky-cookie-maxAge-2" href="#sticky-cookie-maxAge-2" title="#sticky-cookie-maxAge-2">`sticky.`<br />`cookie.maxAge`</a> | Number of seconds until the cookie expires.<br />Negative number, the cookie expires immediately.<br />0, the cookie never expires.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                           | 0                                                                    | No       |
| <a id="strategy" href="#strategy" title="#strategy">`strategy`</a> | Load balancing strategy between the main service's servers.<br />RoundRobin is the only supported value yet.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                  | "RoundRobin"                                                         | No       |
| <a id="weight" href="#weight" title="#weight">`weight`</a> | Service weight.<br />To use only to refer to WRR TraefikService                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | ""                                                                   | No       |
| <a id="nativeLB" href="#nativeLB" title="#nativeLB">`nativeLB`</a> | Allow using the Kubernetes Service load balancing between the pods instead of the one provided by Traefik.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                    | false                                                                | No       |
| <a id="nodePortLB" href="#nodePortLB" title="#nodePortLB">`nodePortLB`</a> | Use the nodePort IP address when the service type is NodePort.<br />It allows services to be reachable when Traefik runs externally from the Kubernetes cluster but within the same network of the nodes.<br />Evaluated only if the kind of the main service is **Service**.                                                                                                                                                                                                                                                                                                     | false                                                                | No       |
| <a id="maxBodySize" href="#maxBodySize" title="#maxBodySize">`maxBodySize`</a> | Maximum size allowed for the body of the request.<br />If the body is larger, the request is not mirrored.<br />-1 means unlimited size.                                                                                                                                                                                                                                                                                                                                                                                                                                          | -1                                                                   | No       |
| <a id="mirrors" href="#mirrors" title="#mirrors">`mirrors`</a> | List of mirrored services to target.<br /> It can be any combination of TraefikService and [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/). <br />More information [here](#services).                                                                                                                                                                                                                                                                                                                                                      |                                                                      | No       |
| <a id="mirrorsm-kind" href="#mirrorsm-kind" title="#mirrorsm-kind">`mirrors[m].`<br />`kind`</a> | Kind of the mirrored service targeted.<br />Two values allowed:<br />- **Service**: Kubernetes Service<br />- **TraefikService**: Traefik Service.<br />More information [here](#services)                                                                                                                                                                                                                                                                                                                                                                                        | ""                                                                   | No       |
| <a id="mirrorsm-name" href="#mirrorsm-name" title="#mirrorsm-name">`mirrors[m].`<br />`name`</a> | Mirrored service name.<br />The character `@` is not authorized.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | ""                                                                   | Yes      |
| <a id="mirrorsm-namespace" href="#mirrorsm-namespace" title="#mirrorsm-namespace">`mirrors[m].`<br />`namespace`</a> | Mirrored service namespace.<br />More information [here](#services).                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | ""                                                                   | No       |
| <a id="mirrorsm-port" href="#mirrorsm-port" title="#mirrorsm-port">`mirrors[m].`<br />`port`</a> | Mirrored service port (number or port name).<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                              | ""                                                                   | No       |
| <a id="mirrorsm-percent" href="#mirrorsm-percent" title="#mirrorsm-percent">`mirrors[m].`<br />`percent`</a> | Part of the traffic to mirror in percent (from 0 to 100)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 0                                                                    | No       |
| <a id="mirrorsm-responseForwarding-flushInterval" href="#mirrorsm-responseForwarding-flushInterval" title="#mirrorsm-responseForwarding-flushInterval">`mirrors[m].`<br />`responseForwarding.`<br />`flushInterval`</a> | Interval, in milliseconds, in between flushes to the client while copying the response body.<br />A negative value means to flush immediately after each write to the client.<br />This configuration is ignored when a response is a streaming response; for such responses, writes are flushed to the client immediately.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                               | 100ms                                                                | No       |
| <a id="mirrorsm-scheme" href="#mirrorsm-scheme" title="#mirrorsm-scheme">`mirrors[m].`<br />`scheme`</a> | Scheme to use for the request to the mirrored service.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                                    | "http"<br />"https" if `port` is 443 or contains the string *https*. | No       |
| <a id="mirrorsm-serversTransport" href="#mirrorsm-serversTransport" title="#mirrorsm-serversTransport">`mirrors[m].`<br />`serversTransport`</a> | Name of ServersTransport resource to use to configure the transport between Traefik and the mirrored service servers.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                     | ""                                                                   | No       |
| <a id="mirrorsm-passHostHeader" href="#mirrorsm-passHostHeader" title="#mirrorsm-passHostHeader">`mirrors[m].`<br />`passHostHeader`</a> | Forward client Host header to the mirrored service servers.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                               | true                                                                 | No       |
| <a id="mirrorsm-healthCheck-scheme" href="#mirrorsm-healthCheck-scheme" title="#mirrorsm-healthCheck-scheme">`mirrors[m].`<br />`healthCheck.scheme`</a> | Server URL scheme for the health check endpoint.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                  | ""                                                                   | No       |
| <a id="mirrorsm-healthCheck-mode" href="#mirrorsm-healthCheck-mode" title="#mirrorsm-healthCheck-mode">`mirrors[m].`<br />`healthCheck.mode`</a> | Health check mode.<br /> If defined to grpc, will use the gRPC health check protocol to probe the server.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                         | "http"                                                               | No       |
| <a id="mirrorsm-healthCheck-path" href="#mirrorsm-healthCheck-path" title="#mirrorsm-healthCheck-path">`mirrors[m].`<br />`healthCheck.path`</a> | Server URL path for the health check endpoint.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                    | ""                                                                   | No       |
| <a id="mirrorsm-healthCheck-interval" href="#mirrorsm-healthCheck-interval" title="#mirrorsm-healthCheck-interval">`mirrors[m].`<br />`healthCheck.interval`</a> | Frequency of the health check calls.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                              | "100ms"                                                              | No       |
| <a id="mirrorsm-healthCheck-unhealthyInterval" href="#mirrorsm-healthCheck-unhealthyInterval" title="#mirrorsm-healthCheck-unhealthyInterval">`mirrors[m].`<br />`healthCheck.unhealthyInterval`</a> | Frequency of the health check calls for unhealthy targets.<br />When not defined, it defaults to the `interval` value.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                            | "100ms"                                                              | No       |
| <a id="mirrorsm-healthCheck-method" href="#mirrorsm-healthCheck-method" title="#mirrorsm-healthCheck-method">`mirrors[m].`<br />`healthCheck.method`</a> | HTTP method for the health check endpoint.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                        | "GET"                                                                | No       |
| <a id="mirrorsm-healthCheck-status" href="#mirrorsm-healthCheck-status" title="#mirrorsm-healthCheck-status">`mirrors[m].`<br />`healthCheck.status`</a> | Expected HTTP status code of the response to the health check request.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type ExternalName.<br />If not set, expect a status between 200 and 399.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                   |                                                                      | No       |
| <a id="mirrorsm-healthCheck-port" href="#mirrorsm-healthCheck-port" title="#mirrorsm-healthCheck-port">`mirrors[m].`<br />`healthCheck.port`</a> | URL port for the health check endpoint.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                           |                                                                      | No       |
| <a id="mirrorsm-healthCheck-timeout" href="#mirrorsm-healthCheck-timeout" title="#mirrorsm-healthCheck-timeout">`mirrors[m].`<br />`healthCheck.timeout`</a> | Maximum duration to wait before considering the server unhealthy.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                 | "5s"                                                                 | No       |
| <a id="mirrorsm-healthCheck-hostname" href="#mirrorsm-healthCheck-hostname" title="#mirrorsm-healthCheck-hostname">`mirrors[m].`<br />`healthCheck.hostname`</a> | Value in the Host header of the health check request.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                             | ""                                                                   | No       |
| <a id="mirrorsm-healthCheck-followRedirect" href="#mirrorsm-healthCheck-followRedirect" title="#mirrorsm-healthCheck-followRedirect">`mirrors[m].`<br />`healthCheck.`<br />`followRedirect`</a> | Follow the redirections during the healtchcheck.<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                  | true                                                                 | No       |
| <a id="mirrorsm-healthCheck-headers" href="#mirrorsm-healthCheck-headers" title="#mirrorsm-healthCheck-headers">`mirrors[m].`<br />`healthCheck.headers`</a> | Map of header to send to the health check endpoint<br />Evaluated only if the kind of the mirrored service is **Service**.<br />Only for [Kubernetes service](https://kubernetes.io/docs/concepts/services-networking/service/) of type [ExternalName](#services).                                                                                                                                                                                                                                                                                                                |                                                                      | No       |
| <a id="mirrorsm-sticky-cookie-name" href="#mirrorsm-sticky-cookie-name" title="#mirrorsm-sticky-cookie-name">`mirrors[m].`<br />`sticky.`<br />`cookie.name`</a> | Name of the cookie used for the stickiness.<br />When sticky sessions are enabled, a `Set-Cookie` header is set on the initial response to let the client know which server handles the first response.<br />On subsequent requests, to keep the session alive with the same server, the client should send the cookie with the value set.<br />If the server pecified in the cookie becomes unhealthy, the request will be forwarded to a new server (and the cookie will keep track of the new server).<br />Evaluated only if the kind of the mirrored service is **Service**. | ""                                                                   | No       |
| <a id="mirrorsm-sticky-cookie-httpOnly" href="#mirrorsm-sticky-cookie-httpOnly" title="#mirrorsm-sticky-cookie-httpOnly">`mirrors[m].`<br />`sticky.`<br />`cookie.httpOnly`</a> | Allow the cookie can be accessed by client-side APIs, such as JavaScript.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                                 | false                                                                | No       |
| <a id="mirrorsm-sticky-cookie-secure" href="#mirrorsm-sticky-cookie-secure" title="#mirrorsm-sticky-cookie-secure">`mirrors[m].`<br />`sticky.`<br />`cookie.secure`</a> | Allow the cookie can only be transmitted over an encrypted connection (i.e. HTTPS).<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                                       | false                                                                | No       |
| <a id="mirrorsm-sticky-cookie-sameSite" href="#mirrorsm-sticky-cookie-sameSite" title="#mirrorsm-sticky-cookie-sameSite">`mirrors[m].`<br />`sticky.`<br />`cookie.sameSite`</a> | [SameSite](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Set-Cookie/SameSite) policy.<br />Allowed values:<br />-`none`<br />-`lax`<br />`strict`<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                             | ""                                                                   | No       |
| <a id="mirrorsm-sticky-cookie-maxAge" href="#mirrorsm-sticky-cookie-maxAge" title="#mirrorsm-sticky-cookie-maxAge">`mirrors[m].`<br />`sticky.`<br />`cookie.maxAge`</a> | Number of seconds until the cookie expires.<br />Negative number, the cookie expires immediately.<br />0, the cookie never expires.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                                                       | 0                                                                    | No       |
| <a id="mirrorsm-strategy" href="#mirrorsm-strategy" title="#mirrorsm-strategy">`mirrors[m].`<br />`strategy`</a> | Load balancing strategy between the servers.<br />RoundRobin is the only supported value yet.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                             | "RoundRobin"                                                         | No       |
| <a id="mirrorsm-weight" href="#mirrorsm-weight" title="#mirrorsm-weight">`mirrors[m].`<br />`weight`</a> | Service weight.<br />To use only to refer to WRR TraefikService                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | ""                                                                   | No       |
| <a id="mirrorsm-nativeLB" href="#mirrorsm-nativeLB" title="#mirrorsm-nativeLB">`mirrors[m].`<br />`nativeLB`</a> | Allow using the Kubernetes Service load balancing between the pods instead of the one provided by Traefik.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                                                                                                                | false                                                                | No       |
| <a id="mirrorsm-nodePortLB" href="#mirrorsm-nodePortLB" title="#mirrorsm-nodePortLB">`mirrors[m].`<br />`nodePortLB`</a> | Use the nodePort IP address when the service type is NodePort.<br />It allows services to be reachable when Traefik runs externally from the Kubernetes cluster but within the same network of the nodes.<br />Evaluated only if the kind of the mirrored service is **Service**.                                                                                                                                                                                                                                                                                                 | false                                                                | No       |
| <a id="mirrorBody" href="#mirrorBody" title="#mirrorBody">`mirrorBody`</a> | Defines whether the request body should be mirrored.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | true                                                                 | No       |
