/*
 *   Copyright 2014-2016 Guy Bedford (http://guybedford.com)
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
var ui = require('../ui');
var fs = require('graceful-fs');
var path = require('path');
var PackageConfig = require('./package');
var JspmSystemConfig = require('./loader').JspmSystemConfig;
var mkdirp = require('mkdirp');
var readJSON = require('../common').readJSON;
var PackageName = require('../package-name');
var stringify = require('../common').stringify;
var inDir = require('../common').inDir;
var toFileURL = require('../common').toFileURL;
var isPlain = require('../common').isPlain;
var install = require('../install');
var wordWrap = require('../ui').wordWrap;
var asp = require('bluebird').Promise.promisify;

var config = module.exports;

exports.version = require('../../package.json').version;

// package and loader configuration objects that are created
exports.pjson = null;
exports.loader = null;

exports.getLoaderConfig = function() {
  var cfg = config.loader.getConfig();

  // this is all that is really needed for this function
  cfg.baseURL = toFileURL(config.pjson.baseURL);

  // set a node path
  // removing any possible overrides to that node path in the config
  cfg.nodeConfig = cfg.nodeConfig || {};
  cfg.nodeConfig.paths = cfg.nodeConfig.paths || {};
  function setPath(name, value) {
    // the only thing that can override cfg.nodeConfig in builder is cfg.devConfig
    if (cfg.devConfig && cfg.devConfig.paths)
      delete cfg.devConfig.paths[name];
    cfg.nodeConfig.paths[name] = value;
  }

  (cfg.packageConfigPaths || []).forEach(function(pkgConfigPath) {
    var registryName = pkgConfigPath.substr(0, pkgConfigPath.indexOf(':'));
    if (config.loader.nodeRegistryPaths) {
      if (config.loader.nodeRegistryPaths[registryName + ':'])
        setPath(registryName + ':', toFileURL(config.loader.nodeRegistryPaths[registryName + ':']));
    }
    else {
      setPath(registryName + ':', toFileURL(path.resolve(config.loader.nodePackagesURL, registryName) + '/'));
    }
  });
  if (config.loader.package)
    setPath(config.pjson.name + '/', toFileURL(config.loader.nodeLibURL));

  // no depCache or bundles
  delete cfg.depCache;
  // delete cfg.bundles;

  return cfg;
};

function updateNodelibTarget(target, range) {
  target.setRegistry('npm');
  target.setVersion(range ? '^0.2.0' : '0.2.0');
  target.setPackage(target.package.replace('jspm/nodelibs-', 'jspm-nodelibs-'));
}

function processNodelibs(loader, pjson, upgrade) {
  var installRequired = false;
  Object.keys(loader.baseMap).forEach(function (key) {
    var target = loader.baseMap[key];

    if (target.name.substr(0, 21) == 'github:jspm/nodelibs-') {
      updateNodelibTarget(target, false);
      installRequired = true;
    }
  });

  Object.keys(loader.depMap).forEach(function (key) {
    var curMap = loader.depMap[key];

    Object.keys(curMap).forEach(function (key) {
      var target = curMap[key];

      if (!upgrade) {
        if (target.name.substr(0, 21) == 'github:jspm/nodelibs-') {
          installRequired = true;
          updateNodelibTarget(target, false);
        }
      } else {
        if (target.name.substr(0, 21) == 'github:jspm/nodelibs-') {
          installRequired = true;
          if (key == target.name.substr(21))
            delete curMap[key];
          else
            updateNodelibTarget(target, false);
        }
      }
    });
  });

  Object.keys(pjson.dependencies).forEach(function (key) {
    var target = pjson.dependencies[key];

    if (target.name.substr(0, 21) == 'github:jspm/nodelibs-') {
      updateNodelibTarget(target, true);
      installRequired = true;

      if (upgrade) {
        pjson.peerDependencies[key] = target;
        delete pjson.dependencies[key];
      }
    }
  });

  Object.keys(pjson.devDependencies).forEach(function (key) {
    var target = pjson.devDependencies[key];

    if (target.name.substr(0, 21) == 'github:jspm/nodelibs-') {
      updateNodelibTarget(target, true);
      installRequired = true;

      if (upgrade) {
        pjson.peerDependencies[key] = target;
        delete pjson.devDependencies[key];
      }
    }
  });

  Object.keys(pjson.peerDependencies).forEach(function (key) {
    var target = pjson.peerDependencies[key];

    if (target.name.substr(0, 21) == 'github:jspm/nodelibs-') {
      updateNodelibTarget(target, true);
      installRequired = true;
    }
  });

  return installRequired;
}

var loadPromise;
exports.loaded = false;
exports.allowNoConfig = false;
exports.load = function(prompts, allowNoConfig) {
  if (config.loaded)
    return Promise.resolve();

  if (loadPromise)
    return loadPromise;

  return (loadPromise = Promise.resolve()
  .then(function() {
    config.allowNoConfig = !!allowNoConfig;

    if (process.env.globalJspm === 'true')
      ui.log('warn', 'Running jspm globally, it is advisable to locally install jspm via %npm install jspm --save-dev%.');

    config.pjsonPath = process.env.jspmConfigPath || path.resolve(process.cwd(), 'package.json');

    config.pjson = new PackageConfig(config.pjsonPath);
    config.loader = new JspmSystemConfig(config.pjson.configFile);

    // beta upgrade of jspm.js -> jspm.config.js
    if (config.loader.emptyConfig) {
      var jspmConfig = new JspmSystemConfig(path.resolve(config.pjson.baseURL, 'jspm.js'));
      if (!jspmConfig.emptyConfig) {
        config.loader = jspmConfig;
        config.loader.file.rename(config.pjson.configFile);
      }
    }

    // load jspm 0.16 default config path for upgrade
    if (config.loader.emptyConfig) {
      var upgradeConfig = new JspmSystemConfig(path.resolve(config.pjson.baseURL, 'config.js'));
      if (upgradeConfig.upgrade16 && !upgradeConfig.emptyConfig)
        config.loader = upgradeConfig;
      else
        config.loader.upgrade16 = false;
    }

    if (allowNoConfig && !config.pjson.jspmAware && !config.loader.upgrade16)
      config.loader.transpiler = 'none';

    if ((!process.env.jspmConfigPath || config.pjson.file.timestamp == -1) && !allowNoConfig)
      return ui.confirm('Package.json file does not exist, create it?', true)
      .then(function(create) {
        if (!create)
          throw 'Operation aborted.';
      });

    if (!allowNoConfig && config.loader.file.timestamp == -1 && !config.loader.upgrade16)
      return ui.confirm('Configuration file %' + path.relative(process.cwd(), config.pjson.configFile) + '% does not exist, create it?', true)
      .then(function(create) {
        if (!create)
          throw 'Operation aborted.';
      });
  })
  .then(function() {
    return readJSON(path.resolve(config.pjson.packages, '.dependencies.json'));
  })
  .then(function(depsJSON) {
    config.deps = setSerializedDeps(depsJSON);

    if (config.loader.upgrade16)
      return;

    if (!config.pjson.jspmAware && !allowNoConfig || prompts)
      return initPrompts(!process.env.jspmConfigPath || config.pjson.file.timestamp == -1 || !config.pjson.jspmAware);
  })
  .then(function(initInstalls) {
    if (!config.loader.upgrade16 || !config.pjson.jspmAware)
      return initInstalls;

    if (allowNoConfig)
      throw new Error('The current project needs to be upgraded to jspm 0.17 before any further operations can run. Run %jspm init% to start the upgrade.');

    // NB complete testing here
    return ui.confirm('jspm will now attempt to upgrade your project to the 0.17 configuration.\nAre you sure you want to proceed?', true, {
      info: 'This is an beta release of jspm 0.17, which is not yet fully stable. Make sure you have a backup of your project.'
    })
    .then(function(doUpgrade) {
      if (!doUpgrade)
        return Promise.reject('jspm 0.17-beta upgrade cancelled.');

      config.loader.file.rename(config.pjson.configFile);

      // copy browser configurations from config.loader to config.loaderBrowser
      var baseURL = config.loader.file.getValue(['baseURL'], 'string');
      if (baseURL) {
        config.loader.file.remove(['baseURL']);
        config.loader.file.setValue(['browserConfig', 'baseURL'], baseURL);
      }
      var paths = config.loader.file.getObject(['paths']);
      if (paths) {
        config.loader.file.remove(['paths']);
        config.loader.file.setObject(['browserConfig', 'paths'], paths);
      }

      ui.log('info', '');

      ui.log('info', 'Upgrading jspm 0.16 Node core libraries to jspm 0.17 universal implementations...\n');

      processNodelibs(config.loader, config.pjson, true);

      ui.log('info', 'Checking all overrides into the package.json file to ensure reproducibility independent of the jspm registry...\n');

      var allInstalledPackages = [];

      Object.keys(config.loader.baseMap).forEach(function(key) {
        var dep = config.loader.baseMap[key];
        if (dep.exactName && allInstalledPackages.indexOf(dep.exactName) == -1)
          allInstalledPackages.push(dep.exactName);
      });
      Object.keys(config.loader.depMap).forEach(function(key) {
        var curMap = config.loader.depMap[key];
        Object.keys(curMap).forEach(function(key) {
          var dep = curMap[key];
          if (dep.exactName && allInstalledPackages.indexOf(dep.exactName) == -1)
            allInstalledPackages.push(dep.exactName);
        });
      });

      // for each installed package, retreive its override and add it to the package.json file
      // extending the override in the package.json file itself
      var endpoint = require('../registry').load(require('./global-config').config.defaultRegistry);
      var semver = require('../semver');
      var upgradePackageConfig = require('../package').upgradePackageConfig;
      return Promise.all(allInstalledPackages.map(function(dep) {
        var pkg = new PackageName(dep);

        var overrideVersion = Object.keys(config.pjson.overrides)
        .filter(function(overrideName) {
          return overrideName.startsWith(pkg.name + '@');
        })
        .map(function(overrideName) {
          return overrideName.split('@').pop();
        })
        .filter(function(overrideVersion) {
          return semver.match('^' + overrideVersion, pkg.version);
        })
        .sort(semver.compare).pop();

        var manualOverride = overrideVersion && config.pjson.overrides[pkg.name + '@' + overrideVersion] || {};

        // use registry override + previous package.json manual override
        return endpoint.getOverride(pkg.registry, pkg.package, pkg.version, manualOverride)
        .then(function(override) {
          override = override || {};

          // this is an upgrade by reference
          upgradePackageConfig(override.systemjs || override);

          // persist the override for reproducibility
          config.pjson.overrides[pkg.exactName] = override;
        });
      }))
      .then(function() {
        config.loaded = true;
        ui.log('info', 'Re-downloading all installed packages...\n');
        return require('../install').install(true);
      })
      .then(function() {
        // uninstall babel transpiler
        if (config.loader.transpiler == 'babel') {
          ui.log('info', 'Uninstalling jspm 0.16 Babel transpiler...\n');
          return require('../install').uninstall('babel');
        }
      })
      .then(function() {
        ui.log('info', 'Running jspm 0.17 project configuration prompts...\n');
        return initPrompts();
      })
      .then(function() {
        // always force transpiler download after init
        if (config.loader.transpiler == 'none')
          return;
        if (config.loader.transpiler.substr(0, 7) != 'plugin-')
          config.loader.transpiler = 'plugin-' + config.loader.transpiler;
        var installObj = {};
        installObj[config.loader.transpiler] = 'jspm:' + config.loader.transpiler;
        return require('../install').install(installObj, { dev: true });
      })
      .then(function() {
        return require('../core').checkDlLoader();
      })
      .then(function() {
        ui.log('ok', 'jspm 0.17-beta upgrade complete.\n\n' +
          'Some important breaking changes to note:\n\n' +
          wordWrap('• The %config.js% file has been renamed to %jspm.config.js% unless you were already using a custom config path for this.\n', process.stdout.columns - 4, 2, 0, true) + '\n' +
          wordWrap('• js extensions are required for module imports not inside packages. Eg %System.import(\'./test\')% will need to become %System.import(\'./test.js\')%.', process.stdout.columns - 4, 2, 0, true) + '\n' +
          '\nThere are also other smaller breaking changes in this release, described in the full changelog at https://github.com/jspm/jspm-cli/releases/tag/0.17.0-beta.\n' + '\n' +
          'Please report any issues or feedback to help improve this release and thanks for testing it out.');
      });
    });
  }))
  .then(function(initInstalls) {
    var installRequired = processNodelibs(config.loader, config.pjson, false);
    if (installRequired) {
      config.pjson.write();
      return require('../install').install(true).then(function () {
        return initInstalls;
      });
    } else {
      return initInstalls;
    }
  })
  .then(function(initInstalls) {
    config.loaded = true;

    if (initInstalls)
      return install.install(initInstalls, { dev: true })
      .then(function() {
        ui.log('info', '');
        ui.log('ok', 'jspm init completed. For the standard project layout see _http://jspm.io/0.17-beta-guide/creating-a-project.html_.\n' +
            'To rerun these init prompts at any time use %jspm init%.');
      });
  });
};

exports.loadSync = function(allowNoConfig) {
  if (config.loaded)
    return;

  if (loadPromise)
    throw 'Configuration file is already loading.';

  config.allowNoConfig = !!allowNoConfig;

  config.pjsonPath = process.env.jspmConfigPath || path.resolve(process.cwd(), 'package.json');

  config.pjson = new PackageConfig(config.pjsonPath);

  if (!allowNoConfig) {
    if (!config.pjson.jspmAware)
      throw 'Package.json file has not been initialized by jspm before. Run jspm init first.';

    if (!fs.existsSync(config.pjson.configFile))
      throw 'No project configuration file found looking for `' + config.pjson.configFile + '`.';
  }
  config.loader = new JspmSystemConfig(config.pjson.configFile);

  if (!config.loader.transpiler && allowNoConfig)
    config.loader.transpiler = 'none';

  var depsJSON;
  try {
    depsJSON = JSON.parse(fs.readFileSync(path.resolve(config.pjson.packages, '.dependencies.json')));
  }
  catch(e) {
    if (e.code == 'ENOENT')
      depsJSON = {};
    else
      throw e;
  }

  config.deps = setSerializedDeps(depsJSON);

  config.loaded = true;
  loadPromise = Promise.resolve();
};

function getSerializedDeps(deps) {
  return stringify(deps, function(name, value) {
    if (value instanceof PackageName)
      return value.toString();
    return value;
  });
}
function setSerializedDeps(serializedDeps) {
  var depRanges = {};
  Object.keys(serializedDeps).forEach(function(parentName) {
    depRanges[parentName] = { deps: {}, peerDeps: {} };
    var deps = serializedDeps[parentName].deps;
    var peerDeps = serializedDeps[parentName].peerDeps;

    // backwards compat / upgrade path for 0.17.0-beta.14 and below
    if (typeof deps != 'object') {
      deps = serializedDeps[parentName];
      peerDeps = {};
    }

    Object.keys(deps).forEach(function(dep) {
      depRanges[parentName].deps[dep] = new PackageName(deps[dep]);
    });
    Object.keys(peerDeps).forEach(function(dep) {
      depRanges[parentName].peerDeps[dep] = new PackageName(peerDeps[dep]);
    });
  });
  return depRanges;
}

exports.save = function() {
  if (config.allowNoConfig) {
    config.loader.write();
  }
  else {
    config.loader.write();
    config.pjson.write();
    mkdirp.sync(config.pjson.packages);
    fs.writeFileSync(path.resolve(config.pjson.packages, '.dependencies.json'), getSerializedDeps(config.deps));
  }
};

/*
 * Project creation prompts
 */
function initPrompts(newProject) {
  var pjson = config.pjson;
  var loader = config.loader;

  var base = path.dirname(config.pjsonPath);
  var baseURL = path.relative(base, pjson.baseURL);
  baseURL = baseURL ? baseURL + path.sep : '.';

  var libPath;
  var packagesPath;
  var useBaseURL = true;

  var useDevConfig, useBrowserConfig, useNodeConfig;

  var promptType;

  return Promise.resolve()
  // Init mode
  .then(function() {
    return ui.input('%Init mode (Quick, Standard, Custom)%', 'Quick', {
      info: 'Select an init mode for jspm project configuration.',
      options: ['Quick', 'Standard', 'Custom']
    })
    .then(function(_promptType) {
      promptType = _promptType.toLowerCase();
    });
  })
  // jspm prefix
  .then(function() {
    if (promptType != 'custom')
      return;

    return ui.confirm('%Prefix package.json properties under jspm?%', !!pjson.jspmPrefix)
    .then(function(jspmPrefix) {
      // unprefixing existing flattens
      if (pjson.jspmPrefix && !jspmPrefix && pjson.file.has(['jspm']))
        return ui.confirm('%Flatten existing package.json jspm properties?%', true, {
          info: 'Are you sure you want to flatten the %jspm% package.json properties? Any duplicate package.json property names will be overwritten.'
        })
        .then(function(confirm) {
          if (confirm)
            pjson.setPrefix(false);
        });
      // prefixing existing applies to new properties only
      pjson.setPrefix(jspmPrefix);
    });
  })
  // Project name
  .then(function() {
    return ui.input('%Local package name (recommended, optional)%', pjson.name, {
      edit: true,
      clearOnType: true,
      info: 'Enter a name for the project package.\n\nThis name will be used for importing local code.\neg via %System.import(\'' + (pjson.name || 'app') + '/module.js\')%.',
      validate: function(name) {
        if (name.indexOf(' ') != -1)
          return 'The package name should not contain any spaces.';
        if (name.match(/!|#/))
          return 'The package name should not contain characters ! or #.';
      }
    })
    .then(function(name) {
      pjson.name = name;
      if (name)
        loader.package = loader.package || {};
      else
        loader.package = null;
    });
  })
  // baseURL
  .then(function() {
    return ui.input('%package.json directories.baseURL%', baseURL || '.', {
      edit: true,
      info: 'Enter the file path to the shared public folder served to the browser.\n\n' +
          (loader.package ? 'If omitted, %paths% locations must be individually configured between the browser and Node environments (which will then be prompted here).' : 'By default this is taken to be the root project folder.'),
      validate: function(baseURL) {
        if (path.resolve(baseURL) != base && !inDir(path.resolve(baseURL), base))
          return 'The directories.baseURL path should be a subfolder within the project.';
      }
    })
    .then(function(_baseURL) {
      _baseURL = _baseURL || '.';
      if (_baseURL[_baseURL.length - 1] == path.sep)
        _baseURL = _baseURL.substr(0, baseURL.length - 1);
      pjson.baseURL = path.resolve(base, _baseURL);
      baseURL = path.relative(base, pjson.baseURL);
      baseURL = baseURL ? baseURL + path.sep : '.';
      // populate default paths given the new baseURL value
      if (newProject)
        pjson.populateDefaultPaths();
    });
  })
  // jspm config folder
  .then(function() {
    if (promptType == 'custom')
      return;

    // to make the prompts simple, we assume the config files are called jspm.config.js and
    // jspm.dev.js the question here is then just what folder they are located in
    var configDir = path.dirname(pjson.configFile);
    if (pjson.configFileDev && path.dirname(pjson.configFileDev) != configDir ||
        pjson.configFileBrowser && path.dirname(pjson.configFileBrowser) != configDir ||
        pjson.configFileNode && path.dirname(pjson.configFileNode) != configDir)
      return;

    return ui.input('%package.json configFiles folder%', (path.relative(base, configDir) || '.') + path.sep, {
      info: 'Enter the path to the folder to contain the SystemJS config files for jspm.',
      validate: function(configFolder) {
        if (path.relative(base, configFolder)[0] == '.')
          return 'The config file path should be a subfolder within the project.';
      }
    })
    .then(function(configFolder) {
      pjson.configFile = path.resolve(base, configFolder, path.basename(pjson.configFile));
      if (pjson.configFileDev)
        pjson.configFileDev = path.resolve(base, configFolder, path.basename(pjson.configFileDev));
      if (pjson.configFileBrowser)
        pjson.configFileBrowser = path.resolve(base, configFolder, path.basename(pjson.configFileBrowser));
      if (pjson.configFileNode)
        pjson.configFileNode = path.resolve(base, configFolder, path.basename(pjson.configFileNode));
    });
  })
  // (custom) jspm config file path
  .then(function() {
    if (promptType != 'custom')
      return;

    return ui.input('%package.json configFiles.jspm%', path.relative(base, pjson.configFile), {
      info: 'Enter a custom config file path.\n\nOnly necessary if you would like to customize the config file name or location.',
      validate: function(configFile) {
        if (!inDir(path.resolve(base, configFile), base))
          return 'The config file path should be a subfolder within the project.';
      }
    })
    .then(function(configFile) {
      pjson.configFile = path.resolve(base, configFile);
    });
  })
  // create a separate jspm.dev.js file?
  .then(function() {
    return ui.confirm('%Use package.json configFiles.jspm:dev?%', !!loader.devFile, {
      info: 'Use a separate %jspm.dev.js% config file?\n\n' +
          'This would be used for all %--dev% install configuration.'
    })
    .then(function(useDev) {
      useDevConfig = useDev;
    });
  })
  // (custom) jspm dev config file path
  .then(function() {
    if (!loader.devFile || promptType != 'custom')
      return;

    return ui.input('%package.json configFiles.jspm:dev%', path.relative(base, pjson.configFileDev), {
      info: 'Enter a custom file path to the local dev loader configuration file.'
    })
    .then(function(configFile) {
      pjson.configFileDev = path.resolve(base, configFile);
    });
  })
  .then(function() {
    // we read the config file after setting the name to support picking up existing files
    // (assuming the first file did not exist, otherwise we still default to that content)
    if (useDevConfig)
      loader.createConfigFile('dev');
    else
      loader.removeConfigFile('dev');
  })
  // create a separate jspm.browser.js file?
  .then(function() {
    if (!loader.browserFile && promptType == 'quick')
      return;

    return ui.confirm('%Use package.json configFiles.jspm:browser?%', !!loader.browserFile, {
      info: 'Use a separate %jspm.browser.js% config file?\n\n' +
          'This would be used for browser paths and bundle configuration.'
    })
    .then(function(useBrowser) {
      useBrowserConfig = useBrowser;
    });
  })
  // (custom) jspm browser config file path
  .then(function() {
    if (!loader.browserFile || promptType != 'custom')
      return;

    return ui.input('%package.json configFiles.jspm:browser%', path.relative(base, pjson.configFileBrowser), {
      info: 'Enter a custom file path to the local browser loader configuration file.'
    })
    .then(function(configFile) {
      pjson.configFileBrowser = path.resolve(base, configFile);
    });
  })
  .then(function() {
    if (useBrowserConfig)
      loader.createConfigFile('browser');
    else
      loader.removeConfigFile('browser');
  })
  // create a separate jspm.node.js file?
  .then(function() {
    if (!loader.nodeFile && promptType != 'custom')
      return;

    return ui.confirm('%Use package.json configFiles.jspm:node?%', !!loader.nodeFile, {
      info: 'Use a separate %jspm.node.js% config file?\n\n' +
          'This would be used for all %--node% install configuration.'
    })
    .then(function(useNode) {
      useNodeConfig = useNode;
    });
  })
  // (custom) jspm node config file path
  .then(function() {
    if (!loader.nodeFile || promptType != 'custom')
      return;

    return ui.input('%package.json configFiles.jspm:node%', path.relative(base, pjson.configFileNode), {
      info: 'Enter a custom file path to the local node loader configuration file.'
    })
    .then(function(configFile) {
      pjson.configFileNode = path.resolve(base, configFile);
    });
  })
  .then(function() {
    if (useNodeConfig)
      loader.createConfigFile('node');
    else
      loader.removeConfigFile('node');
  })
  // baseURL in browser
  .then(function() {
    return ui.input('%SystemJS.config browser baseURL' + (loader.package ? ' (optional)%' : '%'), loader.browserBaseURL || (newProject ? '/' : ''), {
      edit: true,
      info: 'Enter the browser baseURL.\n\nThis must be the absolute URL matching the local %' + baseURL + '% public folder served in the browser.',
      validate: function(baseURL) {
        if (!baseURL && !loader.package)
          return 'The baseURL is only optional when a local package name (previously prompted) is used to alias local paths. Usually this should be %/% or the full path to the server.';
      }
    })
    .then(function(baseURL) {
      loader.browserBaseURL = baseURL;

      // we can opt-out of baseURL here
      if (!baseURL) {
        useBaseURL = false;
        loader.browserBaseURL = null;
      }
    });
  })
  // Local package path
  .then(function() {
    if (!loader.package)
      return;

    var defaultLibPath = loader.nodeLibURL ? path.relative(base, loader.nodeLibURL) : (path.relative(base, pjson.lib) || '.') + path.sep;

    return ui.input('%SystemJS.config Node local project path%', defaultLibPath, {
      info: 'Enter the path to the folder containing the local project code.\n\nThis is the folder containing the %' + pjson.name + '% package code.',
      validate: function(lib) {
        if (!inDir(path.resolve(base, lib) + path.sep, pjson.baseURL, false)) {
          return 'The local project path must be within ' + (pjson.baseURL == base ? 'the project root.' : 'the %' + path.relative(base, pjson.baseURL) + '% baseURL.');
        }
      }
    })
    .then(function(lib) {
      libPath = lib;
      // clear libURL so it is regenerated
      if (newProject)
        loader.browserLibURL = null;
      loader.nodeLibURL = path.resolve(base, lib);
      return asp(mkdirp)(path.dirname(loader.nodeLibURL));
    });
  })
  // URL to local package in browser
  .then(function() {
    if (!loader.package)
      return;

    var defaultLibURL = loader.browserLibURL || (loader.browserBaseURL ? '' : '/') + (path.relative(pjson.baseURL, pjson.lib) ? path.relative(pjson.baseURL, pjson.lib) + '/' : '');

    if (promptType == 'quick' && useBaseURL && isPlain(defaultLibURL)) {
      loader.browserLibURL = defaultLibURL;
      return;
    }

    return ui.input('%SystemJS.config browser local project URL to ' + libPath + '%', defaultLibURL, {
      info: 'Enter the browser URL for the folder containing the local project code.\n\nThis should be the served %' + libPath + '% folder.' +
        (loader.browserBaseURL ? ' Non-absolute plain paths are taken to be baseURL-relative.' : ''),
    })
    .then(function(libURL) {
      loader.browserLibURL = libURL;
    });
  })
  // jspm_packages folder path
  .then(function() {
    if (promptType == 'quick' && useBaseURL && isPlain(loader.browserPackagesURL))
      return;

    return ui.input('%package.json directories.packages%', (path.relative(base, pjson.packages) || '.') + path.sep, {
      info: 'Enter the path to the local jspm packages folder.\n\nOnly necessary if you would like to customize this folder name or location.',
      validate: function(packages) {
        if (!inDir(path.resolve(base, packages), pjson.baseURL))
          return 'The directories.packages path should be a subfolder within ' + (pjson.baseURL == base ? 'the project.' : 'the %' + path.relative(base, pjson.baseURL) + '% baseURL.');
      }
    })
    .then(function(packages) {
      pjson.packages = path.resolve(base, packages);
      loader.nodePackagesURL = pjson.packages;
      packagesPath = path.relative(base, pjson.packages);
    });
  })
  // URL to jspm_packages in browser
  .then(function() {

    var defaultPackagesURL = loader.browserPackagesURL || (loader.browserBaseURL ? '' : '/') + (path.relative(base, pjson.packages) ? path.relative(base, pjson.packages) + '/' : '');

    if (promptType == 'quick' && useBaseURL && isPlain(defaultPackagesURL)) {
      loader.browserPackagesURL = defaultPackagesURL;
      return;
    }

    return ui.input('%SystemJS.config browser URL to ' + packagesPath + '%', defaultPackagesURL, {
      info: 'Enter the browser URL for the jspm_packages folder.' + (loader.browserBaseURL ? '\n\nNon-absolute plain paths are taken to be baseURL-relative.' : ''),
      validate: function(packagesURL) {
        if (defaultPackagesURL[0] != '.' && packagesURL[0] == '.')
          return 'The jspm_packages URL should not be a relative URL.';
      }
    })
    .then(function(packagesURL) {
      loader.browserPackagesURL = packagesURL;
    });
  })
  // main entry point
  .then(function() {
    if (!loader.package)
      return;

    var defaultMain = loader.package.main || pjson.main || (pjson.name + '.js');

    return ui.input('%SystemJS.config local package main%', defaultMain, {
      info: 'Enter the main entry point of your package within the %' + libPath + '% folder.\n\n' +
          'This allows %System.import(\'' + pjson.name + '\')% to resolve to %' + libPath + '/' + defaultMain + '%.'
    })
    .then(function(main) {
      pjson.main = main;
      loader.package.main = main;
    });
  })
  // format
  .then(function() {
    if (promptType == 'quick')
      return;

    if (!loader.package)
      return;

    return ui.input('%SystemJS.config local package format (esm, cjs, amd)%', loader.package.format || 'esm', {
      info: 'Enter the module format of your local project code (within `' + libPath + '`).\n\nThe default option is esm (ECMAScript Module).',
      options: ['esm', 'cjs', 'amd', 'global', 'system', 'systemjs', 'register'],
      edit: true,
      optionalOptions: true
    })
    .then(function(format) {
      format = format.toLowerCase();
      if (format == 'system' || format == 'systemjs')
        format = 'register';
      if (format)
        loader.package.format = format;
      else
        delete loader.package.format;
    });
  })
  // transpiler
  .then(function() {
    var transpilers = ['babel', 'traceur', 'typescript', 'none'];

    var curTranspiler = loader.transpiler;

    // default transpiler will be typescript only if the main loader file ends with .ts or .tsx
    var defTranspiler = /\.tsx?$/.test(loader.package && loader.package.main) ? 'typescript' : 'babel';

    if (curTranspiler && (curTranspiler.substr(0, 7) == 'plugin-' || curTranspiler.substr(0, 7) == 'loader-') &&
        transpilers.indexOf(curTranspiler.substr(7).toLowerCase()) != -1)
      curTranspiler = curTranspiler.substr(7).toLowerCase();
    else if (curTranspiler)
      transpilers.push(curTranspiler);

    return ui.input('%SystemJS.config transpiler (Babel, Traceur, TypeScript, None)%', curTranspiler || defTranspiler, {
      info: 'Select a transpiler to use for ES module conversion.\n\n' +
          'The transpiler is used when detecting modules with %import% or %export% statements, or ' +
          'for modules with %format: "esm"% metadata set.',
      options: transpilers,
      validate: function(transpiler) {
        if (transpiler !== curTranspiler && transpilers.indexOf(transpiler.toLowerCase()) == -1)
          return 'Invalid transpiler option.';
      }
    })
    .then(function(transpiler) {
      if (transpiler === curTranspiler || transpiler === 'none')
        return;

      // set transpiler on BOTH the transpiler and local package loader config
      if (loader.package) {
        var tsExts = transpiler === 'typescript' && loader.package.main && /\.(tsx?$)/i.exec(loader.package.main);
        // when typescript is the transpiler set the default package extension to .ts or .tsx
        if (!loader.package.defaultExtension && tsExts)
            loader.package.defaultExtension = tsExts[1];

        var pkgMeta = loader.package.meta = loader.package.meta || {};
        var pkgExt = transpiler === 'typescript' && tsExts ? '*.' + tsExts[1] : '*.js';

        pkgMeta[pkgExt] = pkgMeta[pkgExt] || {};
        if (!pkgMeta[pkgExt].loader || pkgMeta[pkgExt].loader == loader.transpiler)
          pkgMeta[pkgExt].loader = pkgMeta[pkgExt].loader || 'plugin-' + transpiler.toLowerCase();
      }
      loader.transpiler = 'plugin-' + transpiler.toLowerCase();

      // download transpiler
      var installObj = {};
      installObj[loader.transpiler] = 'jspm:' + loader.transpiler;

      // the init function itself returns an install object, which is run in init
      return installObj;
    });
  });
}
