/*
 * Copyright (c) 2019, 2020 Oracle and/or its affiliates. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package main

import (
	"bytes"
	"fmt"
	"io"
	"os"
	"sort"
	"strings"
	"text/tabwriter"
	"text/template"

	"github.com/fnproject/cli/commands"
	"github.com/fnproject/cli/common"
	"github.com/fnproject/cli/common/color"
	"github.com/fnproject/cli/config"
	"github.com/spf13/viper"
	"github.com/urfave/cli"
)

func newFn() *cli.App {
	app := cli.NewApp()
	app.Name = "fn"
	app.Version = config.Version
	app.Authors = []cli.Author{{Name: "Fn Project"}}
	app.Description = "Fn Command Line Tool"
	app.EnableBashCompletion = true
	app.BashComplete = common.DefaultBashComplete
	app.Before = func(c *cli.Context) error {
		err := config.LoadConfiguration(c)
		if err != nil {
			return err
		}
		commandArgOverrides(c)
		return nil
	}
	app.Flags = []cli.Flag{
		cli.BoolFlag{
			Name:        "verbose,v", // v is taken for version by default with urfave/cli
			Usage:       "Use --verbose to enable verbose mode for debugging",
			Destination: &common.GlobalVerbose,
		},
		cli.StringFlag{
			Name:   "context",
			Usage:  "Use --context to select context configuration file",
			EnvVar: "FN_CONTEXT",
		},
		cli.StringFlag{
			Name:  "registry",
			Usage: "Use --registry to select registry",
		},
	}
	cli.VersionFlag = cli.BoolFlag{
		Name:  "version",
		Usage: "Display version",
	}

	// Override app template
	// AppHelpTemplate is the text template for the Default help topic.
	// cli.go uses text/template to render templates. You can render custom help text by setting this variable.
	cli.AppHelpTemplate = `
{{if not .ArgsUsage}}{{boldred .Description}}	{{boldred "-"}}	{{boldred "Version "}}{{boldred .Version}}
	
{{bold "ENVIRONMENT VARIABLES"}}
	FN_API_URL		 {{italic "Fn server address"}}
	FN_REGISTRY		 {{italic "Docker / Podman registry to push images to, use username only to push to Docker Hub - [[registry.hub.docker.com/]USERNAME]"}}{{if .VisibleCommands}}
		
{{bold "GENERAL COMMANDS"}}{{end}}{{else}}{{range .VisibleCategories}}{{if .Name}}{{bold .Name}}{{end}}{{end}}
	{{boldcyan .HelpName}}{{if .Usage}}{{" - "}}{{italic .Usage}}
	
{{bold "USAGE"}}
	{{boldcyan "fn"}}{{if .VisibleFlags}}{{cyan " [global options] "}}{{end}}` + color.BoldCyan(`{{trim .HelpName "fn"}}`) + `{{" "}}{{if .Flags}}{{yellow "[command options] "}}{{end}}{{if .ArgsUsage}}{{brightred .ArgsUsage}}{{end}}{{if .Description}}
	
{{bold "DESCRIPTION"}}
	{{.Description}}{{end}}{{end}}{{end}}{{range .VisibleCategories}}{{if .Name}}
	
{{bold .Name}}{{end}}{{range .VisibleCommands}}
	{{join .Names ", "}}				 {{.Usage}}{{end}}{{end}}{{if .VisibleFlags}}
		
{{if not .ArgsUsage}}{{bold "GLOBAL OPTIONS"}}{{else}}{{bold "COMMAND OPTIONS"}}{{end}}
  {{range $index, $option := .VisibleFlags}}{{if $index}}
  {{end}}{{$option}}{{end}}{{end}}
		
{{bold "FURTHER HELP:"}}	{{italic "See "}}{{"'"}}{{brightcyan .HelpName}}{{brightcyan " <command> --help"}}{{"'"}}{{italic " for more information about a command."}}{{if not .ArgsUsage}}
	
{{bold "LEARN MORE:"}}	{{underlinebrightred "https://github.com/fnproject/fn"}}{{else}}{{end}}
	`
	// Override subcommand template
	// SubcommandHelpTemplate is the text template for the subcommand help topic.
	// cli.go uses text/template to render templates. You can render custom help text by setting this variable.
	cli.SubcommandHelpTemplate = `
{{range .VisibleCategories}}{{if .Name}}{{bold .Name}}{{end}}{{end}}
	{{boldcyan .Name}}{{if .Usage}}{{" - "}}{{italic .Usage}}
		
{{bold "USAGE"}}
	{{boldcyan "fn"}}{{if .VisibleFlags}}{{cyan " [global options]"}}{{end}}` + color.BoldCyan(`{{trim .HelpName "fn"}}`) + `{{" "}}{{if .Flags}}{{yellow "[command options] "}}{{end}}{{if .ArgsUsage}}{{brightred .ArgsUsage}}{{end}}{{end}}{{if .Description}}
		
{{bold "DESCRIPTION"}}
	{{.Description}}{{end}}{{if .Commands}}
		
{{bold "SUBCOMMANDS"}}{{range .Commands}}
	{{join .Names ", "}}			{{.Usage}}{{end}}{{end}}{{if .VisibleFlags}}
		
{{bold "COMMAND OPTIONS"}}{{range .VisibleFlags}}
	{{.}}{{end}}{{end}}{{if .Commands}}

{{bold "FURTHER HELP:"}}	{{italic "See "}}{{"'"}}{{brightcyan .HelpName}}{{brightcyan " <subcommand> --help"}}{{"'"}}{{italic " for more information about a command."}}{{end}}
`
	//Override command template
	// CommandHelpTemplate is the text template for the command help topic.
	// cli.go uses text/template to render templates. You can render custom help text by setting this variable.
	cli.CommandHelpTemplate = `
{{if .Category}}{{bold .Category}}{{end}}
	{{boldcyan .HelpName}}{{if .Usage}}{{" - "}}{{italic .Usage}}
		
{{bold "USAGE"}}
	{{boldcyan "fn"}}{{cyan " [global options]"}}` + color.BoldCyan(`{{trim .HelpName "fn"}}`) + `{{" "}}{{if .Flags}}{{yellow "[command options] "}}{{end}}{{if .ArgsUsage}}{{brightred .ArgsUsage}}{{" "}}{{end}}{{end}}{{if .Description}}
		
{{bold "DESCRIPTION"}}
	{{.Description}}{{end}}{{if .Subcommands}}
		
{{bold "SUBCOMMANDS"}}{{range .Subcommands}}
	{{join .Names ", "}}			{{.Usage}}{{end}}{{end}}{{if .VisibleFlags}}
		
{{bold "COMMAND OPTIONS"}}
	{{range .Flags}}{{.}}
	{{end}}{{if .Subcommands}}
		
{{bold "FURTHER HELP:"}}	{{italic "See "}}){{"'"}}{{brightcyan .HelpName}}{{brightcyan " <subcommand> --help"}}{{"'"}}{{italic "for more information about a command."}}{{end}}{{end}}
`
	app.CommandNotFound = func(c *cli.Context, cmd string) {
		fmt.Fprintf(os.Stderr, color.Bold("\nFn: ")+"'"+color.Red("%v")+"' is not a Fn Command ", cmd)
		//fmt.Fprintf(os.Stderr, "\n\nNote the fn CLI command structure has changed, please change your command to use the new structure.\n")
		fmt.Fprintf(os.Stderr, color.Italic("\n\nSee ")+"'"+color.BrightCyan("fn <command> --help")+"'"+color.Italic(" for more information."))
		fmt.Fprintf(os.Stderr, color.BrightCyan(" Note ")+"the fn CLI command structure has changed, please change your command to use the new structure.\n")
		os.Exit(1)
	}

	app.Commands = append(app.Commands, commands.GetCommands(commands.Commands)...)
	app.Commands = append(app.Commands, commands.VersionCommand())

	sort.Sort(cli.FlagsByName(app.Flags))
	sort.Sort(cli.CommandsByName(app.Commands))

	prepareCmdArgsValidation(app.Commands)

	cli.HelpPrinter = func(w io.Writer, templ string, data interface{}) {
		printHelpCustom(w, templ, data, color.Colors)
	}

	return app
}

//Trim HelpName, removing 'fn' from the HelpName string
func TrimLeftChars(s string, n int) string {
	m := 0
	for i := range s {
		if m >= n {
			return s[i:]
		}
		m++
	}
	return s[:0]
}

//Override function for customised app template
func printHelpCustom(out io.Writer, templ string, data interface{}, customFunc map[string]interface{}) {
	funcMap := color.Colors
	for key, value := range customFunc {
		funcMap[key] = value
	}

	w := tabwriter.NewWriter(out, 1, 8, 2, ' ', 0)
	t := template.Must(template.New("temp").Funcs(funcMap).Parse(templ))
	err := t.Execute(w, data)
	if err != nil {
		fmt.Println("CLI TEMPLATE ERROR:")
		return
	}
	w.Flush()
}

func parseArgs(c *cli.Context) ([]string, []string) {
	args := strings.Split(c.Command.ArgsUsage, " ")
	var reqArgs []string
	var optArgs []string
	for _, arg := range args {
		if strings.HasPrefix(arg, "[") {
			optArgs = append(optArgs, arg)
		} else if strings.Trim(arg, " ") != "" {
			reqArgs = append(reqArgs, arg)
		}
	}
	return reqArgs, optArgs
}

func prepareCmdArgsValidation(cmds []cli.Command) {
	// TODO: refactor fn to use urfave/cli.v2
	// v1 doesn't let us validate args before the cmd.Action

	for i, cmd := range cmds {
		prepareCmdArgsValidation(cmd.Subcommands)
		if cmd.Action == nil {
			continue
		}
		action := cmd.Action
		cmd.Action = func(c *cli.Context) error {
			reqArgs, _ := parseArgs(c)
			if c.NArg() < len(reqArgs) {
				var help bytes.Buffer
				cli.HelpPrinter(&help, cli.CommandHelpTemplate, c.Command)
				if len(reqArgs)-c.NArg() == 1 {
					fmt.Fprintf(os.Stderr, color.Bold("\nFn: ")+c.Command.Usage+" using "+color.Cyan(c.Command.HelpName)+" requires the missing argument '"+color.Red("%v")+"'\n", strings.Join(reqArgs[c.NArg():], " "))
				} else {
					fmt.Fprintf(os.Stderr, color.Bold("\nFn: ")+c.Command.Usage+" using "+color.Cyan(c.Command.HelpName)+" requires the missing arguments '"+color.Red("%v")+"'\n", strings.Join(reqArgs[c.NArg():], " "))
				}
				fmt.Fprintf(os.Stderr, color.Italic("\nSee ")+"'"+color.BrightCyan(c.Command.HelpName+" --help")+"'"+color.Italic(" for more information.\n"))
				os.Exit(1)
			}
			return cli.HandleAction(action, c)
		}
		cmds[i] = cmd
	}
}

func init() {
	err := config.Init()
	if err != nil {
		fmt.Fprintf(os.Stderr, color.Bold("\nERROR: %v"), err)
		os.Exit(1)
	}
}

func commandArgOverrides(c *cli.Context) {
	if registry := c.String(config.EnvFnRegistry); registry != "" {
		viper.Set(config.EnvFnRegistry, registry)
	}
}

func main() {

	app := newFn()
	err := app.Run(os.Args)

	if err != nil {
		// TODO: this doesn't seem to get called even when an error returns from a command, but maybe urfave is doing a non zero exit anyways? nope: https://github.com/urfave/cli/issues/610
		fmt.Fprintf(os.Stderr, color.Bold("\nFn:")+" %v", err)
		fmt.Fprintf(os.Stderr, color.Italic("\n\nSee ")+"'"+color.BrightCyan("fn <command> --help")+"'"+color.Italic(" for more information."))
		fmt.Fprintf(os.Stderr, " Client version: %s\n", config.Version)
		os.Exit(1)
	}

}
