use std::{env, process};
use wasi_tests::{assert_errno, create_file, create_tmp_dir, open_scratch_directory, TESTCONFIG};

unsafe fn test_path_symlink_trailing_slashes(dir_fd: wasi::Fd) {
    if TESTCONFIG.support_dangling_filesystem() {
        // Dangling symlink: Link destination shouldn't end with a slash.
        assert_errno!(
            wasi::path_symlink("source", dir_fd, "target/")
                .expect_err("link destination ending with a slash should fail"),
            wasi::ERRNO_NOENT
        );

        // Dangling symlink: Without the trailing slash, this should succeed.
        wasi::path_symlink("source", dir_fd, "target")
            .expect("link destination ending with a slash");
        wasi::path_unlink_file(dir_fd, "target").expect("removing a file");
    }

    // Link destination already exists, target has trailing slash.
    wasi::path_create_directory(dir_fd, "target").expect("creating a directory");
    assert_errno!(
        wasi::path_symlink("source", dir_fd, "target/")
            .expect_err("link destination already exists"),
        unix => wasi::ERRNO_EXIST,
        windows => wasi::ERRNO_NOENT
    );
    wasi::path_remove_directory(dir_fd, "target").expect("removing a directory");

    // Link destination already exists, target has no trailing slash.
    wasi::path_create_directory(dir_fd, "target").expect("creating a directory");
    assert_errno!(
        wasi::path_symlink("source", dir_fd, "target")
            .expect_err("link destination already exists"),
        unix => wasi::ERRNO_EXIST,
        windows => wasi::ERRNO_NOENT
    );
    wasi::path_remove_directory(dir_fd, "target").expect("removing a directory");

    // Link destination already exists, target has trailing slash.
    create_file(dir_fd, "target");

    assert_errno!(
        wasi::path_symlink("source", dir_fd, "target/")
            .expect_err("link destination already exists"),
        unix => wasi::ERRNO_NOTDIR,
        windows => wasi::ERRNO_NOENT,
        wasi::ERRNO_EXIST
    );
    wasi::path_unlink_file(dir_fd, "target").expect("removing a file");

    // Link destination already exists, target has no trailing slash.
    create_file(dir_fd, "target");

    assert_errno!(
        wasi::path_symlink("source", dir_fd, "target")
            .expect_err("link destination already exists"),
        unix => wasi::ERRNO_EXIST,
        windows => wasi::ERRNO_NOENT
    );
    wasi::path_unlink_file(dir_fd, "target").expect("removing a file");
}

fn main() {
    let mut args = env::args();
    let prog = args.next().unwrap();
    let arg = if let Some(arg) = args.next() {
        arg
    } else {
        eprintln!("usage: {} <scratch directory>", prog);
        process::exit(1);
    };

    // Open scratch directory
    let base_dir_fd = match open_scratch_directory(&arg) {
        Ok(dir_fd) => dir_fd,
        Err(err) => {
            eprintln!("{}", err);
            process::exit(1)
        }
    };

    const DIR_NAME: &str = "path_symlink_trailing_slashes_dir.cleanup";
    let dir_fd;
    unsafe {
        dir_fd = create_tmp_dir(base_dir_fd, DIR_NAME);
    }

    // Run the tests.
    unsafe { test_path_symlink_trailing_slashes(dir_fd) }

    unsafe { wasi::path_remove_directory(base_dir_fd, DIR_NAME).expect("failed to remove dir") }
}
