
import { FetchSecretsInventoryFilterConfigLastModifiedAtOperatorEnum } from '../clients/cloud-vault-reporting/src/apis/VaultReportingServiceApi.js';

/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: BUSL-1.1
 */

const flattenFilters = (filters = []) => {
  return filters.reduce((acc, filter) => {
    const {
      field,
      operator,
      value
    } = filter;
    const key = `filter${field.charAt(0).toUpperCase()}${field.slice(1)}`;
    const dateOperators = {
      '>': FetchSecretsInventoryFilterConfigLastModifiedAtOperatorEnum.GREATER_THAN,
      '<': FetchSecretsInventoryFilterConfigLastModifiedAtOperatorEnum.LESS_THAN
    };
    if (value.type === 'duration') {
      const serializedOperator = dateOperators[operator];
      acc[`${key}Operator`] = serializedOperator;
      acc[`${key}RelativeToNow`] = value.value;
    } else {
      acc[key] = value.value;
    }
    return acc;
  }, {});
};

/**
 * Formats an applied filter for display.
 * @param filter The filter to format.
 * @param label An optional label to use instead of the filter's field name.
 * @returns An array of formatted filter tags.
 *
 * Handles different value types:
 * - Boolean: Displays the label if true, nothing if false.
 * - Duration: Converts hours to a human-readable format (e.g., "30 days ago").
 * - String/List: Displays each value with the appropriate operator.
 */

const formatAppliedFilter = (filter, label) => {
  const {
    field,
    operator,
    value
  } = filter;
  const formattedLabel = label || field;

  // Boolean types just show the label if true and nothing if false
  if (value.type === 'boolean') {
    if (value.value === true) {
      return [{
        label: formattedLabel,
        field,
        value: 'true'
      }];
    }
    return [];
  }

  // Since several filter types can have multiple values resulting in multiple tags we will normalize to an array
  const values = Array.isArray(value.value) ? value.value : [value.value];
  return values.map(val => {
    let formattedValue = val;
    let formattedOperator = operator;
    let operatorPrefix = '';
    if (operator === '=') {
      formattedOperator = ':';
    }
    if (value.type === 'duration') {
      operatorPrefix = operator;
      const valueInHours = parseInt(val, 10);
      const isNegative = valueInHours < 0;
      const suffix = isNegative ? ' ago' : '';
      const absHours = Math.abs(valueInHours);
      const unit = absHours % 24 === 0 ? 'days' : 'hours';
      const amount = unit === 'days' ? absHours / 24 : absHours;

      // Invert the operator for negative durations to maintain logical consistency (e.g: < -1h means more than 1 hours ago)
      if (isNegative) {
        formattedOperator = operator === '>' ? '<' : '>';
      }
      formattedValue = `${amount} ${unit}${suffix}`;
    }
    // All but equality(:) operators should have a space before them in the output
    if (formattedOperator !== ':') {
      formattedOperator = ` ${formattedOperator}`;
    }
    return {
      label: `${formattedLabel}${formattedOperator} ${formattedValue}`,
      field,
      value: `${operatorPrefix}${val}`
    };
  });
};

/**
 * Generates lookback filter options for a given array of days.
 * @param days Array of days to generate lookback options for. Defaults to [-7, -30, -60, -90].
 * @returns Array of lookback filter options with name and value properties.
 *
 * Each option's name is formatted as "X days ago" for negative values and "X days" for positive values.
 * The value is formatted as a duration string in hours (e.g., "<168h" for -7 days).
 */
const getLookbackFilterOptions = (days = [-7, -30, -60, -90]) => {
  return days.map(day => {
    return {
      name: day < 0 ? `${Math.abs(day)} days ago` : `${day} days`,
      value: `<${day * 24}h`
    };
  });
};

// Export both named and default to satisfy different import patterns
const filters = {
  flattenFilters
};

export { filters as default, flattenFilters, formatAppliedFilter, getLookbackFilterOptions };
//# sourceMappingURL=filters.js.map
