/**
 * Copyright IBM Corp. 2016, 2025
 * SPDX-License-Identifier: BUSL-1.1
 */

import Component from '@glimmer/component';
import { action } from '@ember/object';
import { underscore } from '@ember/string';
import Form from 'vault/forms/form';

import type PkiRoleModel from 'vault/models/pki/role';
import type PkiConfigGenerateForm from 'vault/forms/secrets/pki/config/generate';
import type PkiKeyForm from 'vault/forms/secrets/pki/key';
import type { HTMLElementEvent } from 'forms';
/**
 * @module PkiKeyParameters
 * PkiKeyParameters components are used to display a list of key bit options depending on the selected key type. The key bits field is disabled until a key type is selected.
 * If the component renders in a group, other attrs may be passed in and will be rendered using the <FormField> component
 * @example
 * ```js
 * <PkiKeyParameters @model={{@model}} @fields={{fields}}/>
 * ```
 * @param {class} model - The pki/role, pki/action, pki/key model.
 * @param {string} fields - Array of attributes from a formFieldGroup generated by the @withFormFields decorator ex: [{ name: 'attrName', type: 'string', options: {...} }]
 */
interface Args {
  model: PkiRoleModel | PkiKeyForm | PkiConfigGenerateForm;
}
interface TypeOptions {
  rsa: string;
  ec: string;
  ed25519: string;
  any: string;
}
interface BitOptions {
  [key: string]: Array<string>;
}

// first value in array is the default bits for that key type
const KEY_BITS_OPTIONS: BitOptions = {
  rsa: ['2048', '3072', '4096', '0'],
  ec: ['256', '224', '384', '521', '0'],
  ed25519: ['0'],
  any: ['0'],
};

export default class PkiKeyParameters extends Component<Args> {
  // shim to support both model and form types until all models can be migrated
  getValue = (key: string) => {
    const { model } = this.args;
    if (model instanceof Form) {
      return model.data[underscore(key) as keyof typeof model.data];
    }
    return model[key as keyof typeof model];
  };

  setValue = (key: string, value: unknown) => {
    const { model } = this.args;
    const modelKey = model instanceof Form ? underscore(key) : key;
    model.set(modelKey, value);
  };

  get keyBitOptions() {
    const keyType = this.getValue('keyType');
    return keyType ? KEY_BITS_OPTIONS[keyType] : [];
  }

  @action handleSelection(name: string, selection: string) {
    this.setValue(name, selection);

    if (['keyType', 'key_type'].includes(name) && Object.keys(KEY_BITS_OPTIONS)?.includes(selection)) {
      const bitOptions = KEY_BITS_OPTIONS[selection as keyof TypeOptions];
      if (bitOptions) {
        this.setValue('keyBits', bitOptions[0]);
      }
    }
  }

  @action onKeyBitsChange(event: HTMLElementEvent<HTMLInputElement>) {
    this.handleSelection(event.target.name, event.target.value);
  }
}
