
import Component from '@glimmer/component';
import FilterControl from './filter-control.js';
import { HdsButton, HdsButtonSet } from '@hashicorp/design-system-components/components';
import { on } from '@ember/modifier';
import { tracked } from '@glimmer/tracking';
import { precompileTemplate } from '@ember/template-compilation';
import { setComponentTemplate } from '@ember/component';
import { g, i } from 'decorator-transforms/runtime';

/**
 * Component for rendering a group of filter controls in the filter bar.
 * Handles staging and applying filters, and supports dynamic control visibility
 * based on dependencies.
 *
 * @class FilterControlGroup
 * @extends Component
 * @typedef {object} Args
 * @property {FilterFieldControl[]} controls - List of filter controls to render.
 * @property {Record<string, Filter>} appliedFilters - Currently applied filters.
 * @property {(filters: Record<string, Filter>) => void} onUpdateFilters - Callback when filters are updated.
 */
class FilterControlGroup extends Component {
  static {
    g(this.prototype, "stagedFilters", [tracked]);
  }
  #stagedFilters = (i(this, "stagedFilters"), void 0);
  /**
  * Filters staged by the user via form changes, but not yet applied.
  * Used to determine which controls are visible and to preview filter changes.
  */
  /**
  * Builds a new filters object from the provided form data.
  * Handles different filter types (list, lookback, single-select, boolean)
  * and merges with the currently applied filters.
  *
  * @param {FormData} [formData] - Form data from the filter controls form.
  * @returns {Record<string, Filter>} - The reconciled filters object.
  */
  reconcileControlGroupFormData(formData = new FormData()) {
    return this.args.controls.reduce((acc, control) => {
      const key = control.name;
      // Most filters can accept multiple values, this retrieves all as array in case of checkbox/radio etc..
      // and filters out any empty values to account for text inputs
      const allValuesForKey = formData.getAll(key).filter(Boolean);
      if (allValuesForKey.length === 0) {
        delete acc[key];
        return acc;
      }
      // default operator and simple string value
      let operator = '=';
      // Most filters accept multiple options
      let value = allValuesForKey;
      let type = 'list';
      // lookback/date filters have different type and operators
      if (control?.type === 'lookback') {
        const rawValue = allValuesForKey?.[0] || '';
        operator = rawValue.startsWith('>') ? '>' : '<';
        value = rawValue.replace(/^[<>]/, '');
        type = 'duration';
      } else if (control?.type === 'single-select') {
        value = allValuesForKey?.[0] || '';
        type = 'string';
      } else if (control?.type === 'boolean') {
        type = 'boolean';
        value = allValuesForKey.includes('on');
      }
      const filter = {
        field: key,
        operator,
        value: {
          type,
          value
        }
      };
      return {
        ...acc,
        [key]: filter
      };
    }, structuredClone(this.args.appliedFilters));
  }
  /**
  * Handles form submission for the filter controls.
  * Prevents default form behavior, reconciles filters from form data,
  * and invokes the onUpdateFilters callback with the new filters.
  *
  * @param {Event} event - The form submit event.
  */
  handleSubmit = event => {
    event.preventDefault();
    const formData = new FormData(event.target);
    const filters = this.reconcileControlGroupFormData(formData);
    this.args.onUpdateFilters(filters);
  };
  /**
  * Handles form change events for the filter controls.
  * Updates the stagedFilters property to reflect the current form state,
  * which can be used to check if conditional fields can be rendered.
  *
  * @param {Event} event - The form change event.
  */
  handleChange = event => {
    const formData = new FormData(event.currentTarget);
    this.stagedFilters = this.reconcileControlGroupFormData(formData);
  };
  clearFilters = () => {
    // Simulate an empty form submission to clear all filters for this group
    const formData = new FormData();
    const filters = this.reconcileControlGroupFormData(formData);
    this.args.onUpdateFilters(filters);
    this.stagedFilters = {};
  };
  /**
  * Returns the list of filter controls to render, filtered by their dependencies.
  * If stagedFilters are present, uses them to determine visibility; otherwise uses appliedFilters
  * to get initial state.
  *
  * @returns {FilterFieldControl[]} - The visible filter controls.
  */
  get controls() {
    // If  there aren't any staged filters yet we should check the initial appliedFilters state
    const filters = this.stagedFilters || this.args.appliedFilters;
    return this.args.controls.filter(control => {
      if (control.dependsOn && filters) {
        return control.dependsOn(filters);
      }
      if (control.dependsOn && !filters) {
        return false;
      }
      return true;
    });
  }
  static {
    setComponentTemplate(precompileTemplate("\n    <form class=\"filter-bar__control-group\" data-test-vault-reporting-filter-bar-control-group {{on \"submit\" this.handleSubmit}} {{on \"change\" this.handleChange}}>\n      <div class=\"filter-bar__control-group-body\">\n        {{#each this.controls as |control|}}\n          <FilterControl @filter={{control}} @appliedFilters={{@appliedFilters}} />\n        {{/each}}\n      </div>\n      <div class=\"filter-bar__control-group-footer\">\n        <HdsButtonSet>\n          <HdsButton data-test-vault-reporting-filter-bar-control-group-submit type=\"submit\" @text=\"Apply\" @size=\"small\" />\n          <HdsButton data-test-vault-reporting-filter-bar-control-group-clear @text=\"Clear\" @size=\"small\" @color=\"secondary\" {{on \"click\" this.clearFilters}} />\n        </HdsButtonSet>\n      </div>\n    </form>\n  ", {
      strictMode: true,
      scope: () => ({
        on,
        FilterControl,
        HdsButtonSet,
        HdsButton
      })
    }), this);
  }
}

export { FilterControlGroup as default };
//# sourceMappingURL=filter-control-group.js.map
