// Copyright 2011 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package packet

import (
	"bytes"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"hash"
	"testing"
	"time"

	"github.com/benburkert/openpgp/algorithm"
)

var privateKeyTests = []struct {
	privateKeyHex string
	creationTime  time.Time
}{
	{
		privKeyRSAHex,
		time.Unix(0x4cc349a8, 0),
	},
	{
		privKeyElGamalHex,
		time.Unix(0x4df9ee1a, 0),
	},
	{
		privKeyECDSA256Hex,
		time.Unix(0x56ce9b87, 0),
	},
	{
		privKeyECDSA384Hex,
		time.Unix(0x56ce9ff9, 0),
	},
	{
		privKeyECDSA521Hex,
		time.Unix(0x56cea099, 0),
	},
	{
		privKeyECDH256Hex,
		time.Unix(0x56d22753, 0),
	},
	{
		privKeyECDH384Hex,
		time.Unix(0x56d22d39, 0),
	},
	{
		privKeyECDH521Hex,
		time.Unix(0x56d22f96, 0),
	},
}

func TestPrivateKeyRead(t *testing.T) {
	for i, test := range privateKeyTests {
		packet, err := Read(readerFromHex(test.privateKeyHex))
		if err != nil {
			t.Errorf("#%d: failed to parse: %s", i, err)
			continue
		}

		privKey := packet.(*PrivateKey)

		if !privKey.Encrypted {
			t.Errorf("#%d: private key isn't encrypted", i)
			continue
		}

		err = privKey.Decrypt([]byte("wrong password"))
		if err == nil {
			t.Errorf("#%d: decrypted with incorrect key", i)
			continue
		}

		err = privKey.Decrypt([]byte("testing"))
		if err != nil {
			t.Errorf("#%d: failed to decrypt: %s", i, err)
			continue
		}

		if !privKey.CreationTime.Equal(test.creationTime) || privKey.Encrypted {
			t.Errorf("#%d: bad result, got: %#v", i, privKey)
		}
	}
}

func populateHash(hashFunc algorithm.Hash, msg []byte) (hash.Hash, error) {
	h := hashFunc.New()
	if _, err := h.Write(msg); err != nil {
		return nil, err
	}
	return h, nil
}

func TestECDSAPrivateKey(t *testing.T) {
	ecdsaPriv, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
	if err != nil {
		t.Fatal(err)
	}

	var buf bytes.Buffer
	if err := NewECDSAPrivateKey(time.Now(), ecdsaPriv).Serialize(&buf); err != nil {
		t.Fatal(err)
	}

	p, err := Read(&buf)
	if err != nil {
		t.Fatal(err)
	}

	priv, ok := p.(*PrivateKey)
	if !ok {
		t.Fatal("didn't parse private key")
	}

	sig := &Signature{
		PubKeyAlgo: algorithm.ECDSA,
		Hash:       algorithm.SHA256,
	}
	msg := []byte("Hello World!")

	h, err := populateHash(sig.Hash, msg)
	if err != nil {
		t.Fatal(err)
	}
	if err := sig.Sign(h, priv, nil); err != nil {
		t.Fatal(err)
	}

	if h, err = populateHash(sig.Hash, msg); err != nil {
		t.Fatal(err)
	}
	if err := priv.VerifySignature(h, sig); err != nil {
		t.Fatal(err)
	}
}

func TestIssue11505(t *testing.T) {
	// parsing a rsa private key with p or q == 1 used to panic due to a divide by zero
	_, _ = Read(readerFromHex("9c3004303030300100000011303030000000000000010130303030303030303030303030303030303030303030303030303030303030303030303030303030303030"))
}

// Generated with `gpg --export-secret-keys "Test Key 2"`
const privKeyRSAHex = "9501fe044cc349a8010400b70ca0010e98c090008d45d1ee8f9113bd5861fd57b88bacb7c68658747663f1e1a3b5a98f32fda6472373c024b97359cd2efc88ff60f77751adfbf6af5e615e6a1408cfad8bf0cea30b0d5f53aa27ad59089ba9b15b7ebc2777a25d7b436144027e3bcd203909f147d0e332b240cf63d3395f5dfe0df0a6c04e8655af7eacdf0011010001fe0303024a252e7d475fd445607de39a265472aa74a9320ba2dac395faa687e9e0336aeb7e9a7397e511b5afd9dc84557c80ac0f3d4d7bfec5ae16f20d41c8c84a04552a33870b930420e230e179564f6d19bb153145e76c33ae993886c388832b0fa042ddda7f133924f3854481533e0ede31d51278c0519b29abc3bf53da673e13e3e1214b52413d179d7f66deee35cac8eacb060f78379d70ef4af8607e68131ff529439668fc39c9ce6dfef8a5ac234d234802cbfb749a26107db26406213ae5c06d4673253a3cbee1fcbae58d6ab77e38d6e2c0e7c6317c48e054edadb5a40d0d48acb44643d998139a8a66bb820be1f3f80185bc777d14b5954b60effe2448a036d565c6bc0b915fcea518acdd20ab07bc1529f561c58cd044f723109b93f6fd99f876ff891d64306b5d08f48bab59f38695e9109c4dec34013ba3153488ce070268381ba923ee1eb77125b36afcb4347ec3478c8f2735b06ef17351d872e577fa95d0c397c88c71b59629a36aec"

// Generated by `gpg --export-secret-keys` followed by a manual extraction of
// the ElGamal subkey from the packets.
const privKeyElGamalHex = "9d0157044df9ee1a100400eb8e136a58ec39b582629cdadf830bc64e0a94ed8103ca8bb247b27b11b46d1d25297ef4bcc3071785ba0c0bedfe89eabc5287fcc0edf81ab5896c1c8e4b20d27d79813c7aede75320b33eaeeaa586edc00fd1036c10133e6ba0ff277245d0d59d04b2b3421b7244aca5f4a8d870c6f1c1fbff9e1c26699a860b9504f35ca1d700030503fd1ededd3b840795be6d9ccbe3c51ee42e2f39233c432b831ddd9c4e72b7025a819317e47bf94f9ee316d7273b05d5fcf2999c3a681f519b1234bbfa6d359b4752bd9c3f77d6b6456cde152464763414ca130f4e91d91041432f90620fec0e6d6b5116076c2985d5aeaae13be492b9b329efcaf7ee25120159a0a30cd976b42d7afe030302dae7eb80db744d4960c4df930d57e87fe81412eaace9f900e6c839817a614ddb75ba6603b9417c33ea7b6c93967dfa2bcff3fa3c74a5ce2c962db65b03aece14c96cbd0038fc"

// Generated with `gpg2 --export-secret-keys`
const (
	privKeyECDSA256Hex = "94a50456ce9b8713082a8648ce3d03010702030422d99a04c7e49deaf7645a56fe5c2eca06a13dbc84e02f024bb20f9bff40520a1eaea636fa9573642cb61203c635b54ad0233bdc7a0bc066f35fc17468f8f0e8fe07030207e110de909edd95e6b90020678a269dc74841719e57125e2e351c4675e6e1b1173beb0c96d1cf11d284fb51527624c7222a8a7802944b528c7f6eec6699d4837ca5cee22160550d18148f6af0368c"
	privKeyECDSA384Hex = "94d20456ce9ff913052b81040022030304bbd188b5018325fbdf1c491bf8e7ead419246bcfc1d0b88161bd3dd64039e5a00afbc56865d8309228d3e9f66567084e5e908c4dc33aa6d8ea914af2ebc117e905dae03288d4bd12050cfd48f5a1f89711c3150692a55b5ddac0149bc758e19dfe070302a0a6bba01d6bfb34e6aae45f44f142acef49e6b46837adcf00ecd9f6a5035c1f6ce41ef0f404e3c22bd0ecd14c4878e19ba8cee510c0b5372c04241ddfba45858ccc25f3ffe21c8c2d1cead8f883146957bde2984bc8b741bbf9a02f000e56"
	privKeyECDSA521Hex = "96000001080456cea09913052b8104002304230401e63fd4ef1398f0d57e42991c32f25e30bd5cb9dce3409d0de1c7a32d2e866bc3e557e6fc6f92013bb332f7df5b778c9c0a58a20fef0404ac2c488453336c9c8cad009e93771ee6b195a102b82171b07ccd06a2eb8ce8ce82c22200dbc513c16fb3aaf68809859efd19d64297ba19d28b83f038c566ca6a2a28bd3ee27275b909593504fe070302cc8e587938576e8be615d4d7e4c25c60c418a0a87e3e547d8cb674cdde722a192e69b20d444f1182613717c544db8f628f432a9016e0c5190b11d75a898ab3fa91cd74c1866a0553de7aa4997b8561c8b13b4c0b420d5c6328927ddb30b89c8b41daba767a158f26cd856a116cbcd5a2a0"

	privKeyECDH256Hex = "9caa0456d2275312082a8648ce3d03010702030489f2ec3b58370df5238e6ca30293c6957bade776a9ecbca117ee2f6296cb34de06b2d7327f9830adf4ff47029adc2e4da4b84be34fde7274e006847be25b40a103010807fe0703029095db180fb7b6ace7d1f3e8c9a5e3c52c66b796601149be8dbf24d76fa1cd197e890f3bf60ec10a767130497ad5011bb24bb4511a1949bcfb21f59f37c6cc73feb2e224da70d4f03f60fa750cddcbeb"
	privKeyECDH384Hex = "9cd70456d22d3912052b81040022030304142eb1dfabb2a7c8e2713726d95da8c6eaec50e3322126a862702460f3d449545d68376b05561453d197a8f5dfaf9abd22428302dbc8c832618e0556806abaf9963148cc3ca9ef1f2f408529eefe0940bfca9f740c8e1e1d91aab114d09c5d2403010909fe070302a41cdfb52649ad00e6feacaeb3ba043878eba47c39f6ffdd5c92ce9ffb62d07cd68b2a02c5113ea6f17a104bcc089dc77de99ce6dd1715b516a7747ad8ab19ad14823d3abb970051c45e149bc7342f7bf2cc02e14f12b84ac5b5988527f45575"
	privKeyECDH521Hex = "9e0000010c0456d22f9612052b810400230423040166e66949b465fa27e11344e77f55b61c40a583718da68cd8fec327ee3642d6f3437ee0e4bad0de0210e15f19dfbc1c9e5b2a7f82b85018f548dc639bf966343f3f00014669a4d53d2403886c0d1b7261cdaab8a6f8c213e081b444554603791449ae30639ed9da45d9e377a44052203b5146cc22cbbdda4c2fbd21f2db8ae4e0a72ac903010a09fe07030272ba571269ec5265e65bd45b17d556b665e0a88ee810fa9e3abeb32ddfc39279e12bc39432266bc3f711c3ffdc25f8a5f347ea1168b5cc2ffc2592038195704fd440280cb6a2c76c25ea1e85a0835b0b1d9de2e60a00a40e727ab246d8fcc5c9f195679397c7887a49f009bebc0cb9bff3"
)
