package encoding

import (
	"bytes"
	"io"
	"testing"
)

var mpiTests = []struct {
	encoded   []byte
	bytes     []byte
	reencoded []byte
	bitLength uint16
	err       error
}{
	{
		encoded:   []byte{0x0, 0x1, 0x1},
		bytes:     []byte{0x1},
		bitLength: 1,
	},
	{
		encoded:   []byte{0x0, 0x9, 0x1, 0xff},
		bytes:     []byte{0x1, 0xff},
		bitLength: 9,
	},
	{
		encoded:   append([]byte{0x1, 0x0, 0xff}, make([]byte, 0x1f)...),
		bytes:     append([]byte{0xff}, make([]byte, 0x1f)...),
		bitLength: 0x100,
	},
	// https://bugs.gnupg.org/gnupg/issue1853
	{
		encoded:   []byte{0x0, 0x10, 0x0, 0x01},
		bytes:     []byte{0x01},
		reencoded: []byte{0x0, 0x8, 0x01},
		bitLength: 8,
	},
	// EOF error,
	{
		encoded: []byte{},
		err:     io.ErrUnexpectedEOF,
	},
	{
		encoded: []byte{0x1, 0x0, 0x0},
		err:     io.ErrUnexpectedEOF,
	},
}

func TestMPI(t *testing.T) {
	for i, test := range mpiTests {
		mpi := new(MPI)
		if _, err := mpi.ReadFrom(bytes.NewBuffer(test.encoded)); err != nil {
			if !sameError(err, test.err) {
				t.Errorf("#%d: ReadFrom error got:%q", i, err)
			}
			continue
		}
		if b := mpi.Bytes(); !bytes.Equal(b, test.bytes) {
			t.Errorf("#%d: bad creation got:%x want:%x", i, b, test.bytes)
		}
		var buf bytes.Buffer
		if _, err := mpi.WriteTo(&buf); err != nil {
			t.Errorf("#%d: WriteTo error: %s", i, err)
		}

		reencoded := test.encoded
		if test.reencoded != nil {
			reencoded = test.reencoded
		}

		if b := buf.Bytes(); !bytes.Equal(b, reencoded) {
			t.Errorf("#%d: bad encoding got:%x want:%x", i, b, reencoded)
		}
		if bl := mpi.BitLength(); bl != test.bitLength {
			t.Errorf("#%d: bad BitLength got:%d want:%d", i, bl, test.bitLength)
		}
	}
}
