package pmp4

import (
	"bytes"
	"testing"

	"github.com/bluenviron/mediacommon/v2/pkg/codecs/mpeg4audio"
	"github.com/bluenviron/mediacommon/v2/pkg/formats/mp4"
	"github.com/stretchr/testify/require"
)

var casesPresentation = []struct {
	name string
	dec  Presentation
	enc  []byte
}{
	{
		"standard",
		Presentation{
			Tracks: []*Track{
				{
					ID:         1,
					TimeScale:  90000,
					TimeOffset: -90000,
					Codec: &mp4.CodecH264{
						SPS: []byte{ // 1920x1080 baseline
							0x67, 0x42, 0xc0, 0x28, 0xd9, 0x00, 0x78, 0x02,
							0x27, 0xe5, 0x84, 0x00, 0x00, 0x03, 0x00, 0x04,
							0x00, 0x00, 0x03, 0x00, 0xf0, 0x3c, 0x60, 0xc9, 0x20,
						},
						PPS: []byte{0x08, 0x06, 0x07, 0x08},
					},
					Samples: []*Sample{
						{
							Duration:    90000,
							PayloadSize: 2,
							GetPayload: func() ([]byte, error) {
								return []byte{1, 2}, nil
							},
						},
						{
							Duration:        90000,
							IsNonSyncSample: true,
							PTSOffset:       45000,
							PayloadSize:     2,
							GetPayload: func() ([]byte, error) {
								return []byte{3, 4}, nil
							},
						},
						{
							Duration:        90000,
							PTSOffset:       -45000,
							IsNonSyncSample: true,
							PayloadSize:     2,
							GetPayload: func() ([]byte, error) {
								return []byte{5, 6}, nil
							},
						},
					},
				},
				{
					ID:        2,
					TimeScale: 90000,
					Codec: &mp4.CodecH265{
						VPS: []byte{0x01, 0x02, 0x03, 0x04},
						SPS: []byte{
							0x42, 0x01, 0x01, 0x01, 0x60, 0x00, 0x00, 0x03,
							0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
							0x00, 0x78, 0xa0, 0x03, 0xc0, 0x80, 0x10, 0xe5,
							0x96, 0x66, 0x69, 0x24, 0xca, 0xe0, 0x10, 0x00,
							0x00, 0x03, 0x00, 0x10, 0x00, 0x00, 0x03, 0x01,
							0xe0, 0x80,
						},
						PPS: []byte{0x08},
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        3,
					TimeScale: 90000,
					Codec: &mp4.CodecVP9{
						Width:             1920,
						Height:            1080,
						Profile:           1,
						BitDepth:          8,
						ChromaSubsampling: 1,
						ColorRange:        false,
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        4,
					TimeScale: 90000,
					Codec: &mp4.CodecAV1{
						SequenceHeader: []byte{
							8, 0, 0, 0, 66, 167, 191, 228, 96, 13, 0, 64,
						},
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        5,
					TimeScale: 90000,
					Codec: &mp4.CodecMPEG4Video{
						Config: []byte{
							0x00, 0x00, 0x01, 0xb0, 0x01, 0x00, 0x00, 0x01,
							0xb5, 0x89, 0x13, 0x00, 0x00, 0x01, 0x00, 0x00,
							0x00, 0x01, 0x20, 0x00, 0xc4, 0x8d, 0x88, 0x00,
							0xf5, 0x3c, 0x04, 0x87, 0x14, 0x63, 0x00, 0x00,
							0x01, 0xb2, 0x4c, 0x61, 0x76, 0x63, 0x35, 0x38,
							0x2e, 0x31, 0x33, 0x34, 0x2e, 0x31, 0x30, 0x30,
						},
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        6,
					TimeScale: 90000,
					Codec: &mp4.CodecMPEG1Video{
						Config: []byte{
							0x00, 0x00, 0x01, 0xb3, 0x78, 0x04, 0x38, 0x35,
							0xff, 0xff, 0xe0, 0x18, 0x00, 0x00, 0x01, 0xb5,
							0x14, 0x4a, 0x00, 0x01, 0x00, 0x00,
						},
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        7,
					TimeScale: 90000,
					Codec: &mp4.CodecMJPEG{
						Width:  640,
						Height: 480,
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        7,
					TimeScale: 90000,
					Codec: &mp4.CodecOpus{
						ChannelCount: 2,
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        8,
					TimeScale: 90000,
					Codec: &mp4.CodecMPEG4Audio{
						Config: mpeg4audio.AudioSpecificConfig{
							Type:         2,
							SampleRate:   44100,
							ChannelCount: 2,
						},
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        9,
					TimeScale: 90000,
					Codec: &mp4.CodecMPEG1Audio{
						SampleRate:   48000,
						ChannelCount: 2,
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        10,
					TimeScale: 90000,
					Codec: &mp4.CodecAC3{
						SampleRate:   48000,
						ChannelCount: 6,
						Fscod:        0x0,
						Bsid:         0x8,
						Bsmod:        0x0,
						Acmod:        0x7,
						LfeOn:        true,
						BitRateCode:  0xf,
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
				{
					ID:        10,
					TimeScale: 90000,
					Codec: &mp4.CodecLPCM{
						BitDepth:     24,
						SampleRate:   48000,
						ChannelCount: 2,
					},
					Samples: []*Sample{{
						Duration:    90000,
						PayloadSize: 2,
						GetPayload: func() ([]byte, error) {
							return []byte{1, 2}, nil
						},
					}},
				},
			},
		},
		[]byte{
			0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
			0x69, 0x73, 0x6f, 0x6d, 0x00, 0x00, 0x00, 0x01,
			0x69, 0x73, 0x6f, 0x6d, 0x69, 0x73, 0x6f, 0x32,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x00, 0x00, 0x19, 0xdf, 0x6d, 0x6f, 0x6f, 0x76,
			0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x07, 0xd0, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x02, 0x6f,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x07, 0xd0, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x07, 0x80, 0x00, 0x00, 0x04, 0x38, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x24, 0x65, 0x64, 0x74, 0x73,
			0x00, 0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x0f, 0xa0, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0xe7,
			0x6d, 0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20,
			0x6d, 0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x5f, 0x90, 0x00, 0x02, 0xbf, 0x20,
			0x55, 0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d,
			0x68, 0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x76, 0x69, 0x64, 0x65,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x56, 0x69, 0x64, 0x65,
			0x6f, 0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72,
			0x00, 0x00, 0x00, 0x01, 0x92, 0x6d, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x14, 0x76, 0x6d, 0x68,
			0x64, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x01, 0x52, 0x73, 0x74, 0x62,
			0x6c, 0x00, 0x00, 0x00, 0x96, 0x73, 0x74, 0x73,
			0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x86, 0x61, 0x76, 0x63,
			0x31, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x07, 0x80, 0x04, 0x38, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x18, 0xff, 0xff, 0x00,
			0x00, 0x00, 0x30, 0x61, 0x76, 0x63, 0x43, 0x01,
			0x42, 0xc0, 0x28, 0x03, 0x01, 0x00, 0x19, 0x67,
			0x42, 0xc0, 0x28, 0xd9, 0x00, 0x78, 0x02, 0x27,
			0xe5, 0x84, 0x00, 0x00, 0x03, 0x00, 0x04, 0x00,
			0x00, 0x03, 0x00, 0xf0, 0x3c, 0x60, 0xc9, 0x20,
			0x01, 0x00, 0x04, 0x08, 0x06, 0x07, 0x08, 0x00,
			0x00, 0x00, 0x18, 0x73, 0x74, 0x74, 0x73, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x03, 0x00, 0x01, 0x5f, 0x90, 0x00,
			0x00, 0x00, 0x14, 0x73, 0x74, 0x73, 0x73, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x28, 0x63,
			0x74, 0x74, 0x73, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0xaf, 0xc8, 0x00, 0x00, 0x00, 0x01, 0xff,
			0xff, 0x50, 0x38, 0x00, 0x00, 0x00, 0x28, 0x73,
			0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x20, 0x73,
			0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x73,
			0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x00, 0x1a, 0x07, 0x00,
			0x00, 0x1a, 0x21, 0x00, 0x00, 0x02, 0x60, 0x74,
			0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c, 0x74,
			0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x07,
			0x80, 0x00, 0x00, 0x04, 0x38, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x24, 0x65, 0x64, 0x74, 0x73, 0x00,
			0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0xd8, 0x6d,
			0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20, 0x6d,
			0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90, 0x55,
			0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x68,
			0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x76, 0x69, 0x64, 0x65, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x56, 0x69, 0x64, 0x65, 0x6f,
			0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00,
			0x00, 0x00, 0x01, 0x83, 0x6d, 0x69, 0x6e, 0x66,
			0x00, 0x00, 0x00, 0x14, 0x76, 0x6d, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24,
			0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00, 0x1c,
			0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c,
			0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x01, 0x43, 0x73, 0x74, 0x62, 0x6c,
			0x00, 0x00, 0x00, 0xc3, 0x73, 0x74, 0x73, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0xb3, 0x68, 0x65, 0x76, 0x31,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x07, 0x80, 0x04, 0x38, 0x00, 0x48, 0x00, 0x00,
			0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x18, 0xff, 0xff, 0x00, 0x00,
			0x00, 0x5d, 0x68, 0x76, 0x63, 0x43, 0x01, 0x01,
			0x60, 0x00, 0x00, 0x00, 0x03, 0x00, 0x90, 0x00,
			0x00, 0x03, 0x78, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x13, 0x03, 0x20, 0x00, 0x01,
			0x00, 0x04, 0x01, 0x02, 0x03, 0x04, 0x21, 0x00,
			0x01, 0x00, 0x2a, 0x42, 0x01, 0x01, 0x01, 0x60,
			0x00, 0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x03, 0x00, 0x78, 0xa0, 0x03, 0xc0,
			0x80, 0x10, 0xe5, 0x96, 0x66, 0x69, 0x24, 0xca,
			0xe0, 0x10, 0x00, 0x00, 0x03, 0x00, 0x10, 0x00,
			0x00, 0x03, 0x01, 0xe0, 0x80, 0x22, 0x00, 0x01,
			0x00, 0x01, 0x08, 0x00, 0x00, 0x00, 0x18, 0x73,
			0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x5f, 0x90, 0x00, 0x00, 0x00, 0x18, 0x63,
			0x74, 0x74, 0x73, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x73,
			0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x18, 0x73, 0x74, 0x73, 0x7a, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
			0x00, 0x00, 0x14, 0x73, 0x74, 0x63, 0x6f, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x1a, 0x0b, 0x00, 0x00, 0x02, 0x17, 0x74,
			0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c, 0x74,
			0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x07,
			0x80, 0x00, 0x00, 0x04, 0x38, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x24, 0x65, 0x64, 0x74, 0x73, 0x00,
			0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x8f, 0x6d,
			0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20, 0x6d,
			0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90, 0x55,
			0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x68,
			0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x76, 0x69, 0x64, 0x65, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x56, 0x69, 0x64, 0x65, 0x6f,
			0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00,
			0x00, 0x00, 0x01, 0x3a, 0x6d, 0x69, 0x6e, 0x66,
			0x00, 0x00, 0x00, 0x14, 0x76, 0x6d, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24,
			0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00, 0x1c,
			0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c,
			0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0xfa, 0x73, 0x74, 0x62, 0x6c,
			0x00, 0x00, 0x00, 0x7a, 0x73, 0x74, 0x73, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x6a, 0x76, 0x70, 0x30, 0x39,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x07, 0x80, 0x04, 0x38, 0x00, 0x48, 0x00, 0x00,
			0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x18, 0xff, 0xff, 0x00, 0x00,
			0x00, 0x14, 0x76, 0x70, 0x63, 0x43, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x0a, 0x82, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x73, 0x74,
			0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01,
			0x5f, 0x90, 0x00, 0x00, 0x00, 0x18, 0x63, 0x74,
			0x74, 0x73, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x73, 0x74,
			0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x18, 0x73, 0x74, 0x73, 0x7a, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
			0x00, 0x14, 0x73, 0x74, 0x63, 0x6f, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x1a, 0x0d, 0x00, 0x00, 0x02, 0x1c, 0x74, 0x72,
			0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c, 0x74, 0x6b,
			0x68, 0x64, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x07, 0x80,
			0x00, 0x00, 0x03, 0x24, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x24, 0x65, 0x64, 0x74, 0x73, 0x00, 0x00,
			0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x94, 0x6d, 0x64,
			0x69, 0x61, 0x00, 0x00, 0x00, 0x20, 0x6d, 0x64,
			0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90, 0x55, 0xc4,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x68, 0x64,
			0x6c, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x76, 0x69, 0x64, 0x65, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x56, 0x69, 0x64, 0x65, 0x6f, 0x48,
			0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00, 0x00,
			0x00, 0x01, 0x3f, 0x6d, 0x69, 0x6e, 0x66, 0x00,
			0x00, 0x00, 0x14, 0x76, 0x6d, 0x68, 0x64, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x64,
			0x69, 0x6e, 0x66, 0x00, 0x00, 0x00, 0x1c, 0x64,
			0x72, 0x65, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x75,
			0x72, 0x6c, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0xff, 0x73, 0x74, 0x62, 0x6c, 0x00,
			0x00, 0x00, 0x7f, 0x73, 0x74, 0x73, 0x64, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x6f, 0x61, 0x76, 0x30, 0x31, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07,
			0x80, 0x03, 0x24, 0x00, 0x48, 0x00, 0x00, 0x00,
			0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x18, 0xff, 0xff, 0x00, 0x00, 0x00,
			0x19, 0x61, 0x76, 0x31, 0x43, 0x81, 0x08, 0x0c,
			0x00, 0x0a, 0x0b, 0x00, 0x00, 0x00, 0x42, 0xa7,
			0xbf, 0xe4, 0x60, 0x0d, 0x00, 0x40, 0x00, 0x00,
			0x00, 0x18, 0x73, 0x74, 0x74, 0x73, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x01, 0x5f, 0x90, 0x00, 0x00,
			0x00, 0x18, 0x63, 0x74, 0x74, 0x73, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x1c, 0x73, 0x74, 0x73, 0x63, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x18, 0x73, 0x74,
			0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x02, 0x00, 0x00, 0x00, 0x14, 0x73, 0x74,
			0x63, 0x6f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x1a, 0x0f, 0x00, 0x00,
			0x02, 0x64, 0x74, 0x72, 0x61, 0x6b, 0x00, 0x00,
			0x00, 0x5c, 0x74, 0x6b, 0x68, 0x64, 0x00, 0x00,
			0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x03, 0xe8, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00,
			0x00, 0x00, 0x03, 0x20, 0x00, 0x00, 0x02, 0x58,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x65, 0x64,
			0x74, 0x73, 0x00, 0x00, 0x00, 0x1c, 0x65, 0x6c,
			0x73, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x03, 0xe8, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x01, 0xdc, 0x6d, 0x64, 0x69, 0x61, 0x00, 0x00,
			0x00, 0x20, 0x6d, 0x64, 0x68, 0x64, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x5f, 0x90, 0x00, 0x01,
			0x5f, 0x90, 0x55, 0xc4, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0x69,
			0x64, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x56, 0x69,
			0x64, 0x65, 0x6f, 0x48, 0x61, 0x6e, 0x64, 0x6c,
			0x65, 0x72, 0x00, 0x00, 0x00, 0x01, 0x87, 0x6d,
			0x69, 0x6e, 0x66, 0x00, 0x00, 0x00, 0x14, 0x76,
			0x6d, 0x68, 0x64, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x24, 0x64, 0x69, 0x6e, 0x66, 0x00,
			0x00, 0x00, 0x1c, 0x64, 0x72, 0x65, 0x66, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x47, 0x73,
			0x74, 0x62, 0x6c, 0x00, 0x00, 0x00, 0xc7, 0x73,
			0x74, 0x73, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xb7, 0x6d,
			0x70, 0x34, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x03, 0x20, 0x02, 0x58, 0x00,
			0x48, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xff,
			0xff, 0x00, 0x00, 0x00, 0x61, 0x65, 0x73, 0x64,
			0x73, 0x00, 0x00, 0x00, 0x00, 0x03, 0x80, 0x80,
			0x80, 0x50, 0x00, 0x05, 0x00, 0x04, 0x80, 0x80,
			0x80, 0x42, 0x20, 0x11, 0x00, 0x00, 0x00, 0x00,
			0x0f, 0x42, 0x40, 0x00, 0x0f, 0x42, 0x40, 0x05,
			0x80, 0x80, 0x80, 0x30, 0x00, 0x00, 0x01, 0xb0,
			0x01, 0x00, 0x00, 0x01, 0xb5, 0x89, 0x13, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x20, 0x00,
			0xc4, 0x8d, 0x88, 0x00, 0xf5, 0x3c, 0x04, 0x87,
			0x14, 0x63, 0x00, 0x00, 0x01, 0xb2, 0x4c, 0x61,
			0x76, 0x63, 0x35, 0x38, 0x2e, 0x31, 0x33, 0x34,
			0x2e, 0x31, 0x30, 0x30, 0x06, 0x80, 0x80, 0x80,
			0x01, 0x02, 0x00, 0x00, 0x00, 0x18, 0x73, 0x74,
			0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01,
			0x5f, 0x90, 0x00, 0x00, 0x00, 0x18, 0x63, 0x74,
			0x74, 0x73, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x73, 0x74,
			0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x18, 0x73, 0x74, 0x73, 0x7a, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
			0x00, 0x14, 0x73, 0x74, 0x63, 0x6f, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x1a, 0x11, 0x00, 0x00, 0x02, 0x4a, 0x74, 0x72,
			0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c, 0x74, 0x6b,
			0x68, 0x64, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03, 0x20,
			0x00, 0x00, 0x02, 0x58, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x24, 0x65, 0x64, 0x74, 0x73, 0x00, 0x00,
			0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x01, 0xc2, 0x6d, 0x64,
			0x69, 0x61, 0x00, 0x00, 0x00, 0x20, 0x6d, 0x64,
			0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90, 0x55, 0xc4,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x68, 0x64,
			0x6c, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x76, 0x69, 0x64, 0x65, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x56, 0x69, 0x64, 0x65, 0x6f, 0x48,
			0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00, 0x00,
			0x00, 0x01, 0x6d, 0x6d, 0x69, 0x6e, 0x66, 0x00,
			0x00, 0x00, 0x14, 0x76, 0x6d, 0x68, 0x64, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x64,
			0x69, 0x6e, 0x66, 0x00, 0x00, 0x00, 0x1c, 0x64,
			0x72, 0x65, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x75,
			0x72, 0x6c, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x01, 0x2d, 0x73, 0x74, 0x62, 0x6c, 0x00,
			0x00, 0x00, 0xad, 0x73, 0x74, 0x73, 0x64, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x9d, 0x6d, 0x70, 0x34, 0x76, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
			0x20, 0x02, 0x58, 0x00, 0x48, 0x00, 0x00, 0x00,
			0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x18, 0xff, 0xff, 0x00, 0x00, 0x00,
			0x47, 0x65, 0x73, 0x64, 0x73, 0x00, 0x00, 0x00,
			0x00, 0x03, 0x80, 0x80, 0x80, 0x36, 0x00, 0x06,
			0x00, 0x04, 0x80, 0x80, 0x80, 0x28, 0x61, 0x11,
			0x00, 0x00, 0x00, 0x00, 0x0f, 0x42, 0x40, 0x00,
			0x0f, 0x42, 0x40, 0x05, 0x80, 0x80, 0x80, 0x16,
			0x00, 0x00, 0x01, 0xb3, 0x78, 0x04, 0x38, 0x35,
			0xff, 0xff, 0xe0, 0x18, 0x00, 0x00, 0x01, 0xb5,
			0x14, 0x4a, 0x00, 0x01, 0x00, 0x00, 0x06, 0x80,
			0x80, 0x80, 0x01, 0x02, 0x00, 0x00, 0x00, 0x18,
			0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x01, 0x5f, 0x90, 0x00, 0x00, 0x00, 0x18,
			0x63, 0x74, 0x74, 0x73, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c,
			0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x18, 0x73, 0x74, 0x73, 0x7a,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02,
			0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x63, 0x6f,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x1a, 0x13, 0x00, 0x00, 0x02, 0x2f,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x02, 0x80, 0x00, 0x00, 0x01, 0xe0, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x24, 0x65, 0x64, 0x74, 0x73,
			0x00, 0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0xa7,
			0x6d, 0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20,
			0x6d, 0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90,
			0x55, 0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d,
			0x68, 0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x76, 0x69, 0x64, 0x65,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x56, 0x69, 0x64, 0x65,
			0x6f, 0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72,
			0x00, 0x00, 0x00, 0x01, 0x52, 0x6d, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x14, 0x76, 0x6d, 0x68,
			0x64, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x01, 0x12, 0x73, 0x74, 0x62,
			0x6c, 0x00, 0x00, 0x00, 0x92, 0x73, 0x74, 0x73,
			0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x82, 0x6d, 0x70, 0x34,
			0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x02, 0x80, 0x01, 0xe0, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x18, 0xff, 0xff, 0x00,
			0x00, 0x00, 0x2c, 0x65, 0x73, 0x64, 0x73, 0x00,
			0x00, 0x00, 0x00, 0x03, 0x80, 0x80, 0x80, 0x1b,
			0x00, 0x07, 0x00, 0x04, 0x80, 0x80, 0x80, 0x0d,
			0x6c, 0x11, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x42,
			0x40, 0x00, 0x0f, 0x42, 0x40, 0x06, 0x80, 0x80,
			0x80, 0x01, 0x02, 0x00, 0x00, 0x00, 0x18, 0x73,
			0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x5f, 0x90, 0x00, 0x00, 0x00, 0x18, 0x63,
			0x74, 0x74, 0x73, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x73,
			0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x18, 0x73, 0x74, 0x73, 0x7a, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
			0x00, 0x00, 0x14, 0x73, 0x74, 0x63, 0x6f, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x1a, 0x15, 0x00, 0x00, 0x01, 0xe0, 0x74,
			0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c, 0x74,
			0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x24, 0x65, 0x64, 0x74, 0x73, 0x00,
			0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x58, 0x6d,
			0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20, 0x6d,
			0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90, 0x55,
			0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x68,
			0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x73, 0x6f, 0x75, 0x6e, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x53, 0x6f, 0x75, 0x6e, 0x64,
			0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00,
			0x00, 0x00, 0x01, 0x03, 0x6d, 0x69, 0x6e, 0x66,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x6d, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e, 0x66,
			0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65, 0x66,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xc7,
			0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00, 0x47,
			0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x37,
			0x4f, 0x70, 0x75, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x10,
			0x00, 0x00, 0x00, 0x00, 0xbb, 0x80, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x13, 0x64, 0x4f, 0x70, 0x73,
			0x00, 0x02, 0x01, 0x38, 0x00, 0x00, 0xbb, 0x80,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x73,
			0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x5f, 0x90, 0x00, 0x00, 0x00, 0x18, 0x63,
			0x74, 0x74, 0x73, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x73,
			0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x18, 0x73, 0x74, 0x73, 0x7a, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
			0x00, 0x00, 0x14, 0x73, 0x74, 0x63, 0x6f, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x1a, 0x17, 0x00, 0x00, 0x02, 0x00, 0x74,
			0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c, 0x74,
			0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x24, 0x65, 0x64, 0x74, 0x73, 0x00,
			0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x78, 0x6d,
			0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20, 0x6d,
			0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90, 0x55,
			0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x68,
			0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x73, 0x6f, 0x75, 0x6e, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x53, 0x6f, 0x75, 0x6e, 0x64,
			0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00,
			0x00, 0x00, 0x01, 0x23, 0x6d, 0x69, 0x6e, 0x66,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x6d, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e, 0x66,
			0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65, 0x66,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xe7,
			0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00, 0x67,
			0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x57,
			0x6d, 0x70, 0x34, 0x61, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x10,
			0x00, 0x00, 0x00, 0x00, 0xac, 0x44, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x33, 0x65, 0x73, 0x64, 0x73,
			0x00, 0x00, 0x00, 0x00, 0x03, 0x80, 0x80, 0x80,
			0x22, 0x00, 0x08, 0x00, 0x04, 0x80, 0x80, 0x80,
			0x14, 0x40, 0x15, 0x00, 0x00, 0x00, 0x00, 0x01,
			0xf7, 0x39, 0x00, 0x01, 0xf7, 0x39, 0x05, 0x80,
			0x80, 0x80, 0x02, 0x12, 0x10, 0x06, 0x80, 0x80,
			0x80, 0x01, 0x02, 0x00, 0x00, 0x00, 0x18, 0x73,
			0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x5f, 0x90, 0x00, 0x00, 0x00, 0x18, 0x63,
			0x74, 0x74, 0x73, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x73,
			0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x18, 0x73, 0x74, 0x73, 0x7a, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
			0x00, 0x00, 0x14, 0x73, 0x74, 0x63, 0x6f, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x1a, 0x19, 0x00, 0x00, 0x01, 0xf9, 0x74,
			0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c, 0x74,
			0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x24, 0x65, 0x64, 0x74, 0x73, 0x00,
			0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x71, 0x6d,
			0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20, 0x6d,
			0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90, 0x55,
			0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x68,
			0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x73, 0x6f, 0x75, 0x6e, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x53, 0x6f, 0x75, 0x6e, 0x64,
			0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00,
			0x00, 0x00, 0x01, 0x1c, 0x6d, 0x69, 0x6e, 0x66,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x6d, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e, 0x66,
			0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65, 0x66,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xe0,
			0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00, 0x60,
			0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x50,
			0x6d, 0x70, 0x34, 0x61, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x10,
			0x00, 0x00, 0x00, 0x00, 0xbb, 0x80, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2c, 0x65, 0x73, 0x64, 0x73,
			0x00, 0x00, 0x00, 0x00, 0x03, 0x80, 0x80, 0x80,
			0x1b, 0x00, 0x09, 0x00, 0x04, 0x80, 0x80, 0x80,
			0x0d, 0x6b, 0x15, 0x00, 0x00, 0x00, 0x00, 0x01,
			0xf7, 0x39, 0x00, 0x01, 0xf7, 0x39, 0x06, 0x80,
			0x80, 0x80, 0x01, 0x02, 0x00, 0x00, 0x00, 0x18,
			0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x01, 0x5f, 0x90, 0x00, 0x00, 0x00, 0x18,
			0x63, 0x74, 0x74, 0x73, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c,
			0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x18, 0x73, 0x74, 0x73, 0x7a,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02,
			0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x63, 0x6f,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x1a, 0x1b, 0x00, 0x00, 0x01, 0xd8,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x24, 0x65, 0x64, 0x74, 0x73,
			0x00, 0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x50,
			0x6d, 0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20,
			0x6d, 0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90,
			0x55, 0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d,
			0x68, 0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x73, 0x6f, 0x75, 0x6e,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x53, 0x6f, 0x75, 0x6e,
			0x64, 0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72,
			0x00, 0x00, 0x00, 0x00, 0xfb, 0x6d, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x10, 0x73, 0x6d, 0x68,
			0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xbf, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0x3f, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x2f, 0x61, 0x63, 0x2d, 0x33, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00,
			0x10, 0x00, 0x00, 0x00, 0x00, 0xbb, 0x80, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x0b, 0x64, 0x61, 0x63,
			0x33, 0x10, 0x3d, 0xe0, 0x00, 0x00, 0x00, 0x18,
			0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x01, 0x5f, 0x90, 0x00, 0x00, 0x00, 0x18,
			0x63, 0x74, 0x74, 0x73, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c,
			0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x18, 0x73, 0x74, 0x73, 0x7a,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02,
			0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x63, 0x6f,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x1a, 0x1d, 0x00, 0x00, 0x01, 0xdb,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x24, 0x65, 0x64, 0x74, 0x73,
			0x00, 0x00, 0x00, 0x1c, 0x65, 0x6c, 0x73, 0x74,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x53,
			0x6d, 0x64, 0x69, 0x61, 0x00, 0x00, 0x00, 0x20,
			0x6d, 0x64, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x5f, 0x90, 0x00, 0x01, 0x5f, 0x90,
			0x55, 0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d,
			0x68, 0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x73, 0x6f, 0x75, 0x6e,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x53, 0x6f, 0x75, 0x6e,
			0x64, 0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72,
			0x00, 0x00, 0x00, 0x00, 0xfe, 0x6d, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x10, 0x73, 0x6d, 0x68,
			0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xc2, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0x42, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x32, 0x69, 0x70, 0x63, 0x6d, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
			0x18, 0x00, 0x00, 0x00, 0x00, 0xbb, 0x80, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x0e, 0x70, 0x63, 0x6d,
			0x43, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00,
			0x00, 0x00, 0x18, 0x73, 0x74, 0x74, 0x73, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x01, 0x5f, 0x90, 0x00,
			0x00, 0x00, 0x18, 0x63, 0x74, 0x74, 0x73, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x1c, 0x73, 0x74, 0x73, 0x63, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x18, 0x73,
			0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x14, 0x73,
			0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x1a, 0x1f, 0x00,
			0x00, 0x00, 0x24, 0x6d, 0x64, 0x61, 0x74, 0x01,
			0x02, 0x03, 0x04, 0x01, 0x02, 0x01, 0x02, 0x01,
			0x02, 0x01, 0x02, 0x01, 0x02, 0x01, 0x02, 0x01,
			0x02, 0x01, 0x02, 0x01, 0x02, 0x01, 0x02, 0x01,
			0x02, 0x05, 0x06,
		},
	},
}

func getSampleData(t *testing.T, p *Presentation) map[int][][]byte {
	sampleData := make(map[int][][]byte)
	for _, track := range p.Tracks {
		var trackData [][]byte
		for _, sample := range track.Samples {
			pl, err := sample.GetPayload()
			require.NoError(t, err)
			trackData = append(trackData, pl)
		}
		sampleData[track.ID] = trackData
	}
	return sampleData
}

func removeGetPayloads(p *Presentation) []func() ([]byte, error) {
	var ret []func() ([]byte, error)
	for _, track := range p.Tracks {
		for _, sample := range track.Samples {
			ret = append(ret, sample.GetPayload)
			sample.GetPayload = nil
		}
	}
	return ret
}

func restoreGetPayloads(p *Presentation, getPayloads []func() ([]byte, error)) {
	i := 0
	for _, track := range p.Tracks {
		for _, sample := range track.Samples {
			sample.GetPayload = getPayloads[i]
			i++
		}
	}
}

func TestPresentationUnmarshal(t *testing.T) {
	for _, ca := range casesPresentation {
		t.Run(ca.name, func(t *testing.T) {
			var p Presentation
			err := p.Unmarshal(bytes.NewReader(ca.enc))
			require.NoError(t, err)

			expectedSampleData := getSampleData(t, &ca.dec)
			sampleData := getSampleData(t, &p)

			removeGetPayloads(&p)
			getPayloads := removeGetPayloads(&ca.dec)

			require.Equal(t, ca.dec, p)
			require.Equal(t, expectedSampleData, sampleData)

			restoreGetPayloads(&ca.dec, getPayloads)
		})
	}
}

func TestPresentationMarshal(t *testing.T) {
	for _, ca := range casesPresentation {
		t.Run(ca.name, func(t *testing.T) {
			var buf bytes.Buffer
			err := ca.dec.Marshal(&buf)
			require.NoError(t, err)
			require.Equal(t, ca.enc, buf.Bytes())
		})
	}
}

func FuzzPresentationUnmarshal(f *testing.F) {
	for _, ca := range casesPresentation {
		f.Add(ca.enc)
	}

	f.Fuzz(func(t *testing.T, b []byte) {
		var p Presentation
		err := p.Unmarshal(bytes.NewReader(b))
		if err != nil {
			return
		}

		require.NotZero(t, len(p.Tracks))

		for _, track := range p.Tracks {
			require.NotZero(t, track.TimeScale)

			for _, sample := range track.Samples {
				_, err = sample.GetPayload()
				if err != nil {
					return
				}
			}
		}

		var buf bytes.Buffer
		err = p.Marshal(&buf)
		require.NoError(t, err)
	})
}
