//nolint:dupl
package mpegts

import (
	"bytes"
	"context"
	"errors"
	"io"
	"testing"

	"github.com/asticode/go-astits"
	"github.com/stretchr/testify/require"

	"github.com/bluenviron/mediacommon/v2/pkg/codecs/h265"
	"github.com/bluenviron/mediacommon/v2/pkg/codecs/mpeg4audio"
)

var testH265SPS = []byte{
	0x42, 0x01, 0x01, 0x02, 0x20, 0x00, 0x00, 0x03,
	0x00, 0xb0, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
	0x00, 0x7b, 0xa0, 0x07, 0x82, 0x00, 0x88, 0x7d,
	0xb6, 0x71, 0x8b, 0x92, 0x44, 0x80, 0x53, 0x88,
	0x88, 0x92, 0xcf, 0x24, 0xa6, 0x92, 0x72, 0xc9,
	0x12, 0x49, 0x22, 0xdc, 0x91, 0xaa, 0x48, 0xfc,
	0xa2, 0x23, 0xff, 0x00, 0x01, 0x00, 0x01, 0x6a,
	0x02, 0x02, 0x02, 0x01,
}

var testH265PPS = []byte{
	0x44, 0x01, 0xc0, 0x25, 0x2f, 0x05, 0x32, 0x40,
}

var testH264SPS = []byte{
	0x67, 0x42, 0xc0, 0x28, 0xd9, 0x00, 0x78, 0x02,
	0x27, 0xe5, 0x84, 0x00, 0x00, 0x03, 0x00, 0x04,
	0x00, 0x00, 0x03, 0x00, 0xf0, 0x3c, 0x60, 0xc9,
	0x20,
}

type sample struct {
	pts  int64
	dts  int64
	data [][]byte
}

var casesReadWriter = []struct {
	name    string
	track   *Track
	samples []sample
	packets []*astits.Packet
}{
	{
		"h265",
		&Track{
			PID:   257,
			Codec: &CodecH265{},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{
					testH265SPS, // SPS
					testH265PPS, // PPS
					{byte(h265.NALUType_CRA_NUT) << 1},
				},
			},
			{
				30*90000 + 2*90000,
				30*90000 + 1*90000,
				[][]byte{
					{byte(h265.NALUType_TRAIL_N) << 1},
				},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x12, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x24, 0xe1, 0x01,
					0xf0, 0x00, 0x75, 0x79, 0x1e, 0xaa,
				}, bytes.Repeat([]byte{0xff}, 162)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                81,
					StuffingLength:        74,
					RandomAccessIndicator: true,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0x00, 0x00,
					0x00, 0x01, 0x46, 0x01, 0x50, 0x00, 0x00, 0x00,
					0x01, 0x42, 0x01, 0x01, 0x02, 0x20, 0x00, 0x00,
					0x03, 0x00, 0xb0, 0x00, 0x00, 0x03, 0x00, 0x00,
					0x03, 0x00, 0x7b, 0xa0, 0x07, 0x82, 0x00, 0x88,
					0x7d, 0xb6, 0x71, 0x8b, 0x92, 0x44, 0x80, 0x53,
					0x88, 0x88, 0x92, 0xcf, 0x24, 0xa6, 0x92, 0x72,
					0xc9, 0x12, 0x49, 0x22, 0xdc, 0x91, 0xaa, 0x48,
					0xfc, 0xa2, 0x23, 0xff, 0x00, 0x01, 0x00, 0x01,
					0x6a, 0x02, 0x02, 0x02, 0x01, 0x00, 0x00, 0x00,
					0x01, 0x44, 0x01, 0xc0, 0x25, 0x2f, 0x05, 0x32,
					0x40, 0x00, 0x00, 0x00, 0x01, 0x2a,
				},
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:         152,
					StuffingLength: 151,
				},
				Header: astits.PacketHeader{
					ContinuityCounter:         1,
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x80, 0xc0,
					0x0a, 0x31, 0x00, 0xaf, 0xe4, 0x01, 0x11, 0x00,
					0xab, 0x24, 0xe1, 0x00, 0x00, 0x00, 0x01, 0x46,
					0x01, 0x50, 0x00, 0x00, 0x00, 0x01, 0x00,
				},
			},
		},
	},
	{
		"h264",
		&Track{
			PID:   256,
			Codec: &CodecH264{},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{
					testH264SPS, // SPS
					{8},         // PPS
					{5},         // IDR
				},
			},
			{
				30*90000 + 2*90000,
				30*90000 + 1*90000,
				[][]byte{
					{1}, // non-IDR
				},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x12, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x00, 0xf0, 0x00, 0x1b, 0xe1, 0x00,
					0xf0, 0x00, 0x15, 0xbd, 0x4d, 0x56,
				}, bytes.Repeat([]byte{0xff}, 162)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                124,
					StuffingLength:        117,
					RandomAccessIndicator: true,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       256,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0x00, 0x00,
					0x00, 0x01, 0x09, 0xf0, 0x00, 0x00, 0x00, 0x01,
					0x67, 0x42, 0xc0, 0x28, 0xd9, 0x00, 0x78, 0x02,
					0x27, 0xe5, 0x84, 0x00, 0x00, 0x03, 0x00, 0x04,
					0x00, 0x00, 0x03, 0x00, 0xf0, 0x3c, 0x60, 0xc9,
					0x20, 0x00, 0x00, 0x00, 0x01, 0x08, 0x00, 0x00,
					0x00, 0x01, 0x05,
				},
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:         153,
					StuffingLength: 152,
				},
				Header: astits.PacketHeader{
					ContinuityCounter:         1,
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       256,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x80, 0xc0,
					0x0a, 0x31, 0x00, 0xaf, 0xe4, 0x01, 0x11, 0x00,
					0xab, 0x24, 0xe1, 0x00, 0x00, 0x00, 0x01, 0x09,
					0xf0, 0x00, 0x00, 0x00, 0x01, 0x01,
				},
			},
		},
	},
	{
		"mpeg-4 video",
		&Track{
			PID:   257,
			Codec: &CodecMPEG4Video{},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{{0, 0, 1, 0xb3}},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x12, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x10, 0xe1, 0x01,
					0xf0, 0x00, 0xd5, 0x3a, 0x92, 0x8a,
				}, bytes.Repeat([]byte{0xff}, 162)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                165,
					StuffingLength:        158,
					RandomAccessIndicator: true,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0x00, 0x00,
					0x01, 0xb3,
				},
			},
		},
	},
	{
		"mpeg-1 video",
		&Track{
			PID:   257,
			Codec: &CodecMPEG1Video{},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{{0, 0, 1, 0xb8, 1, 2, 3, 4}},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x12, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x02, 0xe1, 0x01,
					0xf0, 0x00, 0xc4, 0xf2, 0x53, 0x9c,
				}, bytes.Repeat([]byte{0xff}, 162)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                161,
					StuffingLength:        154,
					RandomAccessIndicator: true,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0x00, 0x00,
					0x01, 0xb8, 0x01, 0x02, 0x03, 0x04,
				},
			},
		},
	},
	{
		"opus",
		&Track{
			PID: 257,
			Codec: &CodecOpus{
				ChannelCount: 2,
			},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{{3}, {2}},
			},
			{
				30*90000 + 2*90000,
				30 * 90000,
				[][]byte{{1}},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x1c, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x06, 0xe1, 0x01,
					0xf0, 0x0a, 0x05, 0x04, 0x4f, 0x70, 0x75, 0x73,
					0x7f, 0x02, 0x80, 0x02, 0xcc, 0x21, 0x3d, 0x58,
				}, bytes.Repeat([]byte{0xff}, 152)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                161,
					StuffingLength:        154,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
					RandomAccessIndicator: true,
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xc0, 0x00, 0x10, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0x7f, 0xe0,
					0x01, 0x03, 0x7f, 0xe0, 0x01, 0x02,
				},
			},
			{ // PMT
				Header: astits.PacketHeader{
					ContinuityCounter:         1,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					ContinuityCounter:         1,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x1c, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x06, 0xe1, 0x01,
					0xf0, 0x0a, 0x05, 0x04, 0x4f, 0x70, 0x75, 0x73,
					0x7f, 0x02, 0x80, 0x02, 0xcc, 0x21, 0x3d, 0x58,
				}, bytes.Repeat([]byte{0xff}, 152)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                165,
					StuffingLength:        164,
					RandomAccessIndicator: true,
				},
				Header: astits.PacketHeader{
					ContinuityCounter:         1,
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xc0, 0x00, 0x0c, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xaf, 0xe4, 0x01, 0x7f, 0xe0,
					0x01, 0x01,
				},
			},
		},
	},
	{
		"mpeg-4 audio",
		&Track{
			PID: 257,
			Codec: &CodecMPEG4Audio{
				Config: mpeg4audio.AudioSpecificConfig{
					Type:         2,
					SampleRate:   48000,
					ChannelCount: 2,
				},
			},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{{3}, {2}},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x12, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x0f, 0xe1, 0x01,
					0xf0, 0x00, 0xec, 0xe2, 0xb0, 0x94,
				}, bytes.Repeat([]byte{0xff}, 162)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                153,
					StuffingLength:        146,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
					RandomAccessIndicator: true,
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xc0, 0x00, 0x18, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0xff, 0xf1,
					0x4c, 0x80, 0x01, 0x1f, 0xfc, 0x03, 0xff, 0xf1,
					0x4c, 0x80, 0x01, 0x1f, 0xfc, 0x02,
				},
			},
		},
	},
	{
		"mpeg-4 audio LATM",
		&Track{
			PID:   257,
			Codec: &CodecMPEG4AudioLATM{},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{{3}, {2}},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x12, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x11, 0xe1, 0x01,
					0xf0, 0x00, 0x9c, 0x37, 0xf5, 0x07,
				}, bytes.Repeat([]byte{0xff}, 162)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                161,
					StuffingLength:        154,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
					RandomAccessIndicator: true,
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xc0, 0x00, 0x10, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0x56, 0xe0,
					0x01, 0x03, 0x56, 0xe0, 0x01, 0x02,
				},
			},
		},
	},
	{
		"mpeg-1 audio",
		&Track{
			PID:   257,
			Codec: &CodecMPEG1Audio{},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{{
					0xff, 0xfa, 0x52, 0x04, 0xa9, 0xbe, 0xe4, 0x8f,
					0xf0, 0xfd, 0x02, 0xdc, 0x80, 0x00, 0x30, 0x00,
					0x22, 0xc1, 0x5b, 0x90, 0x14, 0x23, 0x24, 0x05,
					0x58, 0x3f, 0x72, 0x02, 0x84, 0xc4, 0xc0, 0xc5,
					0x07, 0xae, 0x40, 0x21, 0xbc, 0x98, 0x90, 0xfa,
					0x3a, 0x2d, 0xda, 0x07, 0xe1, 0x4d, 0xa9, 0x9a,
					0xb8, 0xa2, 0x3b, 0x20, 0xc1, 0xc1, 0xba, 0x08,
					0x94, 0x30, 0x8b, 0xc5, 0x69, 0x51, 0x95, 0xd5,
					0xd7, 0x42, 0x91, 0x65, 0x09, 0xfb, 0x7e, 0x7e,
					0xd9, 0xcf, 0x7f, 0x77, 0x45, 0x03, 0x8d, 0x5c,
					0xcd, 0x52, 0x82, 0x19, 0xbc, 0x94, 0x8c, 0x78,
					0x13, 0xe0, 0x94, 0xc2, 0x96, 0x62, 0x82, 0x20,
					0xb9, 0xf1, 0x3a, 0x05, 0xfa, 0x94, 0x06, 0xbd,
					0xf6, 0x67, 0xa3, 0xca, 0xa5, 0x3a, 0xd5, 0xb5,
					0x34, 0xa9, 0xe8, 0x7e, 0x9f, 0x2f, 0x53, 0xde,
					0x8b, 0xd6, 0x3c, 0x2f, 0x2d, 0xb4, 0x56, 0x0c,
					0xc5, 0x3e, 0x7a, 0xa7, 0x81, 0x5c, 0x35, 0x60,
					0xb3, 0x0c, 0x28, 0x2c, 0x08, 0x06, 0xc0, 0xe0,
					0x3c, 0x0a, 0xfa, 0x1a, 0x6f, 0x43, 0x55, 0xbe,
					0x05, 0x5a, 0x53, 0xae, 0xcb, 0x74, 0xa9, 0xe8,
					0x7e, 0x9f, 0x2f, 0x53, 0xde, 0x8b, 0xd6, 0x20,
					0x36, 0xce, 0xcb, 0xcd, 0x95, 0x15, 0x08, 0xaa,
					0x82, 0x13, 0x51, 0x48, 0xc1, 0x09, 0x28, 0x46,
					0x11, 0x0b, 0x3b, 0x41, 0x34, 0x50, 0x24, 0x18,
					0xa7, 0x72, 0x88, 0x99, 0x49, 0x17, 0x63, 0xac,
					0xa7, 0x98, 0x7e, 0x81, 0x7b, 0x13, 0x9d, 0x7f,
					0xd3,
				}},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x12, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x03, 0xe1, 0x01,
					0xf0, 0x00, 0x8d, 0xff, 0x34, 0x11,
				}, bytes.Repeat([]byte{0xff}, 162)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                7,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
					RandomAccessIndicator: true,
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xc0, 0x00, 0xd9, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0xff, 0xfa,
					0x52, 0x04, 0xa9, 0xbe, 0xe4, 0x8f, 0xf0, 0xfd,
					0x02, 0xdc, 0x80, 0x00, 0x30, 0x00, 0x22, 0xc1,
					0x5b, 0x90, 0x14, 0x23, 0x24, 0x05, 0x58, 0x3f,
					0x72, 0x02, 0x84, 0xc4, 0xc0, 0xc5, 0x07, 0xae,
					0x40, 0x21, 0xbc, 0x98, 0x90, 0xfa, 0x3a, 0x2d,
					0xda, 0x07, 0xe1, 0x4d, 0xa9, 0x9a, 0xb8, 0xa2,
					0x3b, 0x20, 0xc1, 0xc1, 0xba, 0x08, 0x94, 0x30,
					0x8b, 0xc5, 0x69, 0x51, 0x95, 0xd5, 0xd7, 0x42,
					0x91, 0x65, 0x09, 0xfb, 0x7e, 0x7e, 0xd9, 0xcf,
					0x7f, 0x77, 0x45, 0x03, 0x8d, 0x5c, 0xcd, 0x52,
					0x82, 0x19, 0xbc, 0x94, 0x8c, 0x78, 0x13, 0xe0,
					0x94, 0xc2, 0x96, 0x62, 0x82, 0x20, 0xb9, 0xf1,
					0x3a, 0x05, 0xfa, 0x94, 0x06, 0xbd, 0xf6, 0x67,
					0xa3, 0xca, 0xa5, 0x3a, 0xd5, 0xb5, 0x34, 0xa9,
					0xe8, 0x7e, 0x9f, 0x2f, 0x53, 0xde, 0x8b, 0xd6,
					0x3c, 0x2f, 0x2d, 0xb4, 0x56, 0x0c, 0xc5, 0x3e,
					0x7a, 0xa7, 0x81, 0x5c, 0x35, 0x60, 0xb3, 0x0c,
					0x28, 0x2c, 0x08, 0x06, 0xc0, 0xe0, 0x3c, 0x0a,
					0xfa, 0x1a, 0x6f, 0x43, 0x55, 0xbe, 0x05, 0x5a,
					0x53, 0xae, 0xcb, 0x74, 0xa9, 0xe8, 0x7e, 0x9f,
				},
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:         136,
					StuffingLength: 135,
				},
				Header: astits.PacketHeader{
					ContinuityCounter:         1,
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: false,
					PID:                       257,
				},
				Payload: []byte{
					0x2f, 0x53, 0xde, 0x8b, 0xd6, 0x20, 0x36, 0xce,
					0xcb, 0xcd, 0x95, 0x15, 0x08, 0xaa, 0x82, 0x13,
					0x51, 0x48, 0xc1, 0x09, 0x28, 0x46, 0x11, 0x0b,
					0x3b, 0x41, 0x34, 0x50, 0x24, 0x18, 0xa7, 0x72,
					0x88, 0x99, 0x49, 0x17, 0x63, 0xac, 0xa7, 0x98,
					0x7e, 0x81, 0x7b, 0x13, 0x9d, 0x7f, 0xd3,
				},
			},
		},
	},
	{
		"ac-3",
		&Track{
			PID: 257,
			Codec: &CodecAC3{
				SampleRate:   48000,
				ChannelCount: 1,
			},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{{
					0x0b, 0x77, 0x47, 0x11, 0x0c, 0x40, 0x2f, 0x84,
					0x2b, 0xc1, 0x07, 0x7a, 0xb0, 0xfa, 0xbb, 0xea,
					0xef, 0x9f, 0x57, 0x7c, 0xf9, 0xf3, 0xf7, 0xcf,
					0x9f, 0x3e, 0x32, 0xfe, 0xd5, 0xc1, 0x50, 0xde,
					0xc5, 0x1e, 0x73, 0xd2, 0x6c, 0xa6, 0x94, 0x46,
					0x4e, 0x92, 0x8c, 0x0f, 0xb9, 0xcf, 0xad, 0x07,
					0x54, 0x4a, 0x2e, 0xf3, 0x7d, 0x07, 0x2e, 0xa4,
					0x2f, 0xba, 0xbf, 0x39, 0xb5, 0xc9, 0x92, 0xa6,
					0xe1, 0xb4, 0x70, 0xc5, 0xc4, 0xb5, 0xe6, 0x5d,
					0x0f, 0xa8, 0x71, 0xa4, 0xcc, 0xc5, 0xbc, 0x75,
					0x67, 0x92, 0x52, 0x4f, 0x7e, 0x62, 0x1c, 0xa9,
					0xd9, 0xb5, 0x19, 0x6a, 0xd7, 0xb0, 0x44, 0x92,
					0x30, 0x3b, 0xf7, 0x61, 0xd6, 0x49, 0x96, 0x66,
					0x98, 0x28, 0x1a, 0x95, 0xa9, 0x42, 0xad, 0xb7,
					0x50, 0x90, 0xad, 0x1c, 0x34, 0x80, 0xe2, 0xef,
					0xcd, 0x41, 0x0b, 0xf0, 0x9d, 0x57, 0x62, 0x78,
					0xfd, 0xc6, 0xc2, 0x19, 0x9e, 0x26, 0x31, 0xca,
					0x1e, 0x75, 0xb1, 0x7a, 0x8e, 0xb5, 0x51, 0x3a,
					0xfe, 0xe4, 0xf1, 0x0b, 0x4f, 0x14, 0x90, 0xdb,
					0x9f, 0x44, 0x50, 0xbb, 0xef, 0x74, 0x00, 0x8c,
					0x1f, 0x97, 0xa1, 0xa2, 0xfa, 0x72, 0x16, 0x47,
					0xc6, 0xc0, 0xe5, 0xfe, 0x67, 0x03, 0x9c, 0xfe,
					0x62, 0x01, 0xa1, 0x00, 0x5d, 0xff, 0xa5, 0x03,
					0x59, 0xfa, 0xa8, 0x25, 0x5f, 0x6b, 0x83, 0x51,
					0xf2, 0xc0, 0x44, 0xff, 0x2d, 0x05, 0x4b, 0xee,
					0xe0, 0x54, 0x9e, 0xae, 0x86, 0x45, 0xf3, 0xbd,
					0x0e, 0x42, 0xf2, 0xbf, 0x0f, 0x7f, 0xc6, 0x09,
					0x07, 0xdc, 0x22, 0x11, 0x77, 0xbe, 0x31, 0x27,
					0x5b, 0xa4, 0x13, 0x47, 0x07, 0x32, 0x9f, 0x1f,
					0xcb, 0xb0, 0xdf, 0x3e, 0x7d, 0x0d, 0xf3, 0xe7,
					0xcf, 0x9f, 0x3e, 0xae, 0xf9, 0xf3, 0xe7, 0xcf,
					0x9f, 0x3e, 0x85, 0x5d, 0xf3, 0xe7, 0xcf, 0x9f,
					0x3e, 0x7c, 0xf9, 0xf3, 0xe7, 0xcf, 0x9f, 0x3f,
					0x53, 0x5d, 0xf3, 0xe7, 0xcf, 0x9f, 0x3e, 0x7c,
					0xf9, 0xf3, 0xe7, 0xcf, 0x9f, 0x3e, 0x7c, 0xf9,
					0xf3, 0xe7, 0xcf, 0x9f, 0x3e, 0x7c, 0xf9, 0xf3,
					0xe7, 0xcf, 0x9f, 0x3e, 0x00, 0x46, 0x28, 0x26,
					0x20, 0x4a, 0x5a, 0xc0, 0x8a, 0xc5, 0xae, 0xa0,
					0x55, 0x78, 0x82, 0x7a, 0x38, 0x10, 0x09, 0xc9,
					0xb8, 0x0c, 0xfa, 0x5b, 0xc9, 0xd2, 0xec, 0x44,
					0x25, 0xf8, 0x20, 0xf2, 0xc8, 0x8a, 0xe9, 0x40,
					0x18, 0x06, 0xc6, 0x2b, 0xc8, 0xed, 0x8f, 0x33,
					0x09, 0x92, 0x28, 0x1e, 0xc4, 0x24, 0xd8, 0x33,
					0xa5, 0x00, 0xf5, 0xea, 0x18, 0xfa, 0x90, 0x97,
					0x97, 0xe8, 0x39, 0x6a, 0xcf, 0xf1, 0xdd, 0xff,
					0x9e, 0x8e, 0x04, 0x02, 0xae, 0x65, 0x87, 0x5c,
					0x4e, 0x72, 0xfd, 0x3c, 0x01, 0x86, 0xfe, 0x56,
					0x59, 0x74, 0x44, 0x3a, 0x40, 0x00, 0xec, 0xfc,
				}},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x12, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x81, 0xe1, 0x01,
					0xf0, 0x00, 0x12, 0x71, 0xfd, 0xb7,
				}, bytes.Repeat([]byte{0xff}, 162)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                7,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
					RandomAccessIndicator: true,
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xc0, 0x01, 0x88, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0x0b, 0x77,
					0x47, 0x11, 0x0c, 0x40, 0x2f, 0x84, 0x2b, 0xc1,
					0x07, 0x7a, 0xb0, 0xfa, 0xbb, 0xea, 0xef, 0x9f,
					0x57, 0x7c, 0xf9, 0xf3, 0xf7, 0xcf, 0x9f, 0x3e,
					0x32, 0xfe, 0xd5, 0xc1, 0x50, 0xde, 0xc5, 0x1e,
					0x73, 0xd2, 0x6c, 0xa6, 0x94, 0x46, 0x4e, 0x92,
					0x8c, 0x0f, 0xb9, 0xcf, 0xad, 0x07, 0x54, 0x4a,
					0x2e, 0xf3, 0x7d, 0x07, 0x2e, 0xa4, 0x2f, 0xba,
					0xbf, 0x39, 0xb5, 0xc9, 0x92, 0xa6, 0xe1, 0xb4,
					0x70, 0xc5, 0xc4, 0xb5, 0xe6, 0x5d, 0x0f, 0xa8,
					0x71, 0xa4, 0xcc, 0xc5, 0xbc, 0x75, 0x67, 0x92,
					0x52, 0x4f, 0x7e, 0x62, 0x1c, 0xa9, 0xd9, 0xb5,
					0x19, 0x6a, 0xd7, 0xb0, 0x44, 0x92, 0x30, 0x3b,
					0xf7, 0x61, 0xd6, 0x49, 0x96, 0x66, 0x98, 0x28,
					0x1a, 0x95, 0xa9, 0x42, 0xad, 0xb7, 0x50, 0x90,
					0xad, 0x1c, 0x34, 0x80, 0xe2, 0xef, 0xcd, 0x41,
					0x0b, 0xf0, 0x9d, 0x57, 0x62, 0x78, 0xfd, 0xc6,
					0xc2, 0x19, 0x9e, 0x26, 0x31, 0xca, 0x1e, 0x75,
					0xb1, 0x7a, 0x8e, 0xb5, 0x51, 0x3a, 0xfe, 0xe4,
					0xf1, 0x0b, 0x4f, 0x14, 0x90, 0xdb, 0x9f, 0x44,
					0x50, 0xbb, 0xef, 0x74, 0x00, 0x8c, 0x1f, 0x97,
				},
			},
			{ // PES
				Header: astits.PacketHeader{
					ContinuityCounter:         1,
					HasPayload:                true,
					PayloadUnitStartIndicator: false,
					PID:                       257,
				},
				Payload: []byte{
					0xa1, 0xa2, 0xfa, 0x72, 0x16, 0x47, 0xc6, 0xc0,
					0xe5, 0xfe, 0x67, 0x03, 0x9c, 0xfe, 0x62, 0x01,
					0xa1, 0x00, 0x5d, 0xff, 0xa5, 0x03, 0x59, 0xfa,
					0xa8, 0x25, 0x5f, 0x6b, 0x83, 0x51, 0xf2, 0xc0,
					0x44, 0xff, 0x2d, 0x05, 0x4b, 0xee, 0xe0, 0x54,
					0x9e, 0xae, 0x86, 0x45, 0xf3, 0xbd, 0x0e, 0x42,
					0xf2, 0xbf, 0x0f, 0x7f, 0xc6, 0x09, 0x07, 0xdc,
					0x22, 0x11, 0x77, 0xbe, 0x31, 0x27, 0x5b, 0xa4,
					0x13, 0x47, 0x07, 0x32, 0x9f, 0x1f, 0xcb, 0xb0,
					0xdf, 0x3e, 0x7d, 0x0d, 0xf3, 0xe7, 0xcf, 0x9f,
					0x3e, 0xae, 0xf9, 0xf3, 0xe7, 0xcf, 0x9f, 0x3e,
					0x85, 0x5d, 0xf3, 0xe7, 0xcf, 0x9f, 0x3e, 0x7c,
					0xf9, 0xf3, 0xe7, 0xcf, 0x9f, 0x3f, 0x53, 0x5d,
					0xf3, 0xe7, 0xcf, 0x9f, 0x3e, 0x7c, 0xf9, 0xf3,
					0xe7, 0xcf, 0x9f, 0x3e, 0x7c, 0xf9, 0xf3, 0xe7,
					0xcf, 0x9f, 0x3e, 0x7c, 0xf9, 0xf3, 0xe7, 0xcf,
					0x9f, 0x3e, 0x00, 0x46, 0x28, 0x26, 0x20, 0x4a,
					0x5a, 0xc0, 0x8a, 0xc5, 0xae, 0xa0, 0x55, 0x78,
					0x82, 0x7a, 0x38, 0x10, 0x09, 0xc9, 0xb8, 0x0c,
					0xfa, 0x5b, 0xc9, 0xd2, 0xec, 0x44, 0x25, 0xf8,
					0x20, 0xf2, 0xc8, 0x8a, 0xe9, 0x40, 0x18, 0x06,
					0xc6, 0x2b, 0xc8, 0xed, 0x8f, 0x33, 0x09, 0x92,
					0x28, 0x1e, 0xc4, 0x24, 0xd8, 0x33, 0xa5, 0x00,
				},
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:         145,
					StuffingLength: 144,
				},
				Header: astits.PacketHeader{
					ContinuityCounter:         2,
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: false,
					PID:                       257,
				},
				Payload: []byte{
					0xf5, 0xea, 0x18, 0xfa, 0x90, 0x97, 0x97, 0xe8,
					0x39, 0x6a, 0xcf, 0xf1, 0xdd, 0xff, 0x9e, 0x8e,
					0x04, 0x02, 0xae, 0x65, 0x87, 0x5c, 0x4e, 0x72,
					0xfd, 0x3c, 0x01, 0x86, 0xfe, 0x56, 0x59, 0x74,
					0x44, 0x3a, 0x40, 0x00, 0xec, 0xfc,
				},
			},
		},
	},
	{
		"klv sync",
		&Track{
			PID: 257,
			Codec: &CodecKLV{
				Synchronous: true,
			},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{{1, 2, 3}},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x28, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x15, 0xe1, 0x01,
					0xf0, 0x16, 0x26, 0x09, 0x01, 0x00, 0xff, 0x4b,
					0x4c, 0x56, 0x41, 0x00, 0x0f, 0x27, 0x09, 0xc0,
					0x00, 0x00, 0xc0, 0x00, 0x00, 0xc0, 0x00, 0x00,
					0x4c, 0x5c, 0xe5, 0x50,
				}, bytes.Repeat([]byte{0xff}, 140)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                161,
					StuffingLength:        154,
					RandomAccessIndicator: true,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xfc, 0x00, 0x10, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0x00, 0x00,
					0xdf, 0x00, 0x03, 0x01, 0x02, 0x03,
				},
			},
		},
	},
	{
		"dvb subtitles",
		&Track{
			PID: 257,
			Codec: &CodecDVBSubtitle{
				Items: []*astits.DescriptorSubtitlingItem{
					{
						AncillaryPageID:   123,
						CompositionPageID: 456,
						Language:          []byte{1, 2, 3},
						Type:              22,
					},
					{
						AncillaryPageID:   33,
						CompositionPageID: 12,
						Language:          []byte{'a', 'b', 'c'},
						Type:              15,
					},
				},
			},
		},
		[]sample{
			{
				30 * 90000,
				30 * 90000,
				[][]byte{{1, 2, 3}},
			},
		},
		[]*astits.Packet{
			{ // PMT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       0,
				},
				Payload: append([]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				}, bytes.Repeat([]byte{0xff}, 167)...),
			},
			{ // PAT
				Header: astits.PacketHeader{
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       4096,
				},
				Payload: append([]byte{
					0x00, 0x02, 0xb0, 0x24, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x01, 0xf0, 0x00, 0x06, 0xe1, 0x01,
					0xf0, 0x12, 0x59, 0x10, 0x01, 0x02, 0x03, 0x16,
					0x01, 0xc8, 0x00, 0x7b, 0x61, 0x62, 0x63, 0x0f,
					0x00, 0x0c, 0x00, 0x21, 0xc2, 0xdc, 0x16, 0x2a,
				}, bytes.Repeat([]byte{0xff}, 144)...),
			},
			{ // PES
				AdaptationField: &astits.PacketAdaptationField{
					Length:                166,
					StuffingLength:        159,
					RandomAccessIndicator: true,
					HasPCR:                true,
					PCR:                   &astits.ClockReference{Base: 2691000},
				},
				Header: astits.PacketHeader{
					HasAdaptationField:        true,
					HasPayload:                true,
					PayloadUnitStartIndicator: true,
					PID:                       257,
				},
				Payload: []byte{
					0x00, 0x00, 0x01, 0xbd, 0x00, 0x0b, 0x80, 0x80,
					0x05, 0x21, 0x00, 0xa5, 0x65, 0xc1, 0x01, 0x02,
					0x03,
				},
			},
		},
	},
}

func TestReader(t *testing.T) {
	for _, ca := range casesReadWriter {
		t.Run(ca.name, func(t *testing.T) {
			var buf bytes.Buffer
			mux := astits.NewMuxer(context.Background(), &buf)

			for _, packet := range ca.packets {
				_, err := mux.WritePacket(packet)
				require.NoError(t, err)
			}

			r, err := NewReader(&buf)
			require.NoError(t, err)
			require.Equal(t, ca.track, r.Tracks()[0])

			i := 0

			switch ca.track.Codec.(type) {
			case *CodecH265:
				r.OnDataH265(ca.track, func(pts int64, dts int64, au [][]byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].dts, dts)
					require.Equal(t, ca.samples[i].data, au)
					i++
					return nil
				})

			case *CodecH264:
				r.OnDataH264(ca.track, func(pts int64, dts int64, au [][]byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].dts, dts)
					require.Equal(t, ca.samples[i].data, au)
					i++
					return nil
				})

			case *CodecMPEG4Video:
				r.OnDataMPEGxVideo(ca.track, func(pts int64, frame []byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].data[0], frame)
					i++
					return nil
				})

			case *CodecMPEG1Video:
				r.OnDataMPEGxVideo(ca.track, func(pts int64, frame []byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].data[0], frame)
					i++
					return nil
				})

			case *CodecOpus:
				r.OnDataOpus(ca.track, func(pts int64, packets [][]byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].data, packets)
					i++
					return nil
				})

			case *CodecMPEG4Audio:
				r.OnDataMPEG4Audio(ca.track, func(pts int64, aus [][]byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].data, aus)
					i++
					return nil
				})

			case *CodecMPEG4AudioLATM:
				r.OnDataMPEG4AudioLATM(ca.track, func(pts int64, aus [][]byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].data, aus)
					i++
					return nil
				})

			case *CodecMPEG1Audio:
				r.OnDataMPEG1Audio(ca.track, func(pts int64, frames [][]byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].data, frames)
					i++
					return nil
				})

			case *CodecAC3:
				r.OnDataAC3(ca.track, func(pts int64, frame []byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].data[0], frame)
					i++
					return nil
				})

			case *CodecKLV:
				r.OnDataKLV(ca.track, func(pts int64, frame []byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].data[0], frame)
					i++
					return nil
				})

			case *CodecDVBSubtitle:
				r.OnDataDVBSubtitle(ca.track, func(pts int64, data []byte) error {
					require.Equal(t, ca.samples[i].pts, pts)
					require.Equal(t, ca.samples[i].data[0], data)
					i++
					return nil
				})

			default:
				panic("unexpected")
			}

			for {
				err = r.Read()
				if errors.Is(err, io.EOF) {
					break
				}
				require.NoError(t, err)
			}

			require.Equal(t, len(ca.samples), i)
		})
	}
}

func TestReaderKLVAsync(t *testing.T) {
	input := []*astits.Packet{
		{
			Header: astits.PacketHeader{
				HasPayload:                true,
				PayloadUnitStartIndicator: true,
			},
			Payload: append(
				[]byte{
					0x00, 0x00, 0xb0, 0x0d, 0x00, 0x00, 0xc1, 0x00,
					0x00, 0x00, 0x01, 0xf0, 0x00, 0x71, 0x10, 0xd8,
					0x78,
				},
				bytes.Repeat([]byte{0xff}, 167)...,
			),
		},
		{
			Header: astits.PacketHeader{
				HasPayload:                true,
				PayloadUnitStartIndicator: true,
				PID:                       4096,
			},
			Payload: append(
				[]byte{
					0x00, 0x02, 0xb0, 0x1d, 0x00, 0x01, 0xc1, 0x00,
					0x00, 0xe1, 0x00, 0xf0, 0x00, 0x1b, 0xe1, 0x00,
					0xf0, 0x00, 0x06, 0xe1, 0x01, 0xf0, 0x06, 0x05,
					0x04, 0x4b, 0x4c, 0x56, 0x41, 0x06, 0x71, 0x49,
					0xd4,
				},
				bytes.Repeat([]byte{0xff}, 151)...,
			),
		},
		{
			Header: astits.PacketHeader{
				HasPayload:                true,
				PayloadUnitStartIndicator: true,
				PID:                       256,
				HasAdaptationField:        true,
			},
			AdaptationField: &astits.PacketAdaptationField{
				PCR: &astits.ClockReference{
					Base: 81000,
				},
				Length:         155,
				StuffingLength: 148,
				HasPCR:         true,
			},
			Payload: []byte{
				0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x80, 0x80,
				0x05, 0x21, 0x00, 0x05, 0xbf, 0x21, 0x00, 0x00,
				0x00, 0x01, 0x09, 0xf0, 0x00, 0x00, 0x00, 0x01,
				0x01, 0x02, 0x03, 0x04,
			},
		},
		{
			Header: astits.PacketHeader{
				HasAdaptationField:        true,
				HasPayload:                true,
				PayloadUnitStartIndicator: true,
				PID:                       257,
			},
			AdaptationField: &astits.PacketAdaptationField{
				Length:                170,
				StuffingLength:        169,
				RandomAccessIndicator: true,
			},
			Payload: []byte{
				0x00, 0x00, 0x01, 0xbd, 0x00, 0x07, 0x80, 0x00,
				0x00, 0x05, 0x06, 0x07, 0x08,
			},
		},
	}

	var buf bytes.Buffer
	mux := astits.NewMuxer(context.Background(), &buf)

	for _, packet := range input {
		_, err := mux.WritePacket(packet)
		require.NoError(t, err)
	}

	r, err := NewReader(&buf)
	require.NoError(t, err)
	require.Equal(t, []*Track{
		{
			PID:   256,
			Codec: &CodecH264{},
		},
		{
			PID: 257,
			Codec: &CodecKLV{
				Synchronous: false,
			},
		},
	}, r.Tracks())

	recv1 := false
	recv2 := false

	r.OnDataH264(r.Tracks()[0], func(pts int64, _ int64, au [][]byte) error {
		require.Equal(t, int64(90000), pts)
		require.Equal(t, [][]byte{{1, 2, 3, 4}}, au)
		recv1 = true
		return nil
	})

	r.OnDataKLV(r.Tracks()[1], func(pts int64, data []byte) error {
		require.Equal(t, int64(90000), pts)
		require.Equal(t, []byte{5, 6, 7, 8}, data)
		recv2 = true
		return nil
	})

	for {
		err = r.Read()
		if errors.Is(err, io.EOF) {
			break
		}
		require.NoError(t, err)
	}

	require.True(t, recv1)
	require.True(t, recv2)
}

func TestReaderDecodeErrors(t *testing.T) {
	for _, ca := range []string{
		"missing pts",
		"h26x invalid avcc",
		"opus pts != dts",
		"opus invalid au",
		"mpeg-4 audio pts != dts",
		"mpeg-4 audio invalid",
		"mpeg-1 audio pts != dts",
		"ac-3 pts != dts",
	} {
		t.Run(ca, func(t *testing.T) {
			var buf bytes.Buffer
			mux := astits.NewMuxer(context.Background(), &buf)

			switch ca {
			case "missing pts", "h26x invalid avcc":
				err := mux.AddElementaryStream(astits.PMTElementaryStream{
					ElementaryPID: 123,
					StreamType:    astits.StreamTypeH264Video,
				})
				require.NoError(t, err)

			case "opus pts != dts", "opus invalid au":
				err := mux.AddElementaryStream(astits.PMTElementaryStream{
					ElementaryPID: 123,
					StreamType:    astits.StreamTypePrivateData,
					ElementaryStreamDescriptors: []*astits.Descriptor{
						{
							Length: 4,
							Tag:    astits.DescriptorTagRegistration,
							Registration: &astits.DescriptorRegistration{
								FormatIdentifier: opusIdentifier,
							},
						},
						{
							Length: 2,
							Tag:    astits.DescriptorTagExtension,
							Extension: &astits.DescriptorExtension{
								Tag:     0x80,
								Unknown: &[]uint8{2},
							},
						},
					},
				})
				require.NoError(t, err)

			case "mpeg-4 audio pts != dts", "mpeg-4 audio invalid":
				err := mux.AddElementaryStream(astits.PMTElementaryStream{
					ElementaryPID: 123,
					StreamType:    astits.StreamTypeAACAudio,
				})
				require.NoError(t, err)

			case "mpeg-1 audio pts != dts":
				err := mux.AddElementaryStream(astits.PMTElementaryStream{
					ElementaryPID: 123,
					StreamType:    astits.StreamTypeMPEG1Audio,
				})
				require.NoError(t, err)

			case "ac-3 pts != dts":
				err := mux.AddElementaryStream(astits.PMTElementaryStream{
					ElementaryPID: 123,
					StreamType:    astits.StreamTypeAC3Audio,
				})
				require.NoError(t, err)
			}

			mux.SetPCRPID(123)

			switch ca {
			case "missing pts":
				_, err := mux.WriteData(&astits.MuxerData{
					PID: 123,
					PES: &astits.PESData{
						Header: &astits.PESHeader{
							OptionalHeader: &astits.PESOptionalHeader{
								MarkerBits:      2,
								PTSDTSIndicator: astits.PTSDTSIndicatorNoPTSOrDTS,
							},
							StreamID: streamIDVideo,
						},
						Data: []byte{1, 2, 3, 4},
					},
				})
				require.NoError(t, err)

			case "h26x invalid avcc", "opus invalid au":
				_, err := mux.WriteData(&astits.MuxerData{
					PID: 123,
					PES: &astits.PESData{
						Header: &astits.PESHeader{
							OptionalHeader: &astits.PESOptionalHeader{
								MarkerBits:      2,
								PTSDTSIndicator: astits.PTSDTSIndicatorOnlyPTS,
								PTS:             &astits.ClockReference{Base: 90000},
							},
							StreamID: streamIDVideo,
						},
						Data: []byte{1, 2, 3, 4},
					},
				})
				require.NoError(t, err)

			case "opus pts != dts", "mpeg-1 audio pts != dts":
				_, err := mux.WriteData(&astits.MuxerData{
					PID: 123,
					PES: &astits.PESData{
						Header: &astits.PESHeader{
							OptionalHeader: &astits.PESOptionalHeader{
								MarkerBits:      2,
								PTSDTSIndicator: astits.PTSDTSIndicatorBothPresent,
								PTS:             &astits.ClockReference{Base: 90000},
								DTS:             &astits.ClockReference{Base: 180000},
							},
							StreamID: streamIDVideo,
						},
						Data: []byte{1, 2, 3, 4},
					},
				})
				require.NoError(t, err)

			case "mpeg-4 audio pts != dts":
				data, _ := mpeg4audio.ADTSPackets{{
					Type:         mpeg4audio.ObjectTypeAACLC,
					SampleRate:   44100,
					ChannelCount: 1,
					AU:           []byte{1, 2, 3, 4},
				}}.Marshal()

				_, err := mux.WriteData(&astits.MuxerData{
					PID: 123,
					PES: &astits.PESData{
						Header: &astits.PESHeader{
							OptionalHeader: &astits.PESOptionalHeader{
								MarkerBits:      2,
								PTSDTSIndicator: astits.PTSDTSIndicatorBothPresent,
								PTS:             &astits.ClockReference{Base: 90000},
								DTS:             &astits.ClockReference{Base: 180000},
							},
							StreamID: streamIDVideo,
						},
						Data: data,
					},
				})
				require.NoError(t, err)

				_, err = mux.WriteData(&astits.MuxerData{
					PID: 123,
					PES: &astits.PESData{
						Header: &astits.PESHeader{
							OptionalHeader: &astits.PESOptionalHeader{
								MarkerBits:      2,
								PTSDTSIndicator: astits.PTSDTSIndicatorBothPresent,
								PTS:             &astits.ClockReference{Base: 90000},
								DTS:             &astits.ClockReference{Base: 180000},
							},
							StreamID: streamIDVideo,
						},
						Data: []byte{1, 2, 3, 4},
					},
				})
				require.NoError(t, err)

			case "mpeg-4 audio invalid":
				data, _ := mpeg4audio.ADTSPackets{{
					Type:         mpeg4audio.ObjectTypeAACLC,
					SampleRate:   44100,
					ChannelCount: 1,
					AU:           []byte{1, 2, 3, 4},
				}}.Marshal()

				_, err := mux.WriteData(&astits.MuxerData{
					PID: 123,
					PES: &astits.PESData{
						Header: &astits.PESHeader{
							OptionalHeader: &astits.PESOptionalHeader{
								MarkerBits:      2,
								PTSDTSIndicator: astits.PTSDTSIndicatorOnlyPTS,
								PTS:             &astits.ClockReference{Base: 90000},
							},
							StreamID: streamIDVideo,
						},
						Data: data,
					},
				})
				require.NoError(t, err)

				_, err = mux.WriteData(&astits.MuxerData{
					PID: 123,
					PES: &astits.PESData{
						Header: &astits.PESHeader{
							OptionalHeader: &astits.PESOptionalHeader{
								MarkerBits:      2,
								PTSDTSIndicator: astits.PTSDTSIndicatorOnlyPTS,
								PTS:             &astits.ClockReference{Base: 90000},
							},
							StreamID: streamIDVideo,
						},
						Data: []byte{1, 2, 3, 4},
					},
				})
				require.NoError(t, err)

			case "ac-3 pts != dts":
				_, err := mux.WriteData(&astits.MuxerData{
					PID: 123,
					PES: &astits.PESData{
						Header: &astits.PESHeader{
							OptionalHeader: &astits.PESOptionalHeader{
								MarkerBits:      2,
								PTSDTSIndicator: astits.PTSDTSIndicatorBothPresent,
								PTS:             &astits.ClockReference{Base: 90000},
								DTS:             &astits.ClockReference{Base: 180000},
							},
							StreamID: streamIDVideo,
						},
						Data: []byte{
							0x0b, 0x77, 0x47, 0x11, 0x0c, 0x40, 0x2f, 0x84,
							0x2b, 0xc1, 0x07, 0x7a, 0xb0, 0xfa, 0xbb, 0xea,
							0xef, 0x9f, 0x57, 0x7c, 0xf9, 0xf3, 0xf7, 0xcf,
							0x9f, 0x3e, 0x32, 0xfe, 0xd5, 0xc1, 0x50, 0xde,
							0xc5, 0x1e, 0x73, 0xd2, 0x6c, 0xa6, 0x94, 0x46,
						},
					},
				})
				require.NoError(t, err)

				_, err = mux.WriteData(&astits.MuxerData{
					PID: 123,
					PES: &astits.PESData{
						Header: &astits.PESHeader{
							OptionalHeader: &astits.PESOptionalHeader{
								MarkerBits:      2,
								PTSDTSIndicator: astits.PTSDTSIndicatorBothPresent,
								PTS:             &astits.ClockReference{Base: 90000},
								DTS:             &astits.ClockReference{Base: 180000},
							},
							StreamID: streamIDVideo,
						},
						Data: []byte{1, 2, 3, 4},
					},
				})
				require.NoError(t, err)
			}

			r, err := NewReader(bytes.NewReader(buf.Bytes()))
			require.NoError(t, err)

			switch ca {
			case "missing pts", "h26x invalid avcc":
				r.OnDataH264(r.Tracks()[0], func(_, _ int64, _ [][]byte) error {
					return nil
				})

			case "opus pts != dts", "opus invalid au":
				r.OnDataOpus(r.Tracks()[0], func(_ int64, _ [][]byte) error {
					return nil
				})

			case "mpeg-4 audio pts != dts", "mpeg-4 audio invalid":
				r.OnDataMPEG4Audio(r.Tracks()[0], func(_ int64, _ [][]byte) error {
					return nil
				})

			case "mpeg-1 audio pts != dts":
				r.OnDataMPEG1Audio(r.Tracks()[0], func(_ int64, _ [][]byte) error {
					return nil
				})

			case "ac-3 pts != dts":
				r.OnDataAC3(r.Tracks()[0], func(_ int64, _ []byte) error {
					return nil
				})
			}

			decodeErrRecv := false

			r.OnDecodeError(func(err error) {
				switch ca {
				case "missing pts":
					require.EqualError(t, err, "PTS is missing")

				case "h26x invalid avcc":
					require.EqualError(t, err, "initial delimiter not found")

				case "opus pts != dts", "mpeg-4 audio pts != dts", "mpeg-1 audio pts != dts", "ac-3 pts != dts":
					require.EqualError(t, err, "PTS is not equal to DTS")

				case "opus invalid au":
					require.EqualError(t, err, "invalid control header: invalid prefix")

				case "mpeg-4 audio invalid":
					require.EqualError(t, err, "invalid ADTS: invalid length")
				}
				decodeErrRecv = true
			})

			for {
				err = r.Read()
				if err != nil {
					require.ErrorIs(t, io.EOF, err)
					break
				}
			}

			require.Equal(t, true, decodeErrRecv)
		})
	}
}

var errCustom = errors.New("custom error")

type dummyReader struct{}

func (dummyReader) Read(_ []byte) (int, error) {
	return 0, errCustom
}

func TestReaderFatalError(t *testing.T) {
	_, err := NewReader(&dummyReader{})
	require.Equal(t, errCustom, err)
}

func TestReaderSkipGarbage(t *testing.T) {
	var buf bytes.Buffer
	mux := astits.NewMuxer(context.Background(), &buf)

	err := mux.AddElementaryStream(astits.PMTElementaryStream{
		ElementaryPID: 123,
		StreamType:    astits.StreamTypeH264Video,
	})
	require.NoError(t, err)

	mux.SetPCRPID(123)

	_, err = mux.WriteData(&astits.MuxerData{
		PID: 123,
		PES: &astits.PESData{
			Header: &astits.PESHeader{
				OptionalHeader: &astits.PESOptionalHeader{
					MarkerBits:      2,
					PTSDTSIndicator: astits.PTSDTSIndicatorOnlyPTS,
					PTS:             &astits.ClockReference{Base: 90000},
				},
				StreamID: streamIDVideo,
			},
			Data: []byte{0, 0, 0, 1, 1, 2, 3, 4},
		},
	})
	require.NoError(t, err)

	// complete random garbage
	buf.Write(bytes.Repeat([]byte{1, 2, 3, 4}, 200/4))

	_, err = mux.WriteData(&astits.MuxerData{
		PID: 123,
		PES: &astits.PESData{
			Header: &astits.PESHeader{
				OptionalHeader: &astits.PESOptionalHeader{
					MarkerBits:      2,
					PTSDTSIndicator: astits.PTSDTSIndicatorOnlyPTS,
					PTS:             &astits.ClockReference{Base: 90000},
				},
				StreamID: streamIDVideo,
			},
			Data: []byte{0, 0, 0, 1, 5, 6, 7, 8},
		},
	})
	require.NoError(t, err)

	// this is eaten by the next garbage
	_, err = mux.WriteData(&astits.MuxerData{
		PID: 123,
		PES: &astits.PESData{
			Header: &astits.PESHeader{
				OptionalHeader: &astits.PESOptionalHeader{
					MarkerBits:      2,
					PTSDTSIndicator: astits.PTSDTSIndicatorOnlyPTS,
					PTS:             &astits.ClockReference{Base: 90000},
				},
				StreamID: streamIDVideo,
			},
			Data: []byte{0, 0, 0, 1, 9, 10, 11, 12},
		},
	})
	require.NoError(t, err)

	// syncword-prefixed garbage
	buf.Write([]byte{0x47})
	buf.Write(bytes.Repeat([]byte{1}, 100))

	// this is eaten by the previous garbage
	_, err = mux.WriteData(&astits.MuxerData{
		PID: 123,
		PES: &astits.PESData{
			Header: &astits.PESHeader{
				OptionalHeader: &astits.PESOptionalHeader{
					MarkerBits:      2,
					PTSDTSIndicator: astits.PTSDTSIndicatorOnlyPTS,
					PTS:             &astits.ClockReference{Base: 90000},
				},
				StreamID: streamIDVideo,
			},
			Data: []byte{0, 0, 0, 1, 13, 14, 15, 16},
		},
	})
	require.NoError(t, err)

	_, err = mux.WriteData(&astits.MuxerData{
		PID: 123,
		PES: &astits.PESData{
			Header: &astits.PESHeader{
				OptionalHeader: &astits.PESOptionalHeader{
					MarkerBits:      2,
					PTSDTSIndicator: astits.PTSDTSIndicatorOnlyPTS,
					PTS:             &astits.ClockReference{Base: 90000},
				},
				StreamID: streamIDVideo,
			},
			Data: []byte{0, 0, 0, 1, 17, 18, 19, 20},
		},
	})
	require.NoError(t, err)

	_, err = mux.WriteData(&astits.MuxerData{
		PID: 123,
		PES: &astits.PESData{
			Header: &astits.PESHeader{
				OptionalHeader: &astits.PESOptionalHeader{
					MarkerBits:      2,
					PTSDTSIndicator: astits.PTSDTSIndicatorOnlyPTS,
					PTS:             &astits.ClockReference{Base: 90000},
				},
				StreamID: streamIDVideo,
			},
			Data: []byte{0, 0, 0, 1, 21, 22, 23, 24},
		},
	})
	require.NoError(t, err)

	r, err := NewReader(bytes.NewReader(buf.Bytes()))
	require.NoError(t, err)

	var decErrs []string

	r.OnDecodeError(func(err error) {
		decErrs = append(decErrs, err.Error())
	})

	var aus [][][]byte

	r.OnDataH264(r.Tracks()[0], func(_, _ int64, au [][]byte) error {
		aus = append(aus, au)
		return nil
	})

	for {
		err = r.Read()
		if err != nil {
			require.ErrorIs(t, io.EOF, err)
			break
		}
	}

	require.Equal(t, []string{
		"skipped 188 bytes",
		"skipped 12 bytes",
		"skipped 101 bytes",
	}, decErrs)

	require.Equal(t, [][][]byte{
		{{1, 2, 3, 4}},
		{{5, 6, 7, 8}},
		{{17, 18, 19, 20}},
		{{21, 22, 23, 24}},
	}, aus)
}

func FuzzReader(f *testing.F) {
	for _, ca := range casesReadWriter {
		var buf bytes.Buffer
		mux := astits.NewMuxer(context.Background(), &buf)
		for _, pkt := range ca.packets {
			mux.WritePacket(pkt) //nolint:errcheck
		}
		f.Add(buf.Bytes())
	}

	f.Add([]byte{
		0x47, 0x40, 0x00, 0x10, 0x00, 0x00, 0xb0, 0x0d,
		0x00, 0x00, 0xc1, 0x00, 0x00, 0x00, 0x01, 0xf0,
		0x00, 0x71, 0x10, 0xd8, 0x78, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0x47, 0x50, 0x00, 0x10,
		0x00, 0x02, 0xb0, 0x12, 0x00, 0x01, 0xc1, 0x00,
		0x00, 0xe1, 0x00, 0xf0, 0x00, 0x06, 0xe1, 0x00,
		0xf0, 0x00, 0xbe, 0x7f, 0xa0, 0x52, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	})

	f.Add([]byte{
		0x47, 0x40, 0x00, 0x10, 0x00, 0x00, 0xb0, 0x0d,
		0x00, 0x00, 0xc1, 0x00, 0x00, 0x00, 0x01, 0xf0,
		0x00, 0x71, 0x10, 0xd8, 0x78, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0x47, 0x50, 0x00, 0x10,
		0x00, 0x02, 0xb0, 0x18, 0x00, 0x01, 0xc1, 0x00,
		0x00, 0xe1, 0x00, 0xf0, 0x00, 0x06, 0xe1, 0x00,
		0xf0, 0x06, 0x05, 0x04, 0x4f, 0x70, 0x75, 0x73,
		0xc7, 0x15, 0x35, 0x31, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	})

	f.Fuzz(func(t *testing.T, b []byte) {
		r := &Reader{R: bytes.NewReader(b)}
		err := r.Initialize()
		if err != nil {
			return
		}

		require.NotZero(t, len(r.Tracks()))
	})
}
