package h264

import (
	"encoding/binary"
	"fmt"
	"testing"

	"github.com/stretchr/testify/require"
)

type sample struct {
	au  [][]byte
	dts int64
	pts int64
}

var casesDTSExtractor = []struct {
	name     string
	sequence []sample
}{
	{
		"with timing info",
		[]sample{
			{
				[][]byte{
					{ // SPS
						0x67, 0x64, 0x00, 0x28, 0xac, 0xd9, 0x40, 0x78,
						0x02, 0x27, 0xe5, 0x84, 0x00, 0x00, 0x03, 0x00,
						0x04, 0x00, 0x00, 0x03, 0x00, 0xf0, 0x3c, 0x60,
						0xc6, 0x58,
					},
					{ // IDR
						0x65, 0x88, 0x84, 0x00, 0x33, 0xff,
					},
				},
				56890 + 0,
				56890 + 0,
			},
			{
				[][]byte{{0x41, 0x9a, 0x21, 0x6c, 0x45, 0xff}},
				56890 + 3000,
				56890 + 3000,
			},
			{
				[][]byte{{0x41, 0x9a, 0x42, 0x3c, 0x21, 0x93}},
				56890 + 6000,
				56890 + 6000,
			},
			{
				[][]byte{{0x41, 0x9a, 0x63, 0x49, 0xe1, 0x0f}},
				56890 + 9000,
				56890 + 9000,
			},
			{
				[][]byte{{0x41, 0x9a, 0x86, 0x49, 0xe1, 0x0f}},
				56890 + 9090,
				56890 + 18000,
			},
			{
				[][]byte{{0x41, 0x9e, 0xa5, 0x42, 0x7f, 0xf9}},
				56890 + 12045,
				56890 + 15000,
			},
			{
				[][]byte{{0x01, 0x9e, 0xc4, 0x69, 0x13, 0xff}},
				56890 + 12135,
				56890 + 12000,
			},
			{
				[][]byte{{0x41, 0x9a, 0xc8, 0x4b, 0xa8, 0x42}},
				56890 + 15101,
				56890 + 24000,
			},
			{
				[][]byte{
					{ // IDR
						0x65, 0x88, 0x84, 0x00, 0x33, 0xff,
					},
				},
				56890 + 18067,
				56890 + 24000,
			},
		},
	},
	{
		"no timing info",
		[]sample{
			{
				[][]byte{
					{ // SPS
						0x27, 0x64, 0x00, 0x20, 0xac, 0x52, 0x18, 0x0f,
						0x01, 0x17, 0xef, 0xff, 0x00, 0x01, 0x00, 0x01,
						0x6a, 0x02, 0x02, 0x03, 0x6d, 0x85, 0x6b, 0xde,
						0xf8, 0x08,
					},
					{ // IDR
						0x25, 0xb8, 0x08, 0x02, 0x1f, 0xff,
					},
				},
				35050 + 0,
				35050 + 0,
			},
			{
				[][]byte{{0x21, 0xe1, 0x05, 0xc7, 0x38, 0xbf}},
				35050 + 1500,
				35050 + 1500,
			},
			{
				[][]byte{{0x21, 0xe2, 0x09, 0xa1, 0xce, 0x0b}},
				35050 + 3000,
				35050 + 3000,
			},
			{
				[][]byte{{0x21, 0xe3, 0x0d, 0xb1, 0xce, 0x02}},
				35050 + 4500,
				35050 + 4500,
			},
			{
				[][]byte{{0x21, 0xe4, 0x11, 0x90, 0x73, 0x80}},
				35050 + 6000,
				35050 + 6000,
			},
			{
				[][]byte{{0x21, 0xe5, 0x19, 0x0e, 0x70, 0x01}},
				35050 + 6045,
				35050 + 6090,
			},
			{
				[][]byte{{0x01, 0xa9, 0x85, 0x7c, 0x93, 0xff}},
				35050 + 6135,
				35050 + 7500,
			},
			{
				[][]byte{{0x21, 0xe6, 0x1d, 0x0e, 0x70, 0x01}},
				35050 + 8317,
				35050 + 10500,
			},
			{
				[][]byte{{0x21, 0xe7, 0x21, 0x0e, 0x70, 0x01}},
				35050 + 10158,
				35050 + 12000,
			},
			{
				[][]byte{{0x21, 0xe8, 0x25, 0x0e, 0x70, 0x01}},
				35050 + 11829,
				35050 + 13500,
			},
			{
				[][]byte{{0x21, 0xe9, 0x29, 0x0e, 0x70, 0x01}},
				35050 + 13414,
				35050 + 15000,
			},
			{
				[][]byte{{0x21, 0xea, 0x31, 0x0e, 0x70, 0x01}},
				35050 + 14942,
				35050 + 18000,
			},
			{
				[][]byte{{0x01, 0xaa, 0xcb, 0x7c, 0x93, 0xff}},
				35050 + 16500,
				35050 + 16500,
			},
		},
	},
	{
		"poc increment = 1",
		[]sample{
			{
				[][]byte{
					{ // SPS
						0x67, 0x64, 0x00, 0x2a, 0xac, 0x2c, 0x6a, 0x81,
						0xe0, 0x08, 0x9f, 0x96, 0x6e, 0x02, 0x02, 0x02,
						0x80, 0x00, 0x03, 0x84, 0x00, 0x00, 0xaf, 0xc8,
						0x02,
					},
					{ // IDR
						0x65, 0xb8, 0x00, 0x00, 0x0b, 0xc8, 0x00, 0x00, 0x00,
					},
				},
				5490,
				5490,
			},
			{
				[][]byte{{0x61, 0xe0, 0x20, 0x00, 0x39, 0x37}},
				9090,
				9090,
			},
			{
				[][]byte{{0x61, 0xe0, 0x40, 0x00, 0x59, 0x37}},
				12690,
				12690,
			},
			{
				[][]byte{{0x61, 0xe0, 0x60, 0x00, 0x79, 0x37}},
				16290,
				16290,
			},
		},
	},
	{
		"B-frames after IDR (OBS 29.1.3 QuickSync on Windows)",
		[]sample{
			{
				[][]byte{
					{ // SPS
						0x27, 0x64, 0x00, 0x2a, 0xac, 0x2d, 0x90, 0x07,
						0x80, 0x22, 0x7e, 0x5c, 0x05, 0xa8, 0x08, 0x08,
						0x0a, 0x00, 0x00, 0x03, 0x00, 0x02, 0x00, 0x00,
						0x03, 0x00, 0xf1, 0xd0, 0x80, 0x04, 0xc4, 0x80,
						0x00, 0x09, 0x89, 0x68, 0xde, 0xf7, 0xc1, 0xda,
						0x1c, 0x31, 0x92,
					},
					{ // IDR
						0x65, 0x88, 0x80, 0x14, 0x3, 0xff, 0xde, 0x8, 0xe4, 0x74,
					},
				},
				172440,
				172440,
			},
			{ // b-frame
				[][]byte{{0x41, 0x9e, 0x3, 0xe4, 0x3f, 0x0, 0x0, 0x3, 0x0, 0x0}},
				172530,
				169470,
			},
			{ // b-frame
				[][]byte{{0x1, 0x9e, 0x5, 0xd4, 0x7f, 0x0, 0x0, 0x3, 0x0, 0x0}},
				172620,
				168030,
			},
			{ // p-frame
				[][]byte{{0x1, 0x9e, 0x5, 0xf4, 0x7f, 0x0, 0x0, 0x3, 0x0, 0x0}},
				172710,
				170910,
			},
			{ // p-frame
				[][]byte{{0x1, 0x9e, 0x5, 0xf4, 0x7f, 0x0, 0x0, 0x3, 0x0, 0x0}},
				172800,
				178470,
			},
		},
	},
	{
		"mbs_only_flag = 0",
		[]sample{
			{
				[][]byte{
					{ // SPS
						0x67, 0x4d, 0x40, 0x28, 0xab, 0x60, 0x3c, 0x02,
						0x23, 0xef, 0x01, 0x10, 0x00, 0x00, 0x03, 0x00,
						0x10, 0x00, 0x00, 0x03, 0x03, 0x2e, 0x94, 0x00,
						0x35, 0x64, 0x06, 0xb2, 0x85, 0x08, 0x0e, 0xe2,
						0xc5, 0x22, 0xc0,
					},
					{0x68, 0xca, 0x41, 0xf2},                                     // PPS
					{0x6, 0x0, 0x6, 0x85, 0x7e, 0x40, 0x0, 0x0, 0x10, 0x1},       // SEI
					{0x65, 0x88, 0x82, 0x80, 0x1f, 0xff, 0xfb, 0xf0, 0xa2, 0x88}, // IDR
					{0x6, 0x1, 0x2, 0x4, 0x24, 0x80},                             // SEI
					{0x41, 0x9a, 0xc, 0x1c, 0x2f, 0xe4, 0xed, 0x23, 0xb5, 0x63},  // non-IDR
				},
				0,
				0,
			},
			{
				[][]byte{
					{0x6, 0x1, 0x2, 0x8, 0x14, 0x80},                             // SEI
					{0x41, 0x9a, 0x18, 0x2a, 0x1f, 0xeb, 0x2f, 0xa2, 0xb1, 0x7e}, // non-IDR
				},
				3600,
				3600,
			},
			{
				[][]byte{
					{0x6, 0x1, 0x2, 0xc, 0x24, 0x80},                            // SEI
					{0x41, 0x9a, 0x1c, 0x3a, 0xf, 0xfa, 0x55, 0xc2, 0x55, 0xea}, // non-IDR
				},
				7200,
				7200,
			},
		},
	},
	{
		"Log2MaxPicOrderCntLsbMinus4 = 12",
		[]sample{
			{
				[][]byte{
					{ // SPS
						0x67, 0x42, 0xc0, 0x1e, 0x8c, 0x8d, 0x40, 0x50, 0x17, 0xfc, 0xb0, 0x0f, 0x08, 0x84, 0x6a,
					},
					{ // PPS
						0x68, 0xce, 0x3c, 0x80,
					},
					{ // IDR
						0x65, 0x88, 0x80, 0x14, 0x3, 0x00, 0x00, 0x00, 0x00, 0x00,
					},
				},
				0,
				0,
			},
			{
				[][]byte{
					{0x61, 0x00, 0xf0, 0xe0, 0x00, 0x40, 0x00, 0xbe, 0x47, 0x9b}, // non-IDR
				},
				3600,
				3600,
			},
		},
	},
	{
		"issue mediamtx/3094 (non-zero IDR POC)",
		[]sample{
			{
				[][]byte{
					{
						0x67, 0x42, 0x80, 0x28, 0x8c, 0x8d, 0x40, 0x5a,
						0x09, 0x22,
					},
					{
						0x68, 0xce, 0x3c, 0x80,
					},
					{
						0x65, 0xb8, 0x00, 0x0c, 0xa2, 0x40, 0x33, 0x93,
						0x14, 0x00, 0x04, 0x1a, 0x6d, 0x6d, 0x6d, 0x6d,
						0x6d, 0x6d, 0x5d, 0xaa, 0xb5, 0xb5, 0xb5, 0xb5,
						0xb5, 0xb5, 0xb5, 0xb5, 0xb5, 0xb5, 0xb5, 0xb5,
						0xb5, 0xb5, 0x76, 0xb6, 0xb6, 0xb6, 0xaa, 0xd6,
						0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6,
						0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd5,
						0xda, 0xda, 0xaa, 0x7a, 0x7a, 0x7a, 0x7a, 0x7a,
						0x79, 0x1e, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde,
						0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde,
						0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde,
						0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde,
						0xde, 0xde, 0xde, 0xde,
					},
				},
				0,
				0,
			},
			{
				[][]byte{
					{
						0x41, 0xe0, 0x00, 0x65, 0x12, 0x80, 0xce, 0x78,
						0x16, 0x00, 0x99, 0xff, 0xff, 0xff, 0xe0, 0xe4,
						0x1a, 0x7f, 0xff, 0xff, 0xea, 0x11, 0x01, 0x01,
						0xff, 0xff, 0xfc, 0x20, 0x08, 0x3f, 0xff, 0xff,
						0xfc, 0x0f, 0x22, 0x7f, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xb8, 0x60, 0x04, 0x87,
						0x02, 0xc8, 0x18, 0x38, 0x60, 0x04, 0x87, 0x03,
						0x00, 0x35, 0xa8, 0x16, 0x40, 0x9b, 0x04, 0xd0,
						0x11, 0x00, 0x24, 0x38, 0x11, 0x01, 0x6c, 0x16,
						0x41, 0x60, 0x2c, 0x82, 0xd8, 0x2c, 0x05, 0x90,
						0x5b, 0x05, 0x80, 0xb2, 0x0b, 0x60, 0xb0, 0x16,
						0x41, 0x6c, 0x16, 0x02, 0xc8, 0x2d, 0x82, 0xc0,
						0x59, 0x05, 0xb0, 0x58,
					},
				},
				10800,
				10800,
			},
			{
				[][]byte{
					{
						0x41, 0xe0, 0x00, 0xa5, 0x13, 0x00, 0xce, 0xf0,
						0x2c, 0x70, 0x20, 0x01, 0x43, 0xc0, 0x8b, 0xc3,
						0x01, 0x99, 0x60, 0x80, 0x04, 0x07, 0x06, 0x39,
						0xe0, 0x80, 0x04, 0x04, 0x37, 0x80, 0x90, 0xe4,
						0x06, 0x9c, 0xa0, 0x23, 0x60, 0x06, 0x25, 0x80,
					},
				},
				14400,
				14400,
			},
		},
	},
	{
		"issue mediamtx/3614 Only SEI received",
		[]sample{
			{
				[][]byte{
					{ // SPS
						0x27, 0x64, 0x00, 0x2a, 0xac, 0x2d, 0x90, 0x07,
						0x80, 0x22, 0x7e, 0x5c, 0x05, 0xa8, 0x08, 0x08,
						0x0a, 0x00, 0x00, 0x03, 0x00, 0x02, 0x00, 0x00,
						0x03, 0x00, 0xf1, 0xd0, 0x80, 0x04, 0xc4, 0x80,
						0x00, 0x09, 0x89, 0x68, 0xde, 0xf7, 0xc1, 0xda,
						0x1c, 0x31, 0x92,
					},
					{0x68, 0xca, 0x41, 0xf2},                                     // PPS
					{0x6, 0x0, 0x6, 0x85, 0x7e, 0x40, 0x0, 0x0, 0x10, 0x1},       // SEI
					{0x65, 0x88, 0x82, 0x80, 0x1f, 0xff, 0xfb, 0xf0, 0xa2, 0x88}, // IDR
					{0x6, 0x1, 0x2, 0x4, 0x24, 0x80},                             // SEI
					{0x41, 0x9a, 0xc, 0x1c, 0x2f, 0xe4, 0xed, 0x23, 0xb5, 0x63},  // non-IDR
				},
				0,
				0,
			},
			{
				[][]byte{
					{0x6, 0x1, 0x2, 0x8, 0x14, 0x80}, // SEI
				},
				0,
				0,
			},
			{
				[][]byte{
					{0x61, 0x00, 0xf0, 0xe0, 0x00, 0x40, 0x00, 0xbe, 0x47, 0x9b}, // non-IDR
				},
				3600,
				3600,
			},
			{
				[][]byte{
					{0x6, 0x1, 0x2, 0x8, 0x14, 0x80}, // SEI
				},
				3600,
				7200,
			},
		},
	},
	{
		"issue mediamtx/4892 (poc increment = 1 after reordered frames)",
		[]sample{
			{
				[][]byte{
					{
						0x27, 0x64, 0x00, 0x28, 0xac, 0x56, 0x50, 0x1e,
						0x00, 0x89, 0xf9, 0x66, 0xa0, 0x20, 0x20, 0x20,
						0x40,
					},
					{
						0x28, 0xee, 0x3c, 0xb0,
					},
					{
						0x65, 0xb8, 0x20, 0x1f, 0xde, 0x08, 0xe5, 0x4c,
						0xff, 0x82, 0xcc, 0x1e, 0x9b, 0x50, 0xdb, 0xb3,
						0x15, 0xf2, 0xac, 0x66,
					},
					{
						0x0c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff,
					},
				},
				0,
				0,
			},
			{
				[][]byte{
					{
						0x41, 0xe1, 0x10, 0x7f, 0xcd, 0xf4, 0xe3, 0x3d,
						0x20, 0x01, 0x62, 0x49, 0x60, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00,
					},
					{
						0x0c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff,
					},
				},
				6000,
				12000,
			},
			{
				[][]byte{
					{
						0x41, 0xa8, 0x82, 0x87, 0xff, 0xee, 0x4d, 0x5c,
						0x1a, 0x30, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00,
					},
					{
						0x0c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff,
					},
				},
				6090,
				6000,
			},
			{
				[][]byte{
					{
						0x01, 0xa8, 0xc1, 0x88, 0x8f, 0xeb, 0xea, 0x6b,
						0x80, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03, 0x00,
						0x00, 0x03, 0x00, 0x00,
					},
					{
						0x0c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff,
					},
				},
				6180,
				3000,
			},
			{
				[][]byte{
					{
						0x01, 0xa8, 0xc3, 0x88, 0x8f, 0xf5, 0x4b, 0xa1,
						0xc0, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03, 0x00,
						0x00, 0x03, 0x00, 0x00,
					},
					{
						0x0c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff,
					},
				},
				7590,
				9000,
			},
			{
				[][]byte{
					{
						0x41, 0xe3, 0x21, 0xa2, 0x3f, 0xcd, 0x95, 0x8a,
						0xc0, 0x19, 0xa0, 0x00, 0x00, 0x03, 0x00, 0x00,
						0x03, 0x00, 0x00, 0x03,
					},
					{
						0x0c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff,
					},
				},
				10325,
				24000,
			},
		},
	},
}

func TestDTSExtractor(t *testing.T) {
	for _, ca := range casesDTSExtractor {
		t.Run(ca.name, func(t *testing.T) {
			ex := NewDTSExtractor()
			for _, sample := range ca.sequence {
				dts, err := ex.Extract(sample.au, sample.pts)
				require.NoError(t, err)
				require.Equal(t, sample.dts, dts)
			}
		})
	}
}

func serializeSequence(seq []sample) []byte {
	var buf []byte

	for _, sample := range seq {
		tmp := make([]byte, 8)
		binary.LittleEndian.PutUint64(tmp, uint64(sample.pts))
		buf = append(buf, tmp...)

		au, _ := AnnexB(sample.au).Marshal()

		tmp = make([]byte, 4)
		binary.LittleEndian.PutUint32(tmp, uint32(len(au)))
		buf = append(buf, tmp...)

		buf = append(buf, au...)
	}

	return buf
}

func unserializeSequence(buf []byte) ([]sample, error) {
	var samples []sample

	for {
		if len(buf) < 8 {
			return nil, fmt.Errorf("not enough bits")
		}
		pts := int64(binary.LittleEndian.Uint64(buf[:8]))
		buf = buf[8:]

		if len(buf) < 4 {
			return nil, fmt.Errorf("not enough bits")
		}
		auLen := binary.LittleEndian.Uint32(buf[:4])
		buf = buf[4:]

		if auLen == 0 {
			return nil, fmt.Errorf("invalid AU len")
		}
		if len(buf) < int(auLen) {
			return nil, fmt.Errorf("not enough bits")
		}
		rawAu := buf[:auLen]
		buf = buf[auLen:]

		var au AnnexB
		err := au.Unmarshal(rawAu)
		if err != nil {
			return nil, fmt.Errorf("not enough bits")
		}

		samples = append(samples, sample{
			au:  au,
			pts: pts,
		})

		if len(buf) == 0 {
			break
		}
	}

	return samples, nil
}

func FuzzDTSExtractor(f *testing.F) {
	for _, ca := range casesDTSExtractor {
		f.Add(serializeSequence(ca.sequence))
	}

	f.Fuzz(func(t *testing.T, buf []byte) {
		seq, err := unserializeSequence(buf)
		if err != nil {
			t.Skip()
			return
		}

		ex := NewDTSExtractor()
		for _, sample := range seq {
			_, err = ex.Extract(sample.au, sample.pts)
			if err != nil {
				break
			}
		}
	})
}
