package message

import (
	"bytes"
	"testing"
	"time"

	"github.com/abema/go-mp4"
	"github.com/stretchr/testify/require"

	"github.com/bluenviron/gortmplib/pkg/amf0"
	"github.com/bluenviron/gortmplib/pkg/bytecounter"
	"github.com/bluenviron/mediacommon/v2/pkg/codecs/mpeg4audio"
)

var readWriterCases = []struct {
	name string
	dec  Message
	enc  []byte
}{
	{
		"acknowledge",
		&Acknowledge{
			Value: 45953968,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x02, 0xbd, 0x33, 0xb0,
		},
	},
	{
		"audio mpeg1",
		&Audio{
			ChunkStreamID:   7,
			DTS:             6013806 * time.Millisecond,
			MessageStreamID: 4534543,
			Codec:           CodecMPEG1Audio,
			Rate:            Rate44100,
			Depth:           Depth16,
			IsStereo:        true,
			Payload:         []byte{0x01, 0x02, 0x03, 0x04},
		},
		[]byte{
			0x07, 0x5b, 0xc3, 0x6e, 0x00, 0x00, 0x05, 0x08,
			0x00, 0x45, 0x31, 0x0f, 0x2f, 0x01, 0x02, 0x03,
			0x04,
		},
	},
	{
		"audio mpeg4",
		&Audio{
			ChunkStreamID:   7,
			DTS:             6013806 * time.Millisecond,
			MessageStreamID: 4534543,
			Codec:           CodecMPEG4Audio,
			Rate:            Rate44100,
			Depth:           Depth16,
			IsStereo:        true,
			AACType:         AudioAACTypeAU,
			Payload:         []byte{0x5A, 0xC0, 0x77, 0x40},
		},
		[]byte{
			0x07, 0x5b, 0xc3, 0x6e, 0x00, 0x00, 0x06, 0x08,
			0x00, 0x45, 0x31, 0x0f, 0xaf, 0x01, 0x5a, 0xc0,
			0x77, 0x40,
		},
	},
	{
		"audio ex sequence start opus",
		&AudioExSequenceStart{
			ChunkStreamID:   0x4,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCOpus,
			OpusHeader: &OpusIDHeader{
				Version:             0x1,
				ChannelCount:        0x2,
				PreSkip:             0x3801,
				InputSampleRate:     0xc05d0000,
				ChannelMappingTable: []uint8{},
			},
		},
		[]byte{
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x08,
			0x01, 0x00, 0x00, 0x00, 0x90, 0x4f, 0x70, 0x75,
			0x73, 0x4f, 0x70, 0x75, 0x73, 0x48, 0x65, 0x61,
			0x64, 0x01, 0x02, 0x38, 0x01, 0xc0, 0x5d, 0x00,
			0x00, 0x00, 0x00, 0x00,
		},
	},
	{
		"audio ex sequence start aac",
		&AudioExSequenceStart{
			ChunkStreamID:   0x4,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCMP4A,
			AACHeader: &mpeg4audio.AudioSpecificConfig{
				Type:         mpeg4audio.ObjectTypeAACLC,
				SampleRate:   48000,
				ChannelCount: 2,
			},
		},
		[]byte{
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x08,
			0x01, 0x00, 0x00, 0x00, 0x90, 0x6d, 0x70, 0x34,
			0x61, 0x11, 0x90,
		},
	},
	{
		"audio ex sequence start ac3",
		&AudioExSequenceStart{
			ChunkStreamID:   0x4,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCAC3,
		},
		[]byte{
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x08,
			0x01, 0x00, 0x00, 0x00, 0x90, 0x61, 0x63, 0x2d,
			0x33,
		},
	},
	{
		"audio ex sequence end",
		&AudioExSequenceEnd{
			ChunkStreamID:   4,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCOpus,
		},
		[]byte{
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x08,
			0x01, 0x00, 0x00, 0x00, 0x92, 0x4f, 0x70, 0x75,
			0x73,
		},
	},
	{
		"audio ex coded frames",
		&AudioExCodedFrames{
			ChunkStreamID:   4,
			DTS:             15100 * time.Millisecond,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCOpus,
			Payload:         []byte{1, 2, 3},
		},
		[]byte{
			0x04, 0x00, 0x3a, 0xfc, 0x00, 0x00, 0x08, 0x08,
			0x01, 0x00, 0x00, 0x00, 0x91, 0x4f, 0x70, 0x75,
			0x73, 0x01, 0x02, 0x03,
		},
	},
	{
		"audio ex multichannel config",
		&AudioExMultichannelConfig{
			ChunkStreamID:       0x4,
			MessageStreamID:     0x1000000,
			FourCC:              0x4f707573,
			AudioChannelOrder:   0x1,
			ChannelCount:        0x2,
			AudioChannelMapping: 0x0,
			AudioChannelFlags:   0x3,
		},
		[]byte{
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x08,
			0x01, 0x00, 0x00, 0x00, 0x94, 0x4f, 0x70, 0x75,
			0x73, 0x01, 0x02, 0x00, 0x00, 0x00, 0x03,
		},
	},
	{
		"audio ex multitrack",
		&AudioExMultitrack{
			MultitrackType: AudioExMultitrackTypeOneTrack,
			TrackID:        1,
			Wrapped: &AudioExSequenceStart{
				ChunkStreamID:   0x4,
				MessageStreamID: 0x1000000,
				FourCC:          FourCCAC3,
			},
		},
		[]byte{
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x08,
			0x01, 0x00, 0x00, 0x00, 0x95, 0x00, 0x61, 0x63,
			0x2d, 0x33, 0x01,
		},
	},
	{
		"command amf0",
		&CommandAMF0{
			ChunkStreamID:   3,
			MessageStreamID: 345243,
			Name:            "i8yythrergre",
			CommandID:       56456,
			Arguments: []interface{}{
				amf0.Object{
					{Key: "k1", Value: "v1"},
					{Key: "k2", Value: "v2"},
				},
				nil,
			},
		},
		[]byte{
			0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2f, 0x14,
			0x00, 0x05, 0x44, 0x9b, 0x02, 0x00, 0x0c, 0x69,
			0x38, 0x79, 0x79, 0x74, 0x68, 0x72, 0x65, 0x72,
			0x67, 0x72, 0x65, 0x00, 0x40, 0xeb, 0x91, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x02, 0x6b,
			0x31, 0x02, 0x00, 0x02, 0x76, 0x31, 0x00, 0x02,
			0x6b, 0x32, 0x02, 0x00, 0x02, 0x76, 0x32, 0x00,
			0x00, 0x09, 0x05,
		},
	},
	{
		"data amf0",
		&DataAMF0{
			ChunkStreamID:   3,
			MessageStreamID: 345243,
			Payload: []interface{}{
				float64(234),
				"string",
				nil,
			},
		},
		[]byte{
			0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x12,
			0x00, 0x05, 0x44, 0x9b, 0x00, 0x40, 0x6d, 0x40,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x06,
			0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 0x05,
		},
	},
	{
		"set chunk size",
		&SetChunkSize{
			Value: 10000,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x27, 0x10,
		},
	},
	{
		"set peer bandwidth",
		&SetChunkSize{
			Value: 10000,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x27, 0x10,
		},
	},
	{
		"set window ack size",
		&SetChunkSize{
			Value: 10000,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x27, 0x10,
		},
	},
	{
		"user control ping request",
		&UserControlPingRequest{
			ServerTime: 569834435,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x21, 0xf6,
			0xfb, 0xc3,
		},
	},
	{
		"user control ping response",
		&UserControlPingResponse{
			ServerTime: 569834435,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x21, 0xf6,
			0xfb, 0xc3,
		},
	},
	{
		"user control set buffer length",
		&UserControlSetBufferLength{
			StreamID:     35534,
			BufferLength: 235345,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x04,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
			0x8a, 0xce, 0x00, 0x03, 0x97, 0x51,
		},
	},
	{
		"user control stream begin",
		&UserControlStreamBegin{
			StreamID: 35534,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x8a, 0xce,
		},
	},
	{
		"user control stream dry",
		&UserControlStreamDry{
			StreamID: 35534,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
			0x8a, 0xce,
		},
	},
	{
		"user control stream eof",
		&UserControlStreamEOF{
			StreamID: 35534,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x8a, 0xce,
		},
	},
	{
		"user control stream is recorded",
		&UserControlStreamIsRecorded{
			StreamID: 35534,
		},
		[]byte{
			0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
			0x8a, 0xce,
		},
	},
	{
		"video",
		&Video{
			ChunkStreamID:   6,
			DTS:             2543534 * time.Millisecond,
			MessageStreamID: 0x1000000,
			Codec:           CodecH264,
			IsKeyFrame:      true,
			Type:            VideoTypeConfig,
			PTSDelta:        10 * time.Millisecond,
			Payload:         []byte{0x01, 0x02, 0x03},
		},
		[]byte{
			0x06, 0x26, 0xcf, 0xae, 0x00, 0x00, 0x08, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00,
			0x0a, 0x01, 0x02, 0x03,
		},
	},
	{
		"video ex sequence start av1",
		&VideoExSequenceStart{
			ChunkStreamID:   6,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCAV1,
			AV1Header: &mp4.Av1C{
				Marker:             0x1,
				Version:            0x1,
				SeqLevelIdx0:       0x8,
				ChromaSubsamplingX: 0x1,
				ChromaSubsamplingY: 0x1,
				ConfigOBUs:         []uint8{0xa, 0xb, 0x0, 0x0, 0x0, 0x42, 0xab, 0xbf, 0xc3, 0x70, 0xb, 0xe0, 0x1},
			},
		},
		[]byte{
			0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x80, 0x61, 0x76, 0x30,
			0x31, 0x81, 0x08, 0x0c, 0x00, 0x0a, 0x0b, 0x00,
			0x00, 0x00, 0x42, 0xab, 0xbf, 0xc3, 0x70, 0x0b,
			0xe0, 0x01,
		},
	},
	{
		"video ex sequence start hevc",
		&VideoExSequenceStart{
			ChunkStreamID:   4,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCHEVC,
			HEVCHeader: &mp4.HvcC{
				ConfigurationVersion: 0x1,
				GeneralProfileIdc:    0x1,
				GeneralProfileCompatibility: [32]bool{
					false, true, false, false, false, false, false, false, false,
					false, false, false, false, false, false, false, false, false,
					false, false, false, false, false, false, false, false, false,
					false, false, false, false, false,
				},
				GeneralConstraintIndicator: [6]uint8{0x90, 0x0, 0x0, 0x0, 0x0, 0x0},
				GeneralLevelIdc:            0x7b,
				Reserved1:                  0xf,
				Reserved2:                  0x3f,
				Reserved3:                  0x3f,
				ChromaFormatIdc:            0x1,
				Reserved4:                  0x1f,
				Reserved5:                  0x1f,
				TemporalIdNested:           0x3,
				LengthSizeMinusOne:         0x3,
				NumOfNaluArrays:            0x3,
				NaluArrays: []mp4.HEVCNaluArray{
					{
						Completeness: true,
						NaluType:     0x20,
						NumNalus:     0x1,
						Nalus: []mp4.HEVCNalu{{
							Length: 0x17,
							NALUnit: []uint8{
								0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x01, 0x40,
								0x00, 0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03,
								0x00, 0x00, 0x03, 0x00, 0x7b, 0xac, 0x09,
							},
						}},
					},
					{
						Completeness: true,
						NaluType:     0x21,
						NumNalus:     0x1,
						Nalus: []mp4.HEVCNalu{{
							Length: 0x3d,
							NALUnit: []uint8{
								0x42, 0x01, 0x01, 0x01, 0x40, 0x00, 0x00, 0x03,
								0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
								0x00, 0x7b, 0xa0, 0x03, 0xc0, 0x80, 0x11, 0x07,
								0xcb, 0x96, 0xb4, 0xa4, 0x25, 0x92, 0xe3, 0x01,
								0x6a, 0x02, 0x02, 0x02, 0x08, 0x00, 0x00, 0x03,
								0x00, 0x08, 0x00, 0x00, 0x03, 0x01, 0xe3, 0x00,
								0x2e, 0xf2, 0x88, 0x00, 0x02, 0x62, 0x5a, 0x00,
								0x00, 0x13, 0x12, 0xd0, 0x20,
							},
						}},
					},
					{
						Completeness: true,
						NaluType:     0x22,
						NumNalus:     0x1,
						Nalus: []mp4.HEVCNalu{{
							Length: 0x7,
							NALUnit: []uint8{
								0x44, 0x01, 0xc0, 0xf7, 0xc0, 0xcc, 0x90,
							},
						}},
					},
				},
			},
		},
		[]byte{
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x80, 0x68, 0x76, 0x63,
			0x31, 0x01, 0x01, 0x40, 0x00, 0x00, 0x00, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x7b, 0xf0, 0x00,
			0xfc, 0xfd, 0xf8, 0xf8, 0x00, 0x00, 0x0f, 0x03,
			0xa0, 0x00, 0x01, 0x00, 0x17, 0x40, 0x01, 0x0c,
			0x01, 0xff, 0xff, 0x01, 0x40, 0x00, 0x00, 0x03,
			0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
			0x00, 0x7b, 0xac, 0x09, 0xa1, 0x00, 0x01, 0x00,
			0x3d, 0x42, 0x01, 0x01, 0x01, 0x40, 0x00, 0x00,
			0x03, 0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00,
			0x03, 0x00, 0x7b, 0xa0, 0x03, 0xc0, 0x80, 0x11,
			0x07, 0xcb, 0x96, 0xb4, 0xa4, 0x25, 0x92, 0xe3,
			0x01, 0x6a, 0x02, 0x02, 0x02, 0x08, 0x00, 0x00,
			0x03, 0x00, 0x08, 0x00, 0x00, 0x03, 0x01, 0xe3,
			0x00, 0x2e, 0xf2, 0x88, 0x00, 0x02, 0x62, 0x5a,
			0x00, 0x00, 0x13, 0x12, 0xd0, 0x20, 0xa2, 0x00,
			0x01, 0x00, 0x07, 0x44, 0xc4, 0x01, 0xc0, 0xf7,
			0xc0, 0xcc, 0x90,
		},
	},
	{
		"video ex sequence start vp9",
		&VideoExSequenceStart{
			ChunkStreamID:   6,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCVP9,
			VP9Header: &mp4.VpcC{
				FullBox:                 mp4.FullBox{Version: 0x1},
				Level:                   0x28,
				BitDepth:                0x8,
				ChromaSubsampling:       0x1,
				ColourPrimaries:         0x2,
				TransferCharacteristics: 0x2,
				MatrixCoefficients:      0x2,
				CodecInitializationData: []uint8{},
			},
		},
		[]byte{
			0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x80, 0x76, 0x70, 0x30,
			0x39, 0x01, 0x00, 0x00, 0x00, 0x00, 0x28, 0x82,
			0x02, 0x02, 0x02, 0x00, 0x00,
		},
	},
	{
		"video ex sequence start h264",
		&VideoExSequenceStart{
			ChunkStreamID:   0x4,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCAVC,
			AVCHeader: &mp4.AVCDecoderConfiguration{
				AnyTypeBox:                 mp4.AnyTypeBox{Type: mp4.BoxType{0x61, 0x76, 0x63, 0x43}},
				ConfigurationVersion:       0x1,
				Profile:                    0x4d,
				ProfileCompatibility:       0x40,
				Level:                      0x1e,
				Reserved:                   0x3f,
				LengthSizeMinusOne:         0x3,
				Reserved2:                  0x7,
				NumOfSequenceParameterSets: 0x1,
				SequenceParameterSets: []mp4.AVCParameterSet{
					{
						Length: 0x23,
						NALUnit: []uint8{
							0x67, 0x4d, 0x40, 0x1e, 0x96, 0x56, 0x05, 0x01,
							0x7f, 0xcb, 0x80, 0xb5, 0x01, 0x01, 0x01, 0x40,
							0x00, 0x00, 0xfa, 0x00, 0x00, 0x3a, 0x98, 0x38,
							0x00, 0x00, 0x7a, 0x10, 0x00, 0x0f, 0x42, 0x5b,
							0xbc, 0xb8, 0x28,
						},
					},
				},
				NumOfPictureParameterSets: 0x1,
				PictureParameterSets: []mp4.AVCParameterSet{
					{
						Length:  0x4,
						NALUnit: []uint8{0x68, 0xee, 0x3c, 0x80},
					},
				},
			},
		},
		[]byte{
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x37, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x80, 0x61, 0x76, 0x63,
			0x31, 0x01, 0x4d, 0x40, 0x1e, 0xff, 0xe1, 0x00,
			0x23, 0x67, 0x4d, 0x40, 0x1e, 0x96, 0x56, 0x05,
			0x01, 0x7f, 0xcb, 0x80, 0xb5, 0x01, 0x01, 0x01,
			0x40, 0x00, 0x00, 0xfa, 0x00, 0x00, 0x3a, 0x98,
			0x38, 0x00, 0x00, 0x7a, 0x10, 0x00, 0x0f, 0x42,
			0x5b, 0xbc, 0xb8, 0x28, 0x01, 0x00, 0x04, 0x68,
			0xee, 0x3c, 0x80,
		},
	},
	{
		"video ex sequence end",
		&VideoExSequenceEnd{
			ChunkStreamID:   4,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCAV1,
		},
		[]byte{
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x82, 0x61, 0x76, 0x30,
			0x31,
		},
	},
	{
		"video ex coded frames",
		&VideoExCodedFrames{
			ChunkStreamID:   4,
			DTS:             15100 * time.Millisecond,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCHEVC,
			PTSDelta:        30 * time.Millisecond,
			Payload:         []byte{0x01, 0x02, 0x03},
		},
		[]byte{
			0x04, 0x00, 0x3a, 0xfc, 0x00, 0x00, 0x0b, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x81, 0x68, 0x76, 0x63,
			0x31, 0x00, 0x00, 0x1e, 0x01, 0x02, 0x03,
		},
	},
	{
		"video ex frames x",
		&VideoExFramesX{
			ChunkStreamID:   4,
			DTS:             15100 * time.Millisecond,
			MessageStreamID: 0x1000000,
			FourCC:          FourCCHEVC,
			Payload:         []byte{0x01, 0x02, 0x03},
		},
		[]byte{
			0x04, 0x00, 0x3a, 0xfc, 0x00, 0x00, 0x08, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x83, 0x68, 0x76, 0x63,
			0x31, 0x01, 0x02, 0x03,
		},
	},
	{
		"video ex metadata",
		&VideoExMetadata{
			ChunkStreamID:   0x6,
			DTS:             0,
			MessageStreamID: 0x1000000,
			FourCC:          0x68766331,
			Payload:         []interface{}{"colorInfo", amf0.Object{{Key: "colorConfig", Value: amf0.Object{}}}},
		},
		[]byte{
			0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x26, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x84, 0x68, 0x76, 0x63,
			0x31, 0x02, 0x00, 0x09, 0x63, 0x6f, 0x6c, 0x6f,
			0x72, 0x49, 0x6e, 0x66, 0x6f, 0x03, 0x00, 0x0b,
			0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x43, 0x6f, 0x6e,
			0x66, 0x69, 0x67, 0x03, 0x00, 0x00, 0x09, 0x00,
			0x00, 0x09,
		},
	},
	{
		"video ex multitrack",
		&VideoExMultitrack{
			MultitrackType: VideoExMultitrackTypeOneTrack,
			TrackID:        1,
			Wrapped: &VideoExSequenceStart{
				ChunkStreamID:   6,
				MessageStreamID: 0x1000000,
				FourCC:          FourCCVP9,
				VP9Header: &mp4.VpcC{
					FullBox:                 mp4.FullBox{Version: 0x1},
					Level:                   0x28,
					BitDepth:                0x8,
					ChromaSubsampling:       0x1,
					ColourPrimaries:         0x2,
					TransferCharacteristics: 0x2,
					MatrixCoefficients:      0x2,
					CodecInitializationData: []uint8{},
				},
			},
		},
		[]byte{
			0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x09,
			0x01, 0x00, 0x00, 0x00, 0x86, 0x00, 0x76, 0x70,
			0x30, 0x39, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x28, 0x82, 0x02, 0x02, 0x02, 0x00, 0x00,
		},
	},
}

func TestReader(t *testing.T) {
	for _, ca := range readWriterCases {
		t.Run(ca.name, func(t *testing.T) {
			bc := bytecounter.NewReader(bytes.NewReader(ca.enc))
			r := NewReader(bc, bc, nil)
			dec, err := r.Read()
			require.NoError(t, err)
			require.Equal(t, ca.dec, dec)
		})
	}
}

func TestReaderNonStandardControlChunkStreamID(t *testing.T) {
	buf := []byte{
		0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
		0x8a, 0xce,
	}

	bc := bytecounter.NewReader(bytes.NewReader(buf))
	r := NewReader(bc, bc, nil)
	dec, err := r.Read()
	require.NoError(t, err)
	require.Equal(t, &UserControlStreamDry{
		StreamID: 35534,
	}, dec)
}

func FuzzReader(f *testing.F) {
	for _, ca := range readWriterCases {
		f.Add(ca.enc)
	}

	f.Fuzz(func(t *testing.T, b []byte) {
		bcr := bytecounter.NewReader(bytes.NewReader(b))
		r := NewReader(bcr, bcr, nil)

		var buf bytes.Buffer
		bcw := bytecounter.NewWriter(&buf)
		w := NewWriter(bcw, bcw, true)

		msg, err := r.Read()
		if err != nil {
			return
		}

		err = w.Write(msg)
		require.NoError(t, err)
	})
}
