package crypto

import (
	"bytes"
	"encoding/hex"
	"testing"

	"github.com/datarhei/gosrt/packet"

	"github.com/stretchr/testify/require"
)

func mustDecodeString(s string) []byte {
	b, err := hex.DecodeString(s)
	if err != nil {
		panic(err)
	}

	return b
}

func TestInvalidKeylength(t *testing.T) {
	_, err := New(42)
	require.Error(t, err, "succeeded to create crypto with invalid keylength")
}

func TestInvalidKM(t *testing.T) {
	c, err := New(16)
	require.NoError(t, err)

	km := &packet.CIFKeyMaterialExtension{}

	km.KeyBasedEncryption = packet.UnencryptedPacket
	km.Salt = mustDecodeString("6c438852715a4d26e0e810b3132ca61f")
	km.Wrap = mustDecodeString("699ab4eac6b7c66c3a9fa0d6836326c2b294a10764233356")

	err = c.UnmarshalKM(km, "foobarfoobar")
	require.ErrorIs(t, err, ErrInvalidKey)

	km = &packet.CIFKeyMaterialExtension{}

	km.KeyBasedEncryption = packet.EvenKeyEncrypted
	km.Salt = mustDecodeString("6c438852715a4d26e0e810b3132ca61f")
	km.Wrap = mustDecodeString("5b901889bd106609ca8a83264b12ed1bfab3f02812bad65784ac396b1f57eb16c53e1020d3a3250b")

	err = c.UnmarshalKM(km, "foobarfoobar")
	require.ErrorIs(t, err, ErrInvalidWrap)
}

func TestUnmarshal(t *testing.T) {
	tests := []struct {
		keylength   int
		salt        string
		passphrase  string
		evenWrap    string
		oddWrap     string
		evenOddWrap string
	}{
		{
			keylength:   16,
			salt:        "6c438852715a4d26e0e810b3132ca61f",
			passphrase:  "foobarfoobar",
			evenWrap:    "699ab4eac6b7c66c3a9fa0d6836326c2b294a10764233356",
			oddWrap:     "ca4decaaf8d7b5c38288e84c8796929c84b7c139f1f769d5",
			evenOddWrap: "5b901889bd106609ca8a83264b12ed1bfab3f02812bad65784ac396b1f57eb16c53e1020d3a3250b",
		},
		{
			keylength:   24,
			salt:        "e636259ccc41e73611b9363bb58586b1",
			passphrase:  "foobarfoobar",
			evenWrap:    "8c6502d6a83e0ab894a43cb5b37b71c2755afc64a682bed9d46912138b60f384",
			oddWrap:     "1fe56a4475759636674a7c5e44f1cdfb365a9f11d8fe74536e8df6b97eecf1c9",
			evenOddWrap: "7360357d363ebec384885b10c8120528889d1be05624bfc381c5fa090f00f9ecef5d6427f7542a58be144f4aeb07452beca546874a68197d",
		},
		{
			keylength:   32,
			salt:        "3825bb4163f7d5cf2804ec0b31a7370f",
			passphrase:  "foobarfoobar",
			evenWrap:    "7d1578458e41680dd997d1a185c75753f3344c6711542b35833f881f7c480304cbe9bdbe76035914",
			oddWrap:     "cc1af097af558fa25b925417c4e6e9e1adacd8b96916b4ac4fac8e6ecdc3b5c48c01134e92e9e5f6",
			evenOddWrap: "f7373def4e9f61f6cd6a22e78916aa07cac8e5f07669d556ec8a15b7631fa9c631e9d98a3f92dbe187f434569ec71b9e2a53171feafd909a5560233fe02ed0301e576d4992b10c86",
		},
	}

	for _, test := range tests {
		c, err := New(test.keylength)
		require.NoError(t, err)

		km := &packet.CIFKeyMaterialExtension{}

		km.KeyBasedEncryption = packet.EvenKeyEncrypted
		km.Salt = mustDecodeString(test.salt)
		km.Wrap = mustDecodeString(test.evenWrap)

		err = c.UnmarshalKM(km, test.passphrase)
		require.NoError(t, err)

		km.KeyBasedEncryption = packet.OddKeyEncrypted
		km.Salt = mustDecodeString(test.salt)
		km.Wrap = mustDecodeString(test.oddWrap)

		err = c.UnmarshalKM(km, test.passphrase)
		require.NoError(t, err)

		km.KeyBasedEncryption = packet.EvenAndOddKey
		km.Salt = mustDecodeString(test.salt)
		km.Wrap = mustDecodeString(test.evenOddWrap)

		err = c.UnmarshalKM(km, test.passphrase)
		require.NoError(t, err)
	}
}

func TestMarshal(t *testing.T) {
	tests := []struct {
		keylength   int
		salt        string
		evenSEK     string
		oddSEK      string
		passphrase  string
		evenWrap    string
		oddWrap     string
		evenOddWrap string
	}{
		{
			keylength:   16,
			salt:        "6c438852715a4d26e0e810b3132ca61f",
			evenSEK:     "047dc22e7f000be55a25ba56ae2e9180",
			oddSEK:      "240c8e76ccf3637641af473edaf15aaf",
			passphrase:  "foobarfoobar",
			evenWrap:    "699ab4eac6b7c66c3a9fa0d6836326c2b294a10764233356",
			oddWrap:     "ca4decaaf8d7b5c38288e84c8796929c84b7c139f1f769d5",
			evenOddWrap: "5b901889bd106609ca8a83264b12ed1bfab3f02812bad65784ac396b1f57eb16c53e1020d3a3250b",
		},
		{
			keylength:   24,
			salt:        "e636259ccc41e73611b9363bb58586b1",
			evenSEK:     "4dca0ad088da64fdc8e98002d141bc46fed4fa0167b931c8",
			oddSEK:      "2b2bbb64ee3942cfa31bfe58efd1d2102c40b7bc028f8946",
			passphrase:  "foobarfoobar",
			evenWrap:    "8c6502d6a83e0ab894a43cb5b37b71c2755afc64a682bed9d46912138b60f384",
			oddWrap:     "1fe56a4475759636674a7c5e44f1cdfb365a9f11d8fe74536e8df6b97eecf1c9",
			evenOddWrap: "7360357d363ebec384885b10c8120528889d1be05624bfc381c5fa090f00f9ecef5d6427f7542a58be144f4aeb07452beca546874a68197d",
		},
		{
			keylength:   32,
			salt:        "3825bb4163f7d5cf2804ec0b31a7370f",
			evenSEK:     "53a088d93431181075f8a9bc4876359afe48967308120c93f97bbd823d8de62a",
			oddSEK:      "7893e88b6296ffcc5a2eab5f53d48efd7adaeced8cb3a851d4f8e2dbda8db17a",
			passphrase:  "foobarfoobar",
			evenWrap:    "7d1578458e41680dd997d1a185c75753f3344c6711542b35833f881f7c480304cbe9bdbe76035914",
			oddWrap:     "cc1af097af558fa25b925417c4e6e9e1adacd8b96916b4ac4fac8e6ecdc3b5c48c01134e92e9e5f6",
			evenOddWrap: "f7373def4e9f61f6cd6a22e78916aa07cac8e5f07669d556ec8a15b7631fa9c631e9d98a3f92dbe187f434569ec71b9e2a53171feafd909a5560233fe02ed0301e576d4992b10c86",
		},
	}

	for _, test := range tests {
		c, err := New(test.keylength)
		require.NoError(t, err)

		cr := c.(*crypto)

		cr.salt = mustDecodeString(test.salt)
		cr.evenSEK = mustDecodeString(test.evenSEK)
		cr.oddSEK = mustDecodeString(test.oddSEK)

		km := &packet.CIFKeyMaterialExtension{}

		err = c.MarshalKM(km, test.passphrase, packet.EvenKeyEncrypted)
		require.NoError(t, err, "keylength: %d", test.keylength)

		wrap := mustDecodeString(test.evenWrap)

		x := bytes.Compare(km.Wrap, wrap)
		require.Equal(t, 0, x, "keylength: %d", test.keylength)

		km = &packet.CIFKeyMaterialExtension{}

		err = c.MarshalKM(km, test.passphrase, packet.OddKeyEncrypted)
		require.NoError(t, err, "keylength: %d", test.keylength)

		wrap = mustDecodeString(test.oddWrap)

		x = bytes.Compare(km.Wrap, wrap)
		require.Equal(t, 0, x, "keylength: %d", test.keylength)

		km = &packet.CIFKeyMaterialExtension{}

		err = c.MarshalKM(km, test.passphrase, packet.EvenAndOddKey)
		require.NoError(t, err, "keylength: %d", test.keylength)

		wrap = mustDecodeString(test.evenOddWrap)

		x = bytes.Compare(km.Wrap, wrap)
		require.Equal(t, 0, x, "keylength: %d", test.keylength)
	}
}

func TestDecode(t *testing.T) {
	packetSequenceNumber := uint32(0x79ee189e)
	data, _ := hex.DecodeString("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")
	tests := []struct {
		keylength     int
		salt          string
		evenSEK       string
		oddSEK        string
		passphrase    string
		evenEncrypted string
		oddEncrypted  string
	}{
		{
			keylength:     16,
			salt:          "6c438852715a4d26e0e810b3132ca61f",
			evenSEK:       "047dc22e7f000be55a25ba56ae2e9180",
			oddSEK:        "240c8e76ccf3637641af473edaf15aaf",
			passphrase:    "foobarfoobar",
			evenEncrypted: "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",
			oddEncrypted:  "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",
		},
		{
			keylength:     24,
			salt:          "e636259ccc41e73611b9363bb58586b1",
			evenSEK:       "4dca0ad088da64fdc8e98002d141bc46fed4fa0167b931c8",
			oddSEK:        "2b2bbb64ee3942cfa31bfe58efd1d2102c40b7bc028f8946",
			passphrase:    "foobarfoobar",
			evenEncrypted: "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",
			oddEncrypted:  "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",
		},
		{
			keylength:     32,
			salt:          "3825bb4163f7d5cf2804ec0b31a7370f",
			evenSEK:       "53a088d93431181075f8a9bc4876359afe48967308120c93f97bbd823d8de62a",
			oddSEK:        "7893e88b6296ffcc5a2eab5f53d48efd7adaeced8cb3a851d4f8e2dbda8db17a",
			passphrase:    "foobarfoobar",
			evenEncrypted: "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",
			oddEncrypted:  "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",
		},
	}

	for _, test := range tests {
		c, err := New(test.keylength)
		require.NoError(t, err)

		cr := c.(*crypto)

		cr.salt = mustDecodeString(test.salt)
		cr.evenSEK = mustDecodeString(test.evenSEK)
		cr.oddSEK = mustDecodeString(test.oddSEK)

		encrypted := mustDecodeString(test.evenEncrypted)

		err = c.EncryptOrDecryptPayload(encrypted, packet.EvenKeyEncrypted, packetSequenceNumber)
		require.NoError(t, err, "keylength: %d", test.keylength)

		x := bytes.Compare(data, encrypted)
		require.Equal(t, 0, x, "keylength: %d", test.keylength)

		encrypted = mustDecodeString(test.oddEncrypted)

		err = c.EncryptOrDecryptPayload(encrypted, packet.OddKeyEncrypted, packetSequenceNumber)
		require.NoError(t, err, "keylength: %d", test.keylength)

		x = bytes.Compare(data, encrypted)
		require.Equal(t, 0, x, "keylength: %d", test.keylength)
	}
}

func TestEncode(t *testing.T) {
	packetSequenceNumber := uint32(0x79ee189e)
	originalData := "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"
	tests := []struct {
		keylength     int
		salt          string
		evenSEK       string
		oddSEK        string
		passphrase    string
		evenEncrypted string
		oddEncrypted  string
	}{
		{
			keylength:     16,
			salt:          "6c438852715a4d26e0e810b3132ca61f",
			evenSEK:       "047dc22e7f000be55a25ba56ae2e9180",
			oddSEK:        "240c8e76ccf3637641af473edaf15aaf",
			passphrase:    "foobarfoobar",
			evenEncrypted: "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",
			oddEncrypted:  "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",
		},
		{
			keylength:     24,
			salt:          "e636259ccc41e73611b9363bb58586b1",
			evenSEK:       "4dca0ad088da64fdc8e98002d141bc46fed4fa0167b931c8",
			oddSEK:        "2b2bbb64ee3942cfa31bfe58efd1d2102c40b7bc028f8946",
			passphrase:    "foobarfoobar",
			evenEncrypted: "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",
			oddEncrypted:  "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",
		},
		{
			keylength:     32,
			salt:          "3825bb4163f7d5cf2804ec0b31a7370f",
			evenSEK:       "53a088d93431181075f8a9bc4876359afe48967308120c93f97bbd823d8de62a",
			oddSEK:        "7893e88b6296ffcc5a2eab5f53d48efd7adaeced8cb3a851d4f8e2dbda8db17a",
			passphrase:    "foobarfoobar",
			evenEncrypted: "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",
			oddEncrypted:  "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",
		},
	}

	for _, test := range tests {
		c, err := New(test.keylength)
		require.NoError(t, err)

		cr := c.(*crypto)

		cr.salt = mustDecodeString(test.salt)
		cr.evenSEK = mustDecodeString(test.evenSEK)
		cr.oddSEK = mustDecodeString(test.oddSEK)

		data, _ := hex.DecodeString(originalData)

		c.EncryptOrDecryptPayload(data, packet.EvenKeyEncrypted, packetSequenceNumber)

		encrypted := mustDecodeString(test.evenEncrypted)

		x := bytes.Compare(data, encrypted)
		require.Equal(t, 0, x, "keylength: %d", test.keylength)

		data = mustDecodeString(originalData)

		c.EncryptOrDecryptPayload(data, packet.OddKeyEncrypted, packetSequenceNumber)

		encrypted = mustDecodeString(test.oddEncrypted)

		x = bytes.Compare(data, encrypted)
		require.Equal(t, 0, x, "keylength: %d", test.keylength)
	}
}
