package fmp4

import (
	"bytes"
	"testing"

	"github.com/stretchr/testify/require"

	"github.com/bluenviron/mediacommon/v2/pkg/codecs/mpeg4audio"
	"github.com/bluenviron/mediacommon/v2/pkg/formats/fmp4/seekablebuffer"
	"github.com/bluenviron/mediacommon/v2/pkg/formats/mp4"
)

var testSPS = []byte{
	0x67, 0x42, 0xc0, 0x28, 0xd9, 0x00, 0x78, 0x02,
	0x27, 0xe5, 0x84, 0x00, 0x00, 0x03, 0x00, 0x04,
	0x00, 0x00, 0x03, 0x00, 0xf0, 0x3c, 0x60, 0xc9,
	0x20,
}

var testVideoTrack = &mp4.CodecH264{
	SPS: testSPS,
	PPS: []byte{0x08},
}

var testAudioTrack = &mp4.CodecMPEG4Audio{
	Config: mpeg4audio.AudioSpecificConfig{
		Type:         2,
		SampleRate:   44100,
		ChannelCount: 2,
	},
}

var casesInit = []struct {
	name string
	enc  []byte
	dec  Init
}{
	{
		"av1",
		[]byte{
			0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0x74, 0x6d, 0x6f, 0x6f, 0x76,
			0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0xd8,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x07, 0x80, 0x00, 0x00, 0x03, 0x24, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x74, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x56, 0x69, 0x64, 0x65, 0x6f, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
			0x1f, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x14, 0x76, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xdf, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0x93, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x83, 0x61, 0x76, 0x30, 0x31, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x80, 0x03,
			0x24, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x18, 0xff, 0xff, 0x00, 0x00, 0x00, 0x19, 0x61,
			0x76, 0x31, 0x43, 0x81, 0x08, 0x0c, 0x00, 0x0a,
			0x0b, 0x00, 0x00, 0x00, 0x42, 0xa7, 0xbf, 0xe4,
			0x60, 0x0d, 0x00, 0x40, 0x00, 0x00, 0x00, 0x14,
			0x62, 0x74, 0x72, 0x74, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x0f, 0x42, 0x40, 0x00, 0x0f, 0x42, 0x40,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x74, 0x73,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x73, 0x63,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x73, 0x7a,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
			0x6d, 0x76, 0x65, 0x78, 0x00, 0x00, 0x00, 0x20,
			0x74, 0x72, 0x65, 0x78, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{{
				ID:        1,
				TimeScale: 90000,
				Codec: &mp4.CodecAV1{
					SequenceHeader: []byte{
						0x08, 0x00, 0x00, 0x00, 0x42, 0xa7, 0xbf, 0xe4,
						0x60, 0x0d, 0x00, 0x40,
					},
				},
			}},
		},
	},
	{
		"vp9",
		[]byte{
			0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0x6f, 0x6d, 0x6f, 0x6f, 0x76,
			0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0xd3,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x07, 0x80, 0x00, 0x00, 0x04, 0x38, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x6f, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x56, 0x69, 0x64, 0x65, 0x6f, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
			0x1a, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x14, 0x76, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xda, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0x8e, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x7e, 0x76, 0x70, 0x30, 0x39, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x80, 0x04,
			0x38, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x18, 0xff, 0xff, 0x00, 0x00, 0x00, 0x14, 0x76,
			0x70, 0x63, 0x43, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x0a, 0x82, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x14, 0x62, 0x74, 0x72, 0x74, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x0f, 0x42, 0x40, 0x00,
			0x0f, 0x42, 0x40, 0x00, 0x00, 0x00, 0x10, 0x73,
			0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73,
			0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x73,
			0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x10, 0x73, 0x74, 0x63, 0x6f, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x28, 0x6d, 0x76, 0x65, 0x78, 0x00,
			0x00, 0x00, 0x20, 0x74, 0x72, 0x65, 0x78, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{{
				ID:        1,
				TimeScale: 90000,
				Codec: &mp4.CodecVP9{
					Width:             1920,
					Height:            1080,
					Profile:           1,
					BitDepth:          8,
					ChromaSubsampling: 1,
					ColorRange:        false,
				},
			}},
		},
	},
	{
		"h265",
		[]byte{
			0x00, 0x00, 0x00, 0x20,
			'f', 't', 'y', 'p',
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0xb8,
			'm', 'o', 'o', 'v',
			0x00, 0x00, 0x00, 0x6c,
			'm', 'v', 'h', 'd',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x02, 0x1c,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x07, 0x80, 0x00, 0x00, 0x04, 0x38, 0x00, 0x00,
			0x00, 0x00, 0x01, 0xb8, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x56, 0x69, 0x64, 0x65, 0x6f, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
			0x63, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x14, 0x76, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01,
			0x23, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0xd7, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xc7, 0x68, 0x65, 0x76, 0x31, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x80, 0x04,
			0x38, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x18, 0xff, 0xff, 0x00, 0x00, 0x00, 0x5d, 0x68,
			0x76, 0x63, 0x43, 0x01, 0x01, 0x60, 0x00, 0x00,
			0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03, 0x78,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x13, 0x03, 0x20, 0x00, 0x01, 0x00, 0x04, 0x01,
			0x02, 0x03, 0x04, 0x21, 0x00, 0x01, 0x00, 0x2a,
			0x42, 0x01, 0x01, 0x01, 0x60, 0x00, 0x00, 0x03,
			0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
			0x00, 0x78, 0xa0, 0x03, 0xc0, 0x80, 0x10, 0xe5,
			0x96, 0x66, 0x69, 0x24, 0xca, 0xe0, 0x10, 0x00,
			0x00, 0x03, 0x00, 0x10, 0x00, 0x00, 0x03, 0x01,
			0xe0, 0x80, 0x22, 0x00, 0x01, 0x00, 0x01, 0x08,
			0x00, 0x00, 0x00, 0x14, 0x62, 0x74, 0x72, 0x74,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x42, 0x40,
			0x00, 0x0f, 0x42, 0x40, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14,
			0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x63, 0x6f,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x28, 0x6d, 0x76, 0x65, 0x78,
			0x00, 0x00, 0x00, 0x20, 0x74, 0x72, 0x65, 0x78,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 90000,
					Codec: &mp4.CodecH265{
						VPS: []byte{0x01, 0x02, 0x03, 0x04},
						SPS: []byte{
							0x42, 0x01, 0x01, 0x01, 0x60, 0x00, 0x00, 0x03,
							0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
							0x00, 0x78, 0xa0, 0x03, 0xc0, 0x80, 0x10, 0xe5,
							0x96, 0x66, 0x69, 0x24, 0xca, 0xe0, 0x10, 0x00,
							0x00, 0x03, 0x00, 0x10, 0x00, 0x00, 0x03, 0x01,
							0xe0, 0x80,
						},
						PPS: []byte{0x08},
					},
				},
			},
		},
	},
	{
		"h264",
		[]byte{
			0x00, 0x00, 0x00, 0x20,
			'f', 't', 'y', 'p',
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0x88,
			'm', 'o', 'o', 'v',
			0x00, 0x00, 0x00, 0x6c,
			'm', 'v', 'h', 'd',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0xec,
			't', 'r', 'a', 'k',
			0x00, 0x00, 0x00, 0x5c,
			't', 'k', 'h', 'd',
			0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x07, 0x80, 0x00, 0x00, 0x04, 0x38, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x88,
			'm', 'd', 'i', 'a',
			0x00, 0x00, 0x00, 0x20,
			'm', 'd', 'h', 'd',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d,
			'h', 'd', 'l', 'r',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x56, 0x69, 0x64, 0x65, 0x6f, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
			0x33,
			'm', 'i', 'n', 'f',
			0x00, 0x00, 0x00,
			0x14,
			'v', 'm', 'h', 'd',
			0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24,
			'd', 'i', 'n', 'f',
			0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xf3, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0xa7, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x97, 0x61, 0x76, 0x63, 0x31, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x80, 0x04,
			0x38, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x18, 0xff, 0xff, 0x00, 0x00, 0x00, 0x2d, 0x61,
			0x76, 0x63, 0x43, 0x01, 0x42, 0xc0, 0x28, 0x03,
			0x01, 0x00, 0x19, 0x67, 0x42, 0xc0, 0x28, 0xd9,
			0x00, 0x78, 0x02, 0x27, 0xe5, 0x84, 0x00, 0x00,
			0x03, 0x00, 0x04, 0x00, 0x00, 0x03, 0x00, 0xf0,
			0x3c, 0x60, 0xc9, 0x20, 0x01, 0x00, 0x01, 0x08,
			0x00, 0x00, 0x00, 0x14, 0x62, 0x74, 0x72, 0x74,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x42, 0x40,
			0x00, 0x0f, 0x42, 0x40, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14,
			0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x63, 0x6f,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x28, 0x6d, 0x76, 0x65, 0x78,
			0x00, 0x00, 0x00, 0x20, 0x74, 0x72, 0x65, 0x78,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 90000,
					Codec:     testVideoTrack,
				},
			},
		},
	},
	{
		"mpeg-4 video",
		[]byte{
			0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0xbc, 0x6d, 0x6f, 0x6f, 0x76,
			0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x02, 0x20,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x03, 0x20, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00,
			0x00, 0x00, 0x01, 0xbc, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x56, 0x69, 0x64, 0x65, 0x6f, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
			0x67, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x14, 0x76, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01,
			0x27, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0xdb, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xcb, 0x6d, 0x70, 0x34, 0x76, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x20, 0x02,
			0x58, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x18, 0xff, 0xff, 0x00, 0x00, 0x00, 0x61, 0x65,
			0x73, 0x64, 0x73, 0x00, 0x00, 0x00, 0x00, 0x03,
			0x80, 0x80, 0x80, 0x50, 0x00, 0x01, 0x00, 0x04,
			0x80, 0x80, 0x80, 0x42, 0x20, 0x11, 0x00, 0x00,
			0x00, 0x00, 0x0f, 0x42, 0x40, 0x00, 0x0f, 0x42,
			0x40, 0x05, 0x80, 0x80, 0x80, 0x30, 0x00, 0x00,
			0x01, 0xb0, 0x01, 0x00, 0x00, 0x01, 0xb5, 0x89,
			0x13, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x20, 0x00, 0xc4, 0x8d, 0x88, 0x00, 0xf5, 0x3c,
			0x04, 0x87, 0x14, 0x63, 0x00, 0x00, 0x01, 0xb2,
			0x4c, 0x61, 0x76, 0x63, 0x35, 0x38, 0x2e, 0x31,
			0x33, 0x34, 0x2e, 0x31, 0x30, 0x30, 0x06, 0x80,
			0x80, 0x80, 0x01, 0x02, 0x00, 0x00, 0x00, 0x14,
			0x62, 0x74, 0x72, 0x74, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x0f, 0x42, 0x40, 0x00, 0x0f, 0x42, 0x40,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x74, 0x73,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x73, 0x63,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x73, 0x7a,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
			0x6d, 0x76, 0x65, 0x78, 0x00, 0x00, 0x00, 0x20,
			0x74, 0x72, 0x65, 0x78, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 90000,
					Codec: &mp4.CodecMPEG4Video{
						Config: []byte{
							0x00, 0x00, 0x01, 0xb0, 0x01, 0x00, 0x00, 0x01,
							0xb5, 0x89, 0x13, 0x00, 0x00, 0x01, 0x00, 0x00,
							0x00, 0x01, 0x20, 0x00, 0xc4, 0x8d, 0x88, 0x00,
							0xf5, 0x3c, 0x04, 0x87, 0x14, 0x63, 0x00, 0x00,
							0x01, 0xb2, 0x4c, 0x61, 0x76, 0x63, 0x35, 0x38,
							0x2e, 0x31, 0x33, 0x34, 0x2e, 0x31, 0x30, 0x30,
						},
					},
				},
			},
		},
	},
	{
		"mpeg-1 video",
		[]byte{
			0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0xa2, 0x6d, 0x6f, 0x6f, 0x76,
			0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x02, 0x06,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x03, 0x20, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00,
			0x00, 0x00, 0x01, 0xa2, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x56, 0x69, 0x64, 0x65, 0x6f, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
			0x4d, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x14, 0x76, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01,
			0x0d, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0xc1, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xb1, 0x6d, 0x70, 0x34, 0x76, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x20, 0x02,
			0x58, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x18, 0xff, 0xff, 0x00, 0x00, 0x00, 0x47, 0x65,
			0x73, 0x64, 0x73, 0x00, 0x00, 0x00, 0x00, 0x03,
			0x80, 0x80, 0x80, 0x36, 0x00, 0x01, 0x00, 0x04,
			0x80, 0x80, 0x80, 0x28, 0x61, 0x11, 0x00, 0x00,
			0x00, 0x00, 0x0f, 0x42, 0x40, 0x00, 0x0f, 0x42,
			0x40, 0x05, 0x80, 0x80, 0x80, 0x16, 0x00, 0x00,
			0x01, 0xb3, 0x78, 0x04, 0x38, 0x35, 0xff, 0xff,
			0xe0, 0x18, 0x00, 0x00, 0x01, 0xb5, 0x14, 0x4a,
			0x00, 0x01, 0x00, 0x00, 0x06, 0x80, 0x80, 0x80,
			0x01, 0x02, 0x00, 0x00, 0x00, 0x14, 0x62, 0x74,
			0x72, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f,
			0x42, 0x40, 0x00, 0x0f, 0x42, 0x40, 0x00, 0x00,
			0x00, 0x10, 0x73, 0x74, 0x74, 0x73, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x10, 0x73, 0x74, 0x73, 0x63, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x14, 0x73, 0x74, 0x73, 0x7a, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73, 0x74,
			0x63, 0x6f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x6d, 0x76,
			0x65, 0x78, 0x00, 0x00, 0x00, 0x20, 0x74, 0x72,
			0x65, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 90000,
					Codec: &mp4.CodecMPEG1Video{
						Config: []byte{
							0x00, 0x00, 0x01, 0xb3, 0x78, 0x04, 0x38, 0x35,
							0xff, 0xff, 0xe0, 0x18, 0x00, 0x00, 0x01, 0xb5,
							0x14, 0x4a, 0x00, 0x01, 0x00, 0x00,
						},
					},
				},
			},
		},
	},
	{
		"mjpeg video",
		[]byte{
			0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0x87, 0x6d, 0x6f, 0x6f, 0x76,
			0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0xeb,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x02, 0x80, 0x00, 0x00, 0x01, 0xe0, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x87, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x56, 0x69, 0x64, 0x65, 0x6f, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
			0x32, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x14, 0x76, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xf2, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0xa6, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x96, 0x6d, 0x70, 0x34, 0x76, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x80, 0x01,
			0xe0, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x18, 0xff, 0xff, 0x00, 0x00, 0x00, 0x2c, 0x65,
			0x73, 0x64, 0x73, 0x00, 0x00, 0x00, 0x00, 0x03,
			0x80, 0x80, 0x80, 0x1b, 0x00, 0x01, 0x00, 0x04,
			0x80, 0x80, 0x80, 0x0d, 0x6c, 0x11, 0x00, 0x00,
			0x00, 0x00, 0x0f, 0x42, 0x40, 0x00, 0x0f, 0x42,
			0x40, 0x06, 0x80, 0x80, 0x80, 0x01, 0x02, 0x00,
			0x00, 0x00, 0x14, 0x62, 0x74, 0x72, 0x74, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x0f, 0x42, 0x40, 0x00,
			0x0f, 0x42, 0x40, 0x00, 0x00, 0x00, 0x10, 0x73,
			0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73,
			0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x73,
			0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x10, 0x73, 0x74, 0x63, 0x6f, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x28, 0x6d, 0x76, 0x65, 0x78, 0x00,
			0x00, 0x00, 0x20, 0x74, 0x72, 0x65, 0x78, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 90000,
					Codec: &mp4.CodecMJPEG{
						Width:  640,
						Height: 480,
					},
				},
			},
		},
	},
	{
		"opus",
		[]byte{
			0x00, 0x00, 0x00, 0x20,
			'f', 't', 'y', 'p',
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0x38,
			'm', 'o', 'o', 'v',
			0x00, 0x00, 0x00, 0x6c,
			'm', 'v', 'h', 'd',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0x9c,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x38, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbb, 0x80,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x73, 0x6f, 0x75, 0x6e, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x53, 0x6f, 0x75, 0x6e, 0x64, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x00,
			0xe3, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x10, 0x73, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0xa7, 0x73, 0x74, 0x62,
			0x6c, 0x00, 0x00, 0x00, 0x5b, 0x73, 0x74, 0x73,
			0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x4b, 0x4f, 0x70, 0x75,
			0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x10, 0x00, 0x00, 0x00,
			0x00, 0xbb, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x13, 0x64, 0x4f, 0x70, 0x73, 0x00, 0x02, 0x01,
			0x38, 0x00, 0x00, 0xbb, 0x80, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x14, 0x62, 0x74, 0x72, 0x74,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xf7, 0x39,
			0x00, 0x01, 0xf7, 0x39, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14,
			0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x63, 0x6f,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x28, 0x6d, 0x76, 0x65, 0x78,
			0x00, 0x00, 0x00, 0x20, 0x74, 0x72, 0x65, 0x78,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 48000,
					Codec: &mp4.CodecOpus{
						ChannelCount: 2,
					},
				},
			},
		},
	},
	{
		"mpeg-4 audio",
		[]byte{
			0x00, 0x00, 0x00, 0x20,
			'f', 't', 'y', 'p',
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0x58,
			'm', 'o', 'o', 'v',
			0x00, 0x00, 0x00, 0x6c,
			'm', 'v', 'h', 'd',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0xbc,
			't', 'r', 'a', 'k',
			0x00, 0x00, 0x00, 0x5c,
			't', 'k', 'h', 'd',
			0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x58,
			'm', 'd', 'i', 'a',
			0x00, 0x00, 0x00, 0x20,
			'm', 'd', 'h', 'd',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xac, 0x44,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d,
			'h', 'd', 'l', 'r',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x73, 0x6f, 0x75, 0x6e, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x53, 0x6f, 0x75, 0x6e, 0x64, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
			0x03,
			'm', 'i', 'n', 'f',
			0x00, 0x00, 0x00, 0x10,
			's', 'm', 'h', 'd',
			0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x24,
			'd', 'i', 'n', 'f',
			0x00, 0x00, 0x00,
			0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0xc7, 0x73, 0x74, 0x62,
			0x6c, 0x00, 0x00, 0x00, 0x7b, 0x73, 0x74, 0x73,
			0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x6b,
			'm', 'p', '4', 'a',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x10, 0x00, 0x00, 0x00,
			0x00, 0xac, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x33,
			'e', 's', 'd', 's',
			0x00, 0x00, 0x00,
			0x00, 0x03, 0x80, 0x80, 0x80, 0x22, 0x00, 0x01,
			0x00, 0x04, 0x80, 0x80, 0x80, 0x14, 0x40, 0x15,
			0x00, 0x00, 0x00, 0x00, 0x01, 0xf7, 0x39, 0x00,
			0x01, 0xf7, 0x39, 0x05, 0x80, 0x80, 0x80, 0x02,
			0x12, 0x10, 0x06, 0x80, 0x80, 0x80, 0x01, 0x02,
			0x00, 0x00, 0x00, 0x14,
			'b', 't', 'r', 't',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xf7, 0x39,
			0x00, 0x01, 0xf7, 0x39, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14,
			0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x63, 0x6f,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x28,
			'm', 'v', 'e', 'x',
			0x00, 0x00, 0x00, 0x20,
			't', 'r', 'e', 'x',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: uint32(testAudioTrack.Config.SampleRate),
					Codec:     testAudioTrack,
				},
			},
		},
	},
	{
		"mpeg-1 audio",
		[]byte{
			0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0x51, 0x6d, 0x6f, 0x6f, 0x76,
			0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0xb5,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x51, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x73, 0x6f, 0x75, 0x6e, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x53, 0x6f, 0x75, 0x6e, 0x64, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x00,
			0xfc, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x10, 0x73, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0xc0, 0x73, 0x74, 0x62,
			0x6c, 0x00, 0x00, 0x00, 0x74, 0x73, 0x74, 0x73,
			0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x64, 0x6d, 0x70, 0x34,
			0x61, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x10, 0x00, 0x00, 0x00,
			0x00, 0xbb, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x2c, 0x65, 0x73, 0x64, 0x73, 0x00, 0x00, 0x00,
			0x00, 0x03, 0x80, 0x80, 0x80, 0x1b, 0x00, 0x01,
			0x00, 0x04, 0x80, 0x80, 0x80, 0x0d, 0x6b, 0x15,
			0x00, 0x00, 0x00, 0x00, 0x01, 0xf7, 0x39, 0x00,
			0x01, 0xf7, 0x39, 0x06, 0x80, 0x80, 0x80, 0x01,
			0x02, 0x00, 0x00, 0x00, 0x14, 0x62, 0x74, 0x72,
			0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xf7,
			0x39, 0x00, 0x01, 0xf7, 0x39, 0x00, 0x00, 0x00,
			0x10, 0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x10, 0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x14, 0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x63,
			0x6f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x28, 0x6d, 0x76, 0x65,
			0x78, 0x00, 0x00, 0x00, 0x20, 0x74, 0x72, 0x65,
			0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 90000,
					Codec: &mp4.CodecMPEG1Audio{
						SampleRate:   48000,
						ChannelCount: 2,
					},
				},
			},
		},
	},
	{
		"ac-3",
		[]byte{
			0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0x30, 0x6d, 0x6f, 0x6f, 0x76,
			0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0x94,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x30, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x73, 0x6f, 0x75, 0x6e, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x53, 0x6f, 0x75, 0x6e, 0x64, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x00,
			0xdb, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x10, 0x73, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x9f, 0x73, 0x74, 0x62,
			0x6c, 0x00, 0x00, 0x00, 0x53, 0x73, 0x74, 0x73,
			0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x43, 0x61, 0x63, 0x2d,
			0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x06, 0x00, 0x10, 0x00, 0x00, 0x00,
			0x00, 0xbb, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x0b, 0x64, 0x61, 0x63, 0x33, 0x10, 0x3d, 0xe0,
			0x00, 0x00, 0x00, 0x14, 0x62, 0x74, 0x72, 0x74,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xf7, 0x39,
			0x00, 0x01, 0xf7, 0x39, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14,
			0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x63, 0x6f,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x28, 0x6d, 0x76, 0x65, 0x78,
			0x00, 0x00, 0x00, 0x20, 0x74, 0x72, 0x65, 0x78,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 90000,
					Codec: &mp4.CodecAC3{
						SampleRate:   48000,
						ChannelCount: 6,
						Fscod:        0x0,
						Bsid:         0x8,
						Bsmod:        0x0,
						Acmod:        0x7,
						LfeOn:        true,
						BitRateCode:  0xf,
					},
				},
			},
		},
	},
	{
		"lpcm",
		[]byte{
			0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x02, 0x33, 0x6d, 0x6f, 0x6f, 0x76,
			0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0x97,
			0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
			0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x33, 0x6d, 0x64, 0x69, 0x61,
			0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x73, 0x6f, 0x75, 0x6e, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x53, 0x6f, 0x75, 0x6e, 0x64, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x00,
			0xde, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x10, 0x73, 0x6d, 0x68, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
			0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0xa2, 0x73, 0x74, 0x62,
			0x6c, 0x00, 0x00, 0x00, 0x56, 0x73, 0x74, 0x73,
			0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x46, 0x69, 0x70, 0x63,
			0x6d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x02, 0x00, 0x18, 0x00, 0x00, 0x00,
			0x00, 0xbb, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x0e, 0x70, 0x63, 0x6d, 0x43, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x14, 0x62,
			0x74, 0x72, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0xf7, 0x39, 0x00, 0x01, 0xf7, 0x39, 0x00,
			0x00, 0x00, 0x10, 0x73, 0x74, 0x74, 0x73, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x10, 0x73, 0x74, 0x73, 0x63, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x14, 0x73, 0x74, 0x73, 0x7a, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73,
			0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x6d,
			0x76, 0x65, 0x78, 0x00, 0x00, 0x00, 0x20, 0x74,
			0x72, 0x65, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 90000,
					Codec: &mp4.CodecLPCM{
						BitDepth:     24,
						SampleRate:   48000,
						ChannelCount: 2,
					},
				},
			},
		},
	},
	{
		"h264 + mpeg-4 audio",
		[]byte{
			0x00, 0x00, 0x00, 0x20,
			'f', 't', 'y', 'p',
			0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
			0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
			0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
			0x00, 0x00, 0x04, 0x64,
			'm', 'o', 'o', 'v',
			0x00, 0x00, 0x00, 0x6c,
			'm', 'v', 'h', 'd',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xe8,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0xec,
			't', 'r', 'a', 'k',
			0x00, 0x00, 0x00, 0x5c,
			't', 'k', 'h', 'd',
			0x00, 0x00, 0x00, 0x03,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			0x07, 0x80, 0x00, 0x00, 0x04, 0x38, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x88,
			'm', 'd', 'i', 'a',
			0x00, 0x00, 0x00, 0x20,
			'm', 'd', 'h', 'd',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5f, 0x90,
			0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x2d,
			'h', 'd', 'l', 'r',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x56, 0x69, 0x64, 0x65, 0x6f, 0x48, 0x61, 0x6e,
			0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
			0x33,
			'm', 'i', 'n', 'f',
			0x00, 0x00, 0x00, 0x14,
			'v', 'm', 'h', 'd',
			0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xf3, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0xa7, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x97, 0x61, 0x76, 0x63, 0x31, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x80, 0x04,
			0x38, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x18, 0xff, 0xff, 0x00, 0x00, 0x00, 0x2d, 0x61,
			0x76, 0x63, 0x43, 0x01, 0x42, 0xc0, 0x28, 0x03,
			0x01, 0x00, 0x19, 0x67, 0x42, 0xc0, 0x28, 0xd9,
			0x00, 0x78, 0x02, 0x27, 0xe5, 0x84, 0x00, 0x00,
			0x03, 0x00, 0x04, 0x00, 0x00, 0x03, 0x00, 0xf0,
			0x3c, 0x60, 0xc9, 0x20, 0x01, 0x00, 0x01, 0x08,
			0x00, 0x00, 0x00, 0x14, 0x62, 0x74, 0x72, 0x74,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x42, 0x40,
			0x00, 0x0f, 0x42, 0x40, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14,
			0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x63, 0x6f,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0xbc,
			't', 'r', 'a', 'k',
			0x00, 0x00, 0x00, 0x5c,
			't', 'k', 'h', 'd',
			0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x58,
			'm', 'd', 'i', 'a',
			0x00, 0x00, 0x00, 0x20,
			'm', 'd', 'h', 'd',
			0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0xac, 0x44, 0x00, 0x00, 0x00, 0x00,
			0x55, 0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d,
			'h', 'd', 'l', 'r',
			0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x73, 0x6f, 0x75, 0x6e,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x53, 0x6f, 0x75, 0x6e,
			0x64, 0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72,
			0x00, 0x00, 0x00, 0x01, 0x03, 0x6d, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x10,
			's', 'm', 'h', 'd',
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e,
			0x66, 0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65,
			0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c,
			0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0xc7, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
			0x7b, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x6b, 0x6d, 0x70, 0x34, 0x61, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
			0x10, 0x00, 0x00, 0x00, 0x00, 0xac, 0x44, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x33, 0x65, 0x73, 0x64,
			0x73, 0x00, 0x00, 0x00, 0x00, 0x03, 0x80, 0x80,
			0x80, 0x22, 0x00, 0x02, 0x00, 0x04, 0x80, 0x80,
			0x80, 0x14, 0x40, 0x15, 0x00, 0x00, 0x00, 0x00,
			0x01, 0xf7, 0x39, 0x00, 0x01, 0xf7, 0x39, 0x05,
			0x80, 0x80, 0x80, 0x02, 0x12, 0x10, 0x06, 0x80,
			0x80, 0x80, 0x01, 0x02, 0x00, 0x00, 0x00, 0x14,
			0x62, 0x74, 0x72, 0x74, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0xf7, 0x39, 0x00, 0x01, 0xf7, 0x39,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x74, 0x73,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x73, 0x63,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x73, 0x7a,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
			0x73, 0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48,
			0x6d, 0x76, 0x65, 0x78, 0x00, 0x00, 0x00, 0x20,
			0x74, 0x72, 0x65, 0x78, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
			0x74, 0x72, 0x65, 0x78, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00,
		},
		Init{
			Tracks: []*InitTrack{
				{
					ID:        1,
					TimeScale: 90000,
					Codec:     testVideoTrack,
				},
				{
					ID:        2,
					TimeScale: uint32(testAudioTrack.Config.SampleRate),
					Codec:     testAudioTrack,
				},
			},
		},
	},
}

func TestInitUnmarshal(t *testing.T) {
	for _, ca := range casesInit {
		t.Run(ca.name, func(t *testing.T) {
			var init Init
			err := init.Unmarshal(bytes.NewReader(ca.enc))
			require.NoError(t, err)
			require.Equal(t, ca.dec, init)
		})
	}
}

func TestInitUnmarshalExternal(t *testing.T) {
	for _, ca := range []struct {
		name string
		byts []byte
		init Init
	}{
		{
			"h264",
			[]byte{
				0x00, 0x00, 0x00, 0x1c,
				'f', 't', 'y', 'p',
				0x64, 0x61, 0x73, 0x68, 0x00, 0x00, 0x00, 0x01,
				0x69, 0x73, 0x6f, 0x6d, 0x61, 0x76, 0x63, 0x31,
				0x64, 0x61, 0x73, 0x68, 0x00, 0x00, 0x02, 0x92,
				'm', 'o', 'o', 'v',
				0x00, 0x00, 0x00, 0x6c,
				'm', 'v', 'h', 'd',
				0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x98, 0x96, 0x80, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
				0x00, 0x00, 0x01, 0xf6,
				't', 'r', 'a', 'k',
				0x00, 0x00, 0x00, 0x5c,
				't', 'k', 'h', 'd',
				0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x40, 0x00, 0x00, 0x00, 0x03, 0xc0, 0x00, 0x00,
				0x02, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x01, 0x92,
				'm', 'd', 'i', 'a',
				0x00, 0x00, 0x00, 0x20,
				'm', 'd', 'h', 'd',
				0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x98, 0x96, 0x80, 0x00, 0x00, 0x00, 0x00,
				0x55, 0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38,
				'h', 'd', 'l', 'r',
				0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x76, 0x69, 0x64, 0x65,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x42, 0x72, 0x6f, 0x61,
				0x64, 0x70, 0x65, 0x61, 0x6b, 0x20, 0x56, 0x69,
				0x64, 0x65, 0x6f, 0x20, 0x48, 0x61, 0x6e, 0x64,
				0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01, 0x32,
				'm', 'i', 'n', 'f',
				0x00, 0x00, 0x00, 0x14,
				'v', 'm', 'h', 'd',
				0x00, 0x00, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x24,
				'd', 'i', 'n', 'f',
				0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65, 0x66,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20,
				0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xf2,
				0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00, 0xa6,
				0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x96,
				0x61, 0x76, 0x63, 0x31, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x03, 0xc0, 0x02, 0x1c,
				0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x04, 0x68,
				0x32, 0x36, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
				0xff, 0xff, 0x00, 0x00, 0x00, 0x30, 0x61, 0x76,
				0x63, 0x43, 0x01, 0x42, 0xc0, 0x1f, 0xff, 0xe1,
				0x00, 0x19, 0x67, 0x42, 0xc0, 0x1f, 0xd9, 0x00,
				0xf0, 0x11, 0x7e, 0xf0, 0x11, 0x00, 0x00, 0x03,
				0x00, 0x01, 0x00, 0x00, 0x03, 0x00, 0x30, 0x8f,
				0x18, 0x32, 0x48, 0x01, 0x00, 0x04, 0x68, 0xcb,
				0x8c, 0xb2, 0x00, 0x00, 0x00, 0x10, 0x70, 0x61,
				0x73, 0x70, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x73, 0x74,
				0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73, 0x74,
				0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x73, 0x74,
				0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x10, 0x73, 0x74, 0x63, 0x6f, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x28, 0x6d, 0x76, 0x65, 0x78, 0x00, 0x00,
				0x00, 0x20, 0x74, 0x72, 0x65, 0x78, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			Init{
				Tracks: []*InitTrack{
					{
						ID:        256,
						TimeScale: 10000000,
						Codec: &mp4.CodecH264{
							SPS: []byte{
								0x67, 0x42, 0xc0, 0x1f, 0xd9, 0x00, 0xf0, 0x11,
								0x7e, 0xf0, 0x11, 0x00, 0x00, 0x03, 0x00, 0x01,
								0x00, 0x00, 0x03, 0x00, 0x30, 0x8f, 0x18, 0x32,
								0x48,
							},
							PPS: []byte{
								0x68, 0xcb, 0x8c, 0xb2,
							},
						},
					},
				},
			},
		},
		{
			"h265 apple",
			[]byte{
				0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
				0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
				0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
				0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
				0x00, 0x00, 0x04, 0x96, 0x6d, 0x6f, 0x6f, 0x76,
				0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
				0x00, 0x00, 0x00, 0x00, 0xd5, 0x5b, 0xc6, 0x84,
				0xd5, 0x5b, 0xc6, 0x84, 0x00, 0x00, 0x02, 0x58,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x02, 0x70,
				0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
				0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x01,
				0xd5, 0x5b, 0xc6, 0x84, 0xd5, 0x5b, 0xc6, 0x84,
				0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
				0x03, 0xc0, 0x00, 0x00, 0x02, 0x1c, 0x00, 0x00,
				0x00, 0x00, 0x02, 0x0c, 0x6d, 0x64, 0x69, 0x61,
				0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
				0x00, 0x00, 0x00, 0x00, 0xd5, 0x5b, 0xc6, 0x84,
				0xd5, 0x5b, 0xc6, 0x84, 0x00, 0x00, 0x17, 0x70,
				0x00, 0x00, 0x00, 0x00, 0x15, 0xc7, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x31, 0x68, 0x64, 0x6c, 0x72,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x43, 0x6f, 0x72, 0x65, 0x20, 0x4d, 0x65, 0x64,
				0x69, 0x61, 0x20, 0x56, 0x69, 0x64, 0x65, 0x6f,
				0x00, 0x00, 0x00, 0x01, 0xb3, 0x6d, 0x69, 0x6e,
				0x66, 0x00, 0x00, 0x00, 0x14, 0x76, 0x6d, 0x68,
				0x64, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
				0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
				0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x01, 0x73, 0x73, 0x74, 0x62,
				0x6c, 0x00, 0x00, 0x01, 0x27, 0x73, 0x74, 0x73,
				0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x01, 0x17, 0x68, 0x76, 0x63,
				0x31, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x03, 0xc0, 0x02, 0x1c, 0x00, 0x48, 0x00,
				0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x18, 0xff, 0xff, 0x00,
				0x00, 0x00, 0x8a, 0x68, 0x76, 0x63, 0x43, 0x01,
				0x02, 0x00, 0x00, 0x00, 0x04, 0xb0, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x7b, 0xf0, 0x00, 0xfc, 0xfd,
				0xfa, 0xfa, 0x00, 0x00, 0x0f, 0x03, 0xa0, 0x00,
				0x01, 0x00, 0x18, 0x40, 0x01, 0x0c, 0x01, 0xff,
				0xff, 0x02, 0x20, 0x00, 0x00, 0x03, 0x00, 0xb0,
				0x00, 0x00, 0x03, 0x00, 0x00, 0x03, 0x00, 0x7b,
				0x18, 0xb0, 0x24, 0xa1, 0x00, 0x01, 0x00, 0x3c,
				0x42, 0x01, 0x01, 0x02, 0x20, 0x00, 0x00, 0x03,
				0x00, 0xb0, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
				0x00, 0x7b, 0xa0, 0x07, 0x82, 0x00, 0x88, 0x7d,
				0xb6, 0x71, 0x8b, 0x92, 0x44, 0x80, 0x53, 0x88,
				0x88, 0x92, 0xcf, 0x24, 0xa6, 0x92, 0x72, 0xc9,
				0x12, 0x49, 0x22, 0xdc, 0x91, 0xaa, 0x48, 0xfc,
				0xa2, 0x23, 0xff, 0x00, 0x01, 0x00, 0x01, 0x6a,
				0x02, 0x02, 0x02, 0x01, 0xa2, 0x00, 0x01, 0x00,
				0x08, 0x44, 0x01, 0xc0, 0x25, 0x2f, 0x05, 0x32,
				0x40, 0x00, 0x00, 0x00, 0x13, 0x63, 0x6f, 0x6c,
				0x72, 0x6e, 0x63, 0x6c, 0x78, 0x00, 0x01, 0x00,
				0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x0a,
				0x66, 0x69, 0x65, 0x6c, 0x01, 0x00, 0x00, 0x00,
				0x00, 0x0a, 0x63, 0x68, 0x72, 0x6d, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x10, 0x70, 0x61, 0x73, 0x70,
				0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x74, 0x73,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x73, 0x63,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x73, 0x7a,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
				0x73, 0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x6a,
				0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
				0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x01,
				0xd5, 0x5b, 0xc6, 0x84, 0xd5, 0x5b, 0xc6, 0x84,
				0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x01, 0x06, 0x6d, 0x64, 0x69, 0x61,
				0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
				0x00, 0x00, 0x00, 0x00, 0xd5, 0x5b, 0xc6, 0x84,
				0xd5, 0x5b, 0xc6, 0x84, 0x00, 0x00, 0x75, 0x30,
				0x00, 0x00, 0x00, 0x00, 0x15, 0xc7, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x3a, 0x68, 0x64, 0x6c, 0x72,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x63, 0x6c, 0x63, 0x70, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x43, 0x6f, 0x72, 0x65, 0x20, 0x4d, 0x65, 0x64,
				0x69, 0x61, 0x20, 0x43, 0x6c, 0x6f, 0x73, 0x65,
				0x64, 0x20, 0x43, 0x61, 0x70, 0x74, 0x69, 0x6f,
				0x6e, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x6d, 0x69,
				0x6e, 0x66, 0x00, 0x00, 0x00, 0x0c, 0x6e, 0x6d,
				0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00,
				0x00, 0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x6c, 0x73, 0x74,
				0x62, 0x6c, 0x00, 0x00, 0x00, 0x20, 0x73, 0x74,
				0x73, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x63, 0x36,
				0x30, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x73, 0x74,
				0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73, 0x74,
				0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x73, 0x74,
				0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x10, 0x73, 0x74, 0x63, 0x6f, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x48, 0x6d, 0x76, 0x65, 0x78, 0x00, 0x00,
				0x00, 0x20, 0x74, 0x72, 0x65, 0x78, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x20, 0x74, 0x72, 0x65, 0x78, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			Init{
				Tracks: []*InitTrack{{
					ID:        1,
					TimeScale: 6000,
					Codec: &mp4.CodecH265{
						VPS: []byte{
							0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x02, 0x20,
							0x00, 0x00, 0x03, 0x00, 0xb0, 0x00, 0x00, 0x03,
							0x00, 0x00, 0x03, 0x00, 0x7b, 0x18, 0xb0, 0x24,
						},
						SPS: []byte{
							0x42, 0x01, 0x01, 0x02, 0x20, 0x00, 0x00, 0x03,
							0x00, 0xb0, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
							0x00, 0x7b, 0xa0, 0x07, 0x82, 0x00, 0x88, 0x7d,
							0xb6, 0x71, 0x8b, 0x92, 0x44, 0x80, 0x53, 0x88,
							0x88, 0x92, 0xcf, 0x24, 0xa6, 0x92, 0x72, 0xc9,
							0x12, 0x49, 0x22, 0xdc, 0x91, 0xaa, 0x48, 0xfc,
							0xa2, 0x23, 0xff, 0x00, 0x01, 0x00, 0x01, 0x6a,
							0x02, 0x02, 0x02, 0x01,
						},
						PPS: []byte{
							0x44, 0x01, 0xc0, 0x25, 0x2f, 0x05, 0x32, 0x40,
						},
					},
				}},
			},
		},
		{
			"mpeg-4 audio",
			[]byte{
				0x00, 0x00, 0x00, 0x18,
				'f', 't', 'y', 'p',
				0x69, 0x73, 0x6f, 0x35, 0x00, 0x00, 0x00, 0x01,
				0x69, 0x73, 0x6f, 0x35, 0x64, 0x61, 0x73, 0x68,
				0x00, 0x00, 0x02, 0x43,
				'm', 'o', 'o', 'v',
				0x00, 0x00, 0x00, 0x6c,
				'm', 'v', 'h', 'd',
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x98, 0x96, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0xa7,
				't', 'r', 'a', 'k',
				0x00, 0x00, 0x00, 0x5c,
				't', 'k', 'h', 'd',
				0x00, 0x00, 0x00, 0x07,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x01, 0x43,
				'm', 'd', 'i', 'a',
				0x00, 0x00, 0x00, 0x20,
				'm', 'd', 'h', 'd',
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x98, 0x96, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x55, 0xc4, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x38,
				'h', 'd', 'l', 'r',
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x73, 0x6f, 0x75, 0x6e, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x42, 0x72, 0x6f, 0x61, 0x64, 0x70, 0x65, 0x61,
				0x6b, 0x20, 0x53, 0x6f, 0x75, 0x6e, 0x64, 0x20,
				0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00,
				0x00, 0x00, 0x00, 0xe3,
				'm', 'i', 'n', 'f',
				0x00, 0x00, 0x00, 0x10,
				's', 'm', 'h', 'd',
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x24,
				'd', 'i', 'n', 'f',
				0x00, 0x00, 0x00, 0x1c, 0x64, 0x72, 0x65, 0x66,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20,
				0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xa7,
				0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00, 0x5b,
				0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x4b,
				0x6d, 0x70, 0x34, 0x61, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x10,
				0x00, 0x00, 0x00, 0x00, 0xbb, 0x80, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x27, 0x65, 0x73, 0x64, 0x73,
				0x00, 0x00, 0x00, 0x00, 0x03, 0x19, 0x00, 0x00,
				0x00, 0x04, 0x11, 0x40, 0x15, 0x00, 0x30, 0x00,
				0x00, 0x11, 0x94, 0x00, 0x00, 0x11, 0x94, 0x00,
				0x05, 0x02, 0x11, 0x90, 0x06, 0x01, 0x02, 0x00,
				0x00, 0x00, 0x10, 0x73, 0x74, 0x74, 0x73, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x10, 0x73, 0x74, 0x73, 0x63, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x14, 0x73, 0x74, 0x73, 0x7a, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73,
				0x74, 0x63, 0x6f, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x6d,
				0x76, 0x65, 0x78, 0x00, 0x00, 0x00, 0x20, 0x74,
				0x72, 0x65, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00,
			},
			Init{
				Tracks: []*InitTrack{
					{
						ID:        257,
						TimeScale: 10000000,
						Codec: &mp4.CodecMPEG4Audio{
							Config: mpeg4audio.AudioSpecificConfig{
								Type:         mpeg4audio.ObjectTypeAACLC,
								SampleRate:   48000,
								ChannelCount: 2,
							},
						},
					},
				},
			},
		},
		{
			"ignored closed captions",
			[]byte{
				0x00, 0x00, 0x00, 0x20, 0x66, 0x74, 0x79, 0x70,
				0x6d, 0x70, 0x34, 0x32, 0x00, 0x00, 0x00, 0x01,
				0x6d, 0x70, 0x34, 0x31, 0x6d, 0x70, 0x34, 0x32,
				0x69, 0x73, 0x6f, 0x6d, 0x68, 0x6c, 0x73, 0x66,
				0x00, 0x00, 0x04, 0x3f, 0x6d, 0x6f, 0x6f, 0x76,
				0x00, 0x00, 0x00, 0x6c, 0x6d, 0x76, 0x68, 0x64,
				0x00, 0x00, 0x00, 0x00, 0xd5, 0x5b, 0xc6, 0x62,
				0xd5, 0x5b, 0xc6, 0x62, 0x00, 0x00, 0x02, 0x58,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x02, 0x19,
				0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00, 0x5c,
				0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00, 0x01,
				0xd5, 0x5b, 0xc6, 0x62, 0xd5, 0x5b, 0xc6, 0x62,
				0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
				0x07, 0x80, 0x00, 0x00, 0x04, 0x38, 0x00, 0x00,
				0x00, 0x00, 0x01, 0xb5, 0x6d, 0x64, 0x69, 0x61,
				0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68, 0x64,
				0x00, 0x00, 0x00, 0x00, 0xd5, 0x5b, 0xc6, 0x62,
				0xd5, 0x5b, 0xc6, 0x62, 0x00, 0x00, 0x17, 0x70,
				0x00, 0x00, 0x00, 0x00, 0x15, 0xc7, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x31, 0x68, 0x64, 0x6c, 0x72,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x43, 0x6f, 0x72, 0x65, 0x20, 0x4d, 0x65, 0x64,
				0x69, 0x61, 0x20, 0x56, 0x69, 0x64, 0x65, 0x6f,
				0x00, 0x00, 0x00, 0x01, 0x5c, 0x6d, 0x69, 0x6e,
				0x66, 0x00, 0x00, 0x00, 0x14, 0x76, 0x6d, 0x68,
				0x64, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
				0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
				0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x01, 0x1c, 0x73, 0x74, 0x62,
				0x6c, 0x00, 0x00, 0x00, 0xd0, 0x73, 0x74, 0x73,
				0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0xc0, 0x61, 0x76, 0x63,
				0x31, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x07, 0x80, 0x04, 0x38, 0x00, 0x48, 0x00,
				0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x18, 0xff, 0xff, 0x00,
				0x00, 0x00, 0x33, 0x61, 0x76, 0x63, 0x43, 0x01,
				0x64, 0x00, 0x2a, 0xff, 0xe1, 0x00, 0x1b, 0x27,
				0x64, 0x00, 0x2a, 0xac, 0x52, 0x14, 0x07, 0x80,
				0x22, 0x7e, 0x5f, 0xfc, 0x00, 0x04, 0x00, 0x05,
				0xa8, 0x08, 0x08, 0x0d, 0xb6, 0x15, 0xaf, 0x7b,
				0xe0, 0x20, 0x01, 0x00, 0x05, 0x28, 0xf9, 0x09,
				0x09, 0xcb, 0x00, 0x00, 0x00, 0x13, 0x63, 0x6f,
				0x6c, 0x72, 0x6e, 0x63, 0x6c, 0x78, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
				0x0a, 0x66, 0x69, 0x65, 0x6c, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x0a, 0x63, 0x68, 0x72, 0x6d, 0x01,
				0x01, 0x00, 0x00, 0x00, 0x10, 0x70, 0x61, 0x73,
				0x70, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x74,
				0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x10, 0x73, 0x74, 0x73,
				0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x73,
				0x7a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x10, 0x73, 0x74, 0x63, 0x6f, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x6a, 0x74, 0x72, 0x61, 0x6b, 0x00, 0x00, 0x00,
				0x5c, 0x74, 0x6b, 0x68, 0x64, 0x00, 0x00, 0x00,
				0x01, 0xd5, 0x5b, 0xc6, 0x62, 0xd5, 0x5b, 0xc6,
				0x62, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x01, 0x06, 0x6d, 0x64, 0x69,
				0x61, 0x00, 0x00, 0x00, 0x20, 0x6d, 0x64, 0x68,
				0x64, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x5b, 0xc6,
				0x62, 0xd5, 0x5b, 0xc6, 0x62, 0x00, 0x00, 0x75,
				0x30, 0x00, 0x00, 0x00, 0x00, 0x15, 0xc7, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x3a, 0x68, 0x64, 0x6c,
				0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x63, 0x6c, 0x63, 0x70, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x43, 0x6f, 0x72, 0x65, 0x20, 0x4d, 0x65,
				0x64, 0x69, 0x61, 0x20, 0x43, 0x6c, 0x6f, 0x73,
				0x65, 0x64, 0x20, 0x43, 0x61, 0x70, 0x74, 0x69,
				0x6f, 0x6e, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x6d,
				0x69, 0x6e, 0x66, 0x00, 0x00, 0x00, 0x0c, 0x6e,
				0x6d, 0x68, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x24, 0x64, 0x69, 0x6e, 0x66, 0x00,
				0x00, 0x00, 0x1c, 0x64, 0x72, 0x65, 0x66, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00,
				0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x6c, 0x73,
				0x74, 0x62, 0x6c, 0x00, 0x00, 0x00, 0x20, 0x73,
				0x74, 0x73, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x63,
				0x36, 0x30, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x73,
				0x74, 0x74, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x73,
				0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x73,
				0x74, 0x73, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x10, 0x73, 0x74, 0x63, 0x6f, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x48, 0x6d, 0x76, 0x65, 0x78, 0x00,
				0x00, 0x00, 0x20, 0x74, 0x72, 0x65, 0x78, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x20, 0x74, 0x72, 0x65, 0x78, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
				0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			Init{
				Tracks: []*InitTrack{{
					ID:        1,
					TimeScale: 6000,
					Codec: &mp4.CodecH264{
						SPS: []byte{
							0x27, 0x64, 0x00, 0x2a, 0xac, 0x52, 0x14, 0x07,
							0x80, 0x22, 0x7e, 0x5f, 0xfc, 0x00, 0x04, 0x00,
							0x05, 0xa8, 0x08, 0x08, 0x0d, 0xb6, 0x15, 0xaf,
							0x7b, 0xe0, 0x20,
						},
						PPS: []byte{
							0x28, 0xf9, 0x09, 0x09, 0xcb,
						},
					},
				}},
			},
		},
	} {
		t.Run(ca.name, func(t *testing.T) {
			var init Init
			err := init.Unmarshal(bytes.NewReader(ca.byts))
			require.NoError(t, err)
			require.Equal(t, ca.init, init)
		})
	}
}

func TestInitMarshal(t *testing.T) {
	for _, ca := range casesInit {
		t.Run(ca.name, func(t *testing.T) {
			var buf seekablebuffer.Buffer
			err := ca.dec.Marshal(&buf)
			require.NoError(t, err)
			require.Equal(t, ca.enc, buf.Bytes())
		})
	}
}

func TestInitMarshalEmptyParameters(t *testing.T) {
	for _, ca := range []struct {
		name  string
		codec mp4.Codec
	}{
		{
			"av1",
			&mp4.CodecAV1{},
		},
		{
			"vp9",
			&mp4.CodecVP9{},
		},
		{
			"h265",
			&mp4.CodecH265{},
		},
		{
			"h264",
			&mp4.CodecH264{},
		},
		{
			"mpeg-4 video",
			&mp4.CodecMPEG4Video{},
		},
		{
			"mpeg-1 video",
			&mp4.CodecMPEG1Video{},
		},
		{
			"mjpeg",
			&mp4.CodecMJPEG{},
		},
	} {
		t.Run(ca.name, func(t *testing.T) {
			i := Init{
				Tracks: []*InitTrack{{
					ID:        1,
					TimeScale: 90000,
					Codec:     ca.codec,
				}},
			}

			var buf seekablebuffer.Buffer
			err := i.Marshal(&buf)
			require.Error(t, err)
		})
	}
}

func FuzzInitUnmarshal(f *testing.F) {
	for _, ca := range casesInit {
		f.Add(ca.enc)
	}

	f.Fuzz(func(t *testing.T, b []byte) {
		var init Init
		err := init.Unmarshal(bytes.NewReader(b))
		if err != nil {
			return
		}

		require.NotZero(t, len(init.Tracks))

		for _, track := range init.Tracks {
			require.NotZero(t, track.TimeScale)
		}

		var buf seekablebuffer.Buffer
		err = init.Marshal(&buf)
		require.NoError(t, err)
	})
}

func BenchmarkInitUnmarshal(b *testing.B) {
	for i := 0; i < b.N; i++ {
		for _, ca := range casesInit {
			var init Init
			init.Unmarshal(bytes.NewReader(ca.enc)) //nolint:errcheck
		}
	}
}
