package h265

import (
	"encoding/binary"
	"fmt"
	"testing"

	"github.com/bluenviron/mediacommon/v2/pkg/codecs/h264"
	"github.com/stretchr/testify/require"
)

type sequenceSample struct {
	au  [][]byte
	pts int64
	dts int64
}

var casesDTSExtractor = []struct {
	name     string
	sequence []sequenceSample
}{
	{
		"with timing info, IDR",
		[]sequenceSample{
			{
				[][]byte{
					{ // VPS
						0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x01, 0x60,
						0x00, 0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00, 0x78, 0x99, 0x98, 0x09,
					},
					{ // SPS
						0x42, 0x01, 0x01, 0x01, 0x60, 0x00, 0x00, 0x03,
						0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
						0x00, 0x78, 0xa0, 0x03, 0xc0, 0x80, 0x10, 0xe5,
						0x96, 0x66, 0x69, 0x24, 0xca, 0xe0, 0x10, 0x00,
						0x00, 0x03, 0x00, 0x10, 0x00, 0x00, 0x03, 0x01,
						0xe0, 0x80,
					},
					{ // PPS
						0x44, 0x1, 0xc1, 0x72, 0xb4, 0x62, 0x40,
					},
					{ // IDR_W_RADL
						0x26, 0x1, 0xaf, 0x8, 0x42, 0x23, 0x48, 0x8a, 0x43, 0xe2,
					},
				},
				0,
				-6000,
			},
			{
				[][]byte{{ // TRAIL_R
					0x02, 0x01, 0xd0, 0x19, 0x5f, 0x8c, 0xb4, 0x42,
					0x49, 0x20, 0x40, 0x11, 0x16, 0x92, 0x93, 0xea,
					0x54, 0x57, 0x4e, 0x0a,
				}},
				9000,
				-5910,
			},
			{
				[][]byte{{ // TRAIL_R
					0x02, 0x01, 0xe0, 0x44, 0x97, 0xe0, 0x81, 0x20,
					0x44, 0x52, 0x62, 0x7a, 0x1b, 0x88, 0x0b, 0x21,
					0x26, 0x5f, 0x10, 0x9c,
				}},
				5999,
				-1941,
			},
			{
				[][]byte{{ // TRAIL_N
					0x00, 0x01, 0xe0, 0x24, 0xff, 0xfa, 0x24, 0x0a,
					0x42, 0x25, 0x8c, 0x18, 0xe6, 0x1c, 0xea, 0x5a,
					0x5d, 0x07, 0xc1, 0x8f,
				}},
				2999,
				2999,
			},
			{
				[][]byte{{ // TRAIL_R
					0x02, 0x01, 0xd0, 0x30, 0x97, 0xd7, 0xdc, 0xf9,
					0x0c, 0x10, 0x11, 0x11, 0x20, 0x42, 0x11, 0x18,
					0x63, 0xa5, 0x18, 0x55,
				}},
				18000,
				5999,
			},
			{
				[][]byte{{ // TRAIL_R
					0x02, 0x01, 0xe0, 0xa2, 0x25, 0xd7, 0xf7, 0x08,
					0x12, 0x04, 0x45, 0xa1, 0x83, 0xc0, 0x97, 0x53,
					0xa3, 0x5e, 0x78, 0x14,
				}},
				14999,
				8999,
			},
			{
				[][]byte{{ // TRAIL_N
					0x00, 0x01, 0xe0, 0x82, 0x3f, 0x5f, 0xf6, 0x89,
					0x02, 0x90, 0x88, 0xa3, 0x0c, 0x7d, 0x27, 0x0c,
					0xd4, 0xd9, 0xc2, 0xa5,
				}},
				11999,
				11999,
			},
		},
	},
	{
		"short_term_ref_pic_set_sps_flag",
		[]sequenceSample{
			{
				[][]byte{
					{ // VPS
						0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x01, 0x40,
						0x00, 0x00, 0x03, 0x00, 0x80, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00, 0x99, 0xa5, 0x02, 0x40,
					},
					{ // SPS
						0x42, 0x01, 0x01, 0x01, 0x40, 0x00, 0x00, 0x03,
						0x00, 0x80, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
						0x00, 0x99, 0xa0, 0x03, 0xc0, 0x80, 0x10, 0xe5,
						0x8d, 0xa5, 0x92, 0x42, 0x36, 0x22, 0xec, 0xb8,
						0x80, 0x40, 0x00, 0x00, 0x03, 0x00, 0x40, 0x00,
						0x00, 0x05, 0x0f, 0xe2, 0xc4, 0xa0,
					},
					{ // PPS
						0x44, 0x01, 0xc0, 0xe0, 0x98, 0x93, 0x03, 0x05,
						0x14, 0x90,
					},
					{ // IDR
						0x26, 0x01, 0xaf, 0x3e, 0x3d, 0x3a, 0xca, 0xc0,
						0xf2, 0x2f, 0xc3, 0x0f, 0x86, 0x9f, 0xed, 0xfc,
						0x67, 0x2f, 0x62, 0x69,
					},
				},
				100,
				-4400,
			},
			{
				[][]byte{
					{ // TRAIL_R
						0x02, 0x02, 0xd0, 0x00, 0x0c, 0xc6, 0x27, 0xfe,
						0x6e, 0x6d, 0xe8, 0x10, 0xd5, 0xce, 0x61, 0x1b,
						0x66, 0xf6, 0x21, 0x59,
					},
				},
				6130,
				865,
			},
			{
				[][]byte{
					{ // TRAIL_R
						0x02, 0x02, 0xd0, 0x00, 0x14, 0xc6, 0x7c, 0xfe,
						0x83, 0x29, 0x34, 0xba, 0xce, 0xaa, 0x8b, 0x76,
						0xb0, 0x95, 0x67, 0xb2,
					},
				},
				9100,
				4982,
			},
		},
	},
	{
		"qsv hevc b_frames=3",
		[]sequenceSample{
			{
				[][]byte{ //nolint:dupl
					{ // VPS
						0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x01, 0x40,
						0x00, 0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00, 0x7b, 0x11, 0xc0, 0xc0,
						0x00, 0x00, 0x03, 0x00, 0x40, 0x00, 0x00, 0x0f,
						0x14,
					},
					{ // SPS
						0x42, 0x01, 0x01, 0x01, 0x40, 0x00, 0x00, 0x03,
						0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
						0x00, 0x7b, 0xa0, 0x03, 0xc0, 0x80, 0x11, 0x07,
						0xcb, 0xb1, 0x1e, 0xe4, 0x6c, 0x0a, 0x9f, 0xa6,
						0xb9, 0x97, 0x92, 0xcf, 0x60, 0x2d, 0x40, 0x40,
						0x40, 0x45, 0x00, 0x00, 0x03, 0x00, 0x01, 0x00,
						0x00, 0x03, 0x00, 0x3c, 0x60, 0x35, 0xef, 0x7e,
						0x00, 0x02, 0x62, 0x58, 0x00, 0x26, 0x17, 0x20,
					},
					{ // PPS
						0x44, 0x01, 0xc0, 0x3c, 0xf0, 0x1b, 0x64,
					},
					{
						0x4e, 0x01, 0x00, 0x0a, 0x80, 0x00, 0x00, 0x03,
						0x00, 0x55, 0xda, 0x80, 0x01, 0xe5, 0xd0, 0x01,
						0x07, 0x04, 0x00, 0x00, 0xee, 0x00, 0x00, 0x05,
						0x80,
					},
					{
						0x26, 0x01, 0xae, 0x80, 0x8f, 0x4c, 0xdd, 0xfc,
						0xee, 0x2f, 0x79, 0x7c, 0x9e, 0x21, 0x6b, 0x2a,
						0xe7, 0x6a, 0x57, 0x56, 0x46, 0x6f, 0x32, 0x5a,
						0x7c, 0xbc, 0x47, 0xe8, 0xce, 0x5c, 0x5e, 0xfa,
						0x1e, 0xd0, 0x94, 0x08, 0x4c, 0x98, 0x9d, 0xbb,
						0x5d, 0x4c, 0x54, 0xa1, 0xd9, 0x5b, 0x1b, 0xba,
					},
				},
				182999,
				179999,
			},
			{
				[][]byte{
					{
						0x4e, 0x01, 0x01, 0x07, 0x04, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00, 0x0b, 0x80,
					},
					{
						0x02, 0x01, 0xe2, 0x0a, 0x4f, 0xdd, 0x1e, 0xb7,
						0xb7, 0xa1, 0x80, 0xad, 0xc7, 0x3c, 0x2e, 0x33,
						0x3b, 0xde, 0xcc, 0x77, 0x13, 0x9c, 0x5b, 0xe3,
						0x2c, 0xaa, 0xd4, 0x2e, 0xb0, 0x2b, 0x9e, 0x20,
						0xdd, 0xc9, 0x1b, 0x39, 0xd9, 0x75, 0x06, 0xf5,
						0xa8, 0x1f, 0x66, 0x62, 0x5b, 0xfe, 0x1f, 0xf9,
					},
				},
				188999,
				180089,
			},
			{
				[][]byte{
					{
						0x4e, 0x01, 0x01, 0x07, 0x04, 0x00, 0x00, 0x03,
						0x02, 0x00, 0x00, 0x05, 0x80,
					},
					{
						0x02, 0x01, 0xe1, 0x32, 0x27, 0xe3, 0xa0, 0x51,
						0xcd, 0xff, 0x1a, 0x0b, 0x37, 0xaf, 0xe3, 0xe6,
						0x9e, 0xaa, 0x27, 0x82, 0xcd, 0x28, 0xa3, 0xce,
						0x57, 0x8b, 0x02, 0x3e, 0x62, 0x1f, 0x66, 0x5b,
						0xbd, 0x67, 0x6b, 0xb1, 0x47, 0x9d, 0x1b, 0x26,
						0xb7, 0x2a, 0x04, 0xac, 0x2e, 0x94, 0x1e, 0x22,
					},
				},
				185999,
				182059,
			},
			{
				[][]byte{
					{
						0x4e, 0x01, 0x01, 0x07, 0x04, 0x00, 0x00, 0x04,
						0x00, 0x00, 0x03, 0x01, 0x80,
					},
					{
						0x00, 0x01, 0xe0, 0xcf, 0x8e, 0x80, 0x1e, 0x96,
						0xa6, 0x88, 0xcb, 0x98, 0xf3, 0xd9, 0x2a, 0x4b,
						0xa7, 0xa0, 0xf8, 0xa0, 0x4d, 0x21, 0x89, 0x76,
						0x54, 0xe3, 0x8f, 0x46, 0xf6, 0x93, 0xde, 0x84,
						0x33, 0x26, 0x3e, 0xe8, 0x20, 0x23, 0xef, 0x39,
						0x03, 0x3c, 0x92, 0x11, 0x50, 0x98, 0xd6, 0x13,
					},
				},
				184499,
				184499,
			},
		},
	},
	{
		"issue mediamtx/4892 (no timing info)",
		[]sequenceSample{
			{
				[][]byte{
					{
						0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x01, 0x60,
						0x00, 0x00, 0x03, 0x00, 0xb0, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00, 0x78, 0x15, 0xc0, 0x90,
					},
					{
						0x42, 0x01, 0x01, 0x01, 0x60, 0x00, 0x00, 0x03,
						0x00, 0xb0, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
						0x00, 0x78, 0xa0, 0x03, 0xc0, 0x80, 0x11, 0x07,
						0xcb, 0x88, 0x15, 0xee, 0x45, 0x95, 0x4d, 0x40,
						0x40, 0x40, 0x40, 0x20,
					},
					{
						0x44, 0x01, 0xc0, 0x2c, 0x60, 0xa6, 0x48,
					},
					{
						0x28, 0x01, 0xaf, 0x82, 0x21, 0x80, 0x42, 0x03,
						0x80, 0x36, 0x03, 0x20, 0x32, 0x02, 0xe0, 0x2e,
						0x02, 0xc0, 0x28, 0x02, 0xa0, 0x28, 0x02, 0x60,
						0x26, 0x02, 0x60, 0x24, 0x02, 0x40, 0x24, 0x02,
						0x40, 0x24, 0x02, 0x40, 0x24, 0x02, 0x40, 0x24,
					},
					{
						0x4c, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					},
				},
				0,
				-9000,
			},
			{
				[][]byte{
					{
						0x02, 0x01, 0xd0, 0x04, 0x29, 0x2f, 0xc1, 0x10,
						0x80, 0x90, 0x80, 0x70, 0x70, 0x70, 0x60, 0x60,
						0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60,
						0x50, 0x50, 0x50, 0x50, 0x50, 0x50, 0x50, 0x50,
						0x50, 0x50, 0x50, 0x50, 0x50, 0x50, 0x50, 0x55,
					},
					{
						0x4c, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					},
				},
				12000,
				-8910,
			},
			{
				[][]byte{
					{
						0x02, 0x01, 0xe0, 0x08, 0x92, 0xab, 0xec, 0x11,
						0x18, 0x61, 0xc7, 0x1c, 0x61, 0x86, 0x18, 0x61,
						0x86, 0x18, 0x61, 0x45, 0x14, 0x51, 0x45, 0x14,
						0x51, 0x45, 0x14, 0x51, 0x45, 0x14, 0x51, 0x45,
						0x7f, 0xa8, 0x92, 0x80, 0x00, 0x00, 0x03, 0x01,
					},
					{
						0x4c, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					},
				},
				6000,
				-3940,
			},
			{
				[][]byte{
					{
						0x02, 0x01, 0xe0, 0x04, 0x9f, 0xa3, 0xe8, 0x80,
						0x44, 0x61, 0x87, 0x18, 0x61, 0x86, 0x18, 0x61,
						0x86, 0x18, 0x51, 0x45, 0x14, 0x51, 0x45, 0x14,
						0x51, 0x45, 0x14, 0x51, 0x45, 0x14, 0x51, 0x45,
						0x15, 0xbb, 0x60, 0x91, 0x00, 0x00, 0x03, 0x00,
					},
					{
						0x4c, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					},
				},
				3000,
				3000,
			},
			{
				[][]byte{
					{
						0x02, 0x01, 0xe0, 0x0c, 0xd6, 0x9b, 0xe8, 0x80,
						0x44, 0x70, 0xc1, 0xc3, 0x06, 0x0c, 0x18, 0x30,
						0x60, 0xc1, 0x83, 0x05, 0x0a, 0x14, 0x28, 0x50,
						0xa1, 0x42, 0x85, 0x0a, 0x14, 0x28, 0x50, 0xa1,
						0x42, 0x85, 0x0a, 0x14, 0x29, 0xb8, 0x10, 0x91,
					},
					{
						0x4c, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					},
				},
				9000,
				6000,
			},
			{
				[][]byte{
					{
						0x02, 0x01, 0xd0, 0x08, 0x39, 0x24, 0xb6, 0x08,
						0x84, 0x04, 0x84, 0x03, 0x83, 0x83, 0x83, 0x03,
						0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
						0x02, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
						0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
					},
					{
						0x4c, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					},
				},
				24000,
				9000,
			},
		},
	},
	{
		"libx265, bframes=4",
		[]sequenceSample{
			{
				[][]byte{
					{
						0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x01, 0x60,
						0x00, 0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00, 0x78, 0x95, 0x98, 0x09,
					},
					{
						0x42, 0x01, 0x01, 0x01, 0x60, 0x00, 0x00, 0x03,
						0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
						0x00, 0x78, 0xa0, 0x03, 0xc0, 0x80, 0x11, 0x07,
						0xcb, 0x96, 0x56, 0x64, 0xa4, 0xc2, 0xf0, 0x16,
						0x80, 0x80, 0x00, 0x00, 0x03, 0x00, 0x80, 0x00,
						0x00, 0x0f, 0x04,
					},
					{
						0x44, 0x01, 0xc0, 0x73, 0xc1, 0x89,
					},
					{
						0x28, 0x01, 0xac, 0x64, 0x11, 0x0b, 0x35, 0x45,
						0xb4, 0xa0, 0x95, 0x12, 0x90, 0x42, 0x48, 0xe1,
						0x2e, 0x21, 0xc4, 0x7c, 0xd4, 0x95, 0x63, 0x44,
						0x2f, 0x04, 0x98, 0x87, 0x03, 0xe2, 0xdc, 0x43,
						0x09, 0x61, 0x4c, 0x6f, 0x4a, 0x39, 0x6e, 0x27,
						0xb6, 0x50, 0x3b, 0x03, 0xd0, 0x45, 0x1f, 0x4c,
						0xf1, 0x66,
					},
				},
				0,
				-6000,
			},
			{
				[][]byte{
					{
						0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x01, 0x60,
						0x00, 0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00, 0x78, 0x95, 0x98, 0x09,
					},
					{
						0x42, 0x01, 0x01, 0x01, 0x60, 0x00, 0x00, 0x03,
						0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
						0x00, 0x78, 0xa0, 0x03, 0xc0, 0x80, 0x11, 0x07,
						0xcb, 0x96, 0x56, 0x64, 0xa4, 0xc2, 0xf0, 0x16,
						0x80, 0x80, 0x00, 0x00, 0x03, 0x00, 0x80, 0x00,
						0x00, 0x0f, 0x04,
					},
					{
						0x44, 0x01, 0xc0, 0x73, 0xc1, 0x89,
					},
					{
						0x2a, 0x01, 0xac, 0x14, 0xa5, 0x46, 0x01, 0x10,
						0xb2, 0xca, 0x41, 0x47, 0x18, 0xd2, 0x20, 0x02,
						0xec, 0x71, 0x8b, 0xd1, 0x58, 0x36, 0x88, 0xc9,
						0x1d, 0x2d, 0x82, 0x80, 0x37, 0x05, 0x80, 0x46,
						0x29, 0x43, 0x38, 0x71, 0x10, 0x05, 0x8c, 0x84,
						0x10, 0x61, 0xcd, 0x15, 0xc4, 0x20, 0x2c, 0x02,
						0x8e, 0x30,
					},
				},
				15000,
				-5910,
			},
			{
				[][]byte{
					{
						0x12, 0x01, 0xe0, 0x64, 0x9d, 0x60, 0x81, 0x40,
						0x22, 0x28, 0xc4, 0x21, 0x08, 0x42, 0x10, 0x84,
						0x20, 0x84, 0x21, 0x08, 0x42, 0x10, 0x84, 0x21,
						0x08, 0x78, 0x92, 0x0d, 0xca, 0x31, 0xa0, 0x92,
						0x80, 0x27, 0x20, 0x00, 0x05, 0x04, 0x00, 0x09,
						0x88, 0x00, 0x11, 0xf0, 0x00, 0x1d, 0xf0, 0x00,
						0x31, 0x20,
					},
				},
				9000,
				-2183,
			},
			{
				[][]byte{
					{
						0x10, 0x01, 0xe0, 0x24, 0xf5, 0x5e, 0x89, 0x02,
						0xd0, 0x44, 0x43, 0x22, 0x22, 0x22, 0x22, 0x21,
						0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x76,
						0x33, 0xa7, 0x22, 0x80, 0x83, 0x80, 0x3e, 0xe0,
						0x00, 0x05, 0xfc, 0x00, 0x0b, 0x78, 0x00, 0x15,
						0x10, 0x00, 0x24, 0xa0, 0x00, 0x36, 0xa0, 0x00,
						0x54, 0xc0,
					},
				},
				3000,
				3000,
			},
		},
	},
	{
		"b-frame IDR",
		[]sequenceSample{
			{
				[][]byte{
					{
						0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x01, 0x60,
						0x00, 0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03,
						0x00, 0x00, 0x03, 0x00, 0x78, 0x95, 0x98, 0x09,
					},
					{
						0x42, 0x01, 0x01, 0x01, 0x60, 0x00, 0x00, 0x03,
						0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
						0x00, 0x78, 0xa0, 0x03, 0xc0, 0x80, 0x32, 0x16,
						0x59, 0x59, 0xa4, 0x93, 0x2b, 0xc0, 0x5a, 0x80,
						0x80, 0x80, 0x82, 0x00, 0x00, 0x07, 0xd2, 0x00,
						0x00, 0xbb, 0x80, 0x10,
					},
					{
						0x44, 0x01, 0xc1, 0x72, 0xb4, 0x62, 0x40,
					},
					{
						0x2a, 0x01, 0xae, 0x44, 0x59, 0x48, 0xc9, 0xd4,
						0x68, 0xd1, 0x64, 0x7c, 0x76, 0x77, 0xce, 0x07,
						0x54, 0xca, 0x8f, 0x8d, 0x90, 0x33, 0x64, 0x52,
						0x7c, 0x48, 0x16, 0xc0, 0x9c, 0x80, 0xf8, 0x81,
						0x5b, 0xff, 0xf8, 0x76, 0x7c, 0x92, 0xd6, 0xb7,
						0xa8, 0x55, 0x9c, 0x20, 0xc0, 0x9b, 0x30, 0x11,
						0xdb, 0x80,
					},
				},
				2466180,
				2443658,
			},
			{
				[][]byte{
					{
						0x12, 0x01, 0xf1, 0xe2, 0x27, 0x52, 0xd7, 0xdc,
						0x21, 0xa3, 0x45, 0x03, 0x0c, 0x14, 0x30, 0x41,
						0x16, 0xe1, 0xa9, 0x46, 0x12, 0x06, 0xc1, 0xe0,
						0x14, 0x03, 0xc0, 0x92, 0x00, 0x8c, 0xb0, 0x66,
						0xc2, 0x1c, 0xf7, 0x66, 0xad, 0xf4, 0xcb, 0x6a,
						0x81, 0x90, 0xcc, 0xe9, 0xac, 0xc8, 0xec, 0x8d,
						0x3f, 0x30,
					},
				},
				2458710,
				2443748,
			},
			{
				[][]byte{
					{
						0x10, 0x01, 0xf1, 0xa6, 0xf5, 0x55, 0xf4, 0x89,
						0x04, 0x06, 0x89, 0x0e, 0xb7, 0xff, 0xa5, 0x33,
						0xa8, 0x5c, 0xee, 0x18, 0x0b, 0x06, 0xc1, 0x20,
						0x78, 0x91, 0x00, 0xce, 0xc0, 0xd7, 0x1c, 0x8d,
						0x93, 0x89, 0x2c, 0xb3, 0x03, 0xe8, 0xd1, 0xc3,
						0x4c, 0xe2, 0x9c, 0xee, 0xa9, 0xc2, 0x57, 0xc3,
						0x1c, 0x68,
					},
				},
				2451240,
				2451240,
			},
		},
	},
}

func TestDTSExtractor(t *testing.T) {
	for _, ca := range casesDTSExtractor {
		t.Run(ca.name, func(t *testing.T) {
			ex := NewDTSExtractor()
			for _, sample := range ca.sequence {
				dts, err := ex.Extract(sample.au, sample.pts)
				require.NoError(t, err)
				require.Equal(t, sample.dts, dts)
			}
		})
	}
}

func serializeSequence(seq []sequenceSample) []byte {
	var buf []byte

	for _, sample := range seq {
		tmp := make([]byte, 8)
		binary.LittleEndian.PutUint64(tmp, uint64(sample.pts))
		buf = append(buf, tmp...)

		au, _ := h264.AnnexB(sample.au).Marshal()

		tmp = make([]byte, 4)
		binary.LittleEndian.PutUint32(tmp, uint32(len(au)))
		buf = append(buf, tmp...)

		buf = append(buf, au...)
	}

	return buf
}

func unserializeSequence(buf []byte) ([]sequenceSample, error) {
	var samples []sequenceSample

	for {
		if len(buf) < 8 {
			return nil, fmt.Errorf("not enough bits")
		}
		pts := int64(binary.LittleEndian.Uint64(buf[:8]))
		buf = buf[8:]

		if len(buf) < 4 {
			return nil, fmt.Errorf("not enough bits")
		}
		auLen := binary.LittleEndian.Uint32(buf[:4])
		buf = buf[4:]

		if auLen == 0 {
			return nil, fmt.Errorf("invalid AU len")
		}
		if len(buf) < int(auLen) {
			return nil, fmt.Errorf("not enough bits")
		}
		rawAu := buf[:auLen]
		buf = buf[auLen:]

		var au h264.AnnexB
		err := au.Unmarshal(rawAu)
		if err != nil {
			return nil, fmt.Errorf("not enough bits")
		}

		samples = append(samples, sequenceSample{
			au:  au,
			pts: pts,
		})

		if len(buf) == 0 {
			break
		}
	}

	return samples, nil
}

func FuzzDTSExtractor(f *testing.F) {
	for _, ca := range casesDTSExtractor {
		f.Add(serializeSequence(ca.sequence))
	}

	f.Fuzz(func(t *testing.T, buf []byte) {
		seq, err := unserializeSequence(buf)
		if err != nil {
			t.Skip()
			return
		}

		ex := NewDTSExtractor()
		for _, sample := range seq {
			_, err = ex.Extract(sample.au, sample.pts)
			if err != nil {
				break
			}
		}
	})
}
