package rtpmjpeg

import (
	"errors"
	"testing"

	"github.com/pion/rtp"
	"github.com/stretchr/testify/require"
)

func TestDecode(t *testing.T) {
	for _, ca := range cases {
		t.Run(ca.name, func(t *testing.T) {
			d := &Decoder{}
			err := d.Init()
			require.NoError(t, err)

			for _, pkt := range ca.pkts {
				var image []byte
				image, err = d.Decode(pkt)
				if errors.Is(err, ErrMorePacketsNeeded) {
					continue
				}

				require.NoError(t, err)
				require.Equal(t, ca.image, image)
			}
		})
	}
}

func TestDecodeFixedQuantizationTable(t *testing.T) {
	d := &Decoder{}
	err := d.Init()
	require.NoError(t, err)

	image, err := d.Decode(&rtp.Packet{
		Header: rtp.Header{
			Version:        2,
			Marker:         true,
			PayloadType:    26,
			SequenceNumber: 17645,
			SSRC:           0x9dbb7812,
		},
		Payload: []byte{
			// JPEG header
			0x00, 0x00, 0x00, 0x00, 0x01, 63, 0xf0, 0x87,
			// JPEG data
			1, 2,
		},
	})
	require.NoError(t, err)

	require.Equal(t, []byte{
		0xff, 0xd8, 0xff, 0xdb, 0x00, 0x84, 0x00, 0x0c,
		0x08, 0x09, 0x0a, 0x09, 0x07, 0x0c, 0x0a, 0x0a,
		0x0a, 0x0d, 0x0d, 0x0c, 0x0e, 0x12, 0x1e, 0x13,
		0x12, 0x10, 0x10, 0x12, 0x24, 0x1a, 0x1b, 0x15,
		0x1e, 0x2b, 0x26, 0x2d, 0x2c, 0x2a, 0x26, 0x29,
		0x29, 0x2f, 0x35, 0x44, 0x3a, 0x2f, 0x32, 0x40,
		0x33, 0x29, 0x29, 0x3b, 0x51, 0x3c, 0x40, 0x46,
		0x49, 0x4c, 0x4d, 0x4c, 0x2e, 0x39, 0x54, 0x5a,
		0x53, 0x4a, 0x59, 0x44, 0x4b, 0x4c, 0x49, 0x01,
		0x0d, 0x0d, 0x0d, 0x12, 0x10, 0x12, 0x23, 0x13,
		0x13, 0x23, 0x49, 0x31, 0x29, 0x31, 0x49, 0x49,
		0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49,
		0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49,
		0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49,
		0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49,
		0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49,
		0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49,
		0xff, 0xc0, 0x00, 0x11, 0x08, 0x04, 0x38, 0x07,
		0x80, 0x03, 0x00, 0x22, 0x00, 0x01, 0x11, 0x01,
		0x02, 0x11, 0x01, 0xff, 0xc4, 0x00, 0x1f, 0x00,
		0x00, 0x01, 0x05, 0x01, 0x01, 0x01, 0x01, 0x01,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0xff, 0xc4, 0x00, 0xb5,
		0x10, 0x00, 0x02, 0x01, 0x03, 0x03, 0x02, 0x04,
		0x03, 0x05, 0x05, 0x04, 0x04, 0x00, 0x00, 0x01,
		0x7d, 0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05,
		0x12, 0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61,
		0x07, 0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1,
		0x08, 0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1,
		0xf0, 0x24, 0x33, 0x62, 0x72, 0x82, 0x09, 0x0a,
		0x16, 0x17, 0x18, 0x19, 0x1a, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38,
		0x39, 0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
		0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
		0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
		0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
		0x79, 0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88,
		0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
		0x98, 0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6,
		0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5,
		0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4,
		0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3,
		0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xe1,
		0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
		0xea, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9, 0xfa, 0xff, 0xc4, 0x00, 0x1f, 0x01,
		0x00, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
		0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0xff, 0xc4, 0x00, 0xb5,
		0x11, 0x00, 0x02, 0x01, 0x02, 0x04, 0x04, 0x03,
		0x04, 0x07, 0x05, 0x04, 0x04, 0x00, 0x01, 0x02,
		0x77, 0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05,
		0x21, 0x31, 0x06, 0x12, 0x41, 0x51, 0x07, 0x61,
		0x71, 0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42,
		0x91, 0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33, 0x52,
		0xf0, 0x15, 0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24,
		0x34, 0xe1, 0x25, 0xf1, 0x17, 0x18, 0x19, 0x1a,
		0x26, 0x27, 0x28, 0x29, 0x2a, 0x35, 0x36, 0x37,
		0x38, 0x39, 0x3a, 0x43, 0x44, 0x45, 0x46, 0x47,
		0x48, 0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57,
		0x58, 0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67,
		0x68, 0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77,
		0x78, 0x79, 0x7a, 0x82, 0x83, 0x84, 0x85, 0x86,
		0x87, 0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95,
		0x96, 0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3, 0xa4,
		0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3,
		0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2,
		0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca,
		0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9,
		0xda, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8,
		0xe9, 0xea, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9, 0xfa, 0xff, 0xda, 0x00, 0x0c, 0x03,
		0x00, 0x00, 0x01, 0x11, 0x02, 0x11, 0x00, 0x3f,
		0x00, 0x01, 0x02, 0xff, 0xd9,
	}, image)
}

func FuzzDecoder(f *testing.F) {
	f.Fuzz(func(t *testing.T, a []byte, am bool, b []byte, bm bool) {
		d := &Decoder{}
		err := d.Init()
		require.NoError(t, err)

		d.Decode(&rtp.Packet{ //nolint:errcheck
			Header: rtp.Header{
				Version:        2,
				Marker:         am,
				PayloadType:    96,
				SequenceNumber: 17645,
				Timestamp:      2289527317,
				SSRC:           0x9dbb7812,
			},
			Payload: a,
		})

		d.Decode(&rtp.Packet{ //nolint:errcheck
			Header: rtp.Header{
				Version:        2,
				Marker:         bm,
				PayloadType:    96,
				SequenceNumber: 17645,
				Timestamp:      2289527317,
				SSRC:           0x9dbb7812,
			},
			Payload: b,
		})
	})
}
