package handshake

import (
	"bytes"
	"crypto/rand"
	"fmt"
	"math/big"
)

const (
	dhKeyLength = 128
)

var (
	p1024 = []byte{
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xc9, 0x0f, 0xda, 0xa2, 0x21, 0x68, 0xc2, 0x34,
		0xc4, 0xc6, 0x62, 0x8b, 0x80, 0xdc, 0x1c, 0xd1,
		0x29, 0x02, 0x4e, 0x08, 0x8a, 0x67, 0xcc, 0x74,
		0x02, 0x0b, 0xbe, 0xa6, 0x3b, 0x13, 0x9b, 0x22,
		0x51, 0x4a, 0x08, 0x79, 0x8e, 0x34, 0x04, 0xdd,
		0xef, 0x95, 0x19, 0xb3, 0xcd, 0x3a, 0x43, 0x1b,
		0x30, 0x2b, 0x0a, 0x6d, 0xf2, 0x5f, 0x14, 0x37,
		0x4f, 0xe1, 0x35, 0x6d, 0x6d, 0x51, 0xc2, 0x45,
		0xe4, 0x85, 0xb5, 0x76, 0x62, 0x5e, 0x7e, 0xc6,
		0xf4, 0x4c, 0x42, 0xe9, 0xa6, 0x37, 0xed, 0x6b,
		0x0b, 0xff, 0x5c, 0xb6, 0xf4, 0x06, 0xb7, 0xed,
		0xee, 0x38, 0x6b, 0xfb, 0x5a, 0x89, 0x9f, 0xa5,
		0xae, 0x9f, 0x24, 0x11, 0x7c, 0x4b, 0x1f, 0xe6,
		0x49, 0x28, 0x66, 0x51, 0xec, 0xe6, 0x53, 0x81,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	}
	q1024 = []byte{
		0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xe4, 0x87, 0xed, 0x51, 0x10, 0xb4, 0x61, 0x1a,
		0x62, 0x63, 0x31, 0x45, 0xc0, 0x6e, 0x0e, 0x68,
		0x94, 0x81, 0x27, 0x04, 0x45, 0x33, 0xe6, 0x3a,
		0x01, 0x05, 0xdf, 0x53, 0x1d, 0x89, 0xcd, 0x91,
		0x28, 0xa5, 0x04, 0x3c, 0xc7, 0x1a, 0x02, 0x6e,
		0xf7, 0xca, 0x8c, 0xd9, 0xe6, 0x9d, 0x21, 0x8d,
		0x98, 0x15, 0x85, 0x36, 0xf9, 0x2f, 0x8a, 0x1b,
		0xa7, 0xf0, 0x9a, 0xb6, 0xb6, 0xa8, 0xe1, 0x22,
		0xf2, 0x42, 0xda, 0xbb, 0x31, 0x2f, 0x3f, 0x63,
		0x7a, 0x26, 0x21, 0x74, 0xd3, 0x1b, 0xf6, 0xb5,
		0x85, 0xff, 0xae, 0x5b, 0x7a, 0x03, 0x5b, 0xf6,
		0xf7, 0x1c, 0x35, 0xfd, 0xad, 0x44, 0xcf, 0xd2,
		0xd7, 0x4f, 0x92, 0x08, 0xbe, 0x25, 0x8f, 0xf3,
		0x24, 0x94, 0x33, 0x28, 0xf6, 0x73, 0x29, 0xc0,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	}
)

// https://datatracker.ietf.org/doc/html/rfc2631#section-2.1.5
func dhValidatePublicKey(key []byte) error {
	// 1. y >= 2 && y < p
	var y big.Int
	y.SetBytes(key)
	var two big.Int
	two.SetUint64(2)
	r := y.Cmp(&two)
	if r < 0 {
		return fmt.Errorf("key is < 2")
	}
	var p big.Int
	p.SetBytes(p1024)
	r = y.Cmp(&p)
	if r >= 0 {
		return fmt.Errorf("key is >= p")
	}

	// 2. (y^q mod p) == 1
	var q big.Int
	q.SetBytes(q1024)
	var z big.Int
	z.Exp(&y, &q, &p)
	var one big.Int
	one.SetUint64(1)
	r = z.Cmp(&one)
	if r != 0 {
		return fmt.Errorf("y^q mod p is != 1")
	}

	return nil
}

func dhGenerateKeyPair() ([]byte, []byte, error) {
	priv := make([]byte, dhKeyLength)
	_, err := rand.Read(priv)
	if err != nil {
		return nil, nil, err
	}

	// y = g ^ x mod p
	var g big.Int
	g.SetUint64(2)
	var x big.Int
	x.SetBytes(priv)
	var p big.Int
	p.SetBytes(p1024)
	var y big.Int
	y.Exp(&g, &x, &p)
	pub := y.Bytes()

	if len(pub) < dhKeyLength {
		pub = append(bytes.Repeat([]byte{0}, dhKeyLength-len(pub)), pub...)
	}

	return priv, pub, nil
}

// https://datatracker.ietf.org/doc/html/rfc2631#section-2.1.1
func dhComputeSharedSecret(priv []byte, pub []byte) []byte {
	// ZZ = (ya ^ xb)  mod p
	var y big.Int
	y.SetBytes(pub)
	var x big.Int
	x.SetBytes(priv)
	var p big.Int
	p.SetBytes(p1024)
	var z big.Int
	z.Exp(&y, &x, &p)
	sec := z.Bytes()

	if len(sec) < dhKeyLength {
		sec = append(bytes.Repeat([]byte{0}, dhKeyLength-len(sec)), sec...)
	}

	return sec
}
