package mjpeg

// RFC 2435. Appendix A

// don't know why two tables are not respect RFC
// https://github.com/FFmpeg/FFmpeg/blob/master/libavformat/rtpdec_jpeg.c

var jpeg_luma_quantizer = [64]byte{
	16, 11, 12, 14, 12, 10, 16, 14,
	13, 14, 18, 17, 16, 19, 24, 40,
	26, 24, 22, 22, 24, 49, 35, 37,
	29, 40, 58, 51, 61, 60, 57, 51,
	56, 55, 64, 72, 92, 78, 64, 68,
	87, 69, 55, 56, 80, 109, 81, 87,
	95, 98, 103, 104, 103, 62, 77, 113,
	121, 112, 100, 120, 92, 101, 103, 99,
}
var jpeg_chroma_quantizer = [64]byte{
	17, 18, 18, 24, 21, 24, 47, 26,
	26, 47, 99, 66, 56, 66, 99, 99,
	99, 99, 99, 99, 99, 99, 99, 99,
	99, 99, 99, 99, 99, 99, 99, 99,
	99, 99, 99, 99, 99, 99, 99, 99,
	99, 99, 99, 99, 99, 99, 99, 99,
	99, 99, 99, 99, 99, 99, 99, 99,
	99, 99, 99, 99, 99, 99, 99, 99,
}

func MakeTables(q byte) (lqt, cqt []byte) {
	var factor int

	switch {
	case q < 1:
		factor = 1
	case q > 99:
		factor = 99
	default:
		factor = int(q)
	}

	if q < 50 {
		factor = 5000 / factor
	} else {
		factor = 200 - factor*2
	}

	lqt = make([]byte, 64)
	cqt = make([]byte, 64)

	for i := 0; i < 64; i++ {
		lq := (int(jpeg_luma_quantizer[i])*factor + 50) / 100
		cq := (int(jpeg_chroma_quantizer[i])*factor + 50) / 100

		/* Limit the quantizers to 1 <= q <= 255 */
		switch {
		case lq < 1:
			lqt[i] = 1
		case lq > 255:
			lqt[i] = 255
		default:
			lqt[i] = byte(lq)
		}

		switch {
		case cq < 1:
			cqt[i] = 1
		case cq > 255:
			cqt[i] = 255
		default:
			cqt[i] = byte(cq)
		}
	}

	return
}

// RFC 2435. Appendix B

var lum_dc_codelens = []byte{
	0, 1, 5, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
}
var lum_dc_symbols = []byte{
	0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11,
}
var lum_ac_codelens = []byte{
	0, 2, 1, 3, 3, 2, 4, 3, 5, 5, 4, 4, 0, 0, 1, 0x7d,
}
var lum_ac_symbols = []byte{
	0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12,
	0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07,
	0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1, 0x08,
	0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0,
	0x24, 0x33, 0x62, 0x72, 0x82, 0x09, 0x0a, 0x16,
	0x17, 0x18, 0x19, 0x1a, 0x25, 0x26, 0x27, 0x28,
	0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
	0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
	0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
	0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
	0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79,
	0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
	0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
	0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6,
	0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5,
	0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4,
	0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xe1, 0xe2,
	0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea,
	0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
	0xf9, 0xfa,
}
var chm_dc_codelens = []byte{
	0, 3, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
}
var chm_dc_symbols = []byte{
	0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11,
}
var chm_ac_codelens = []byte{
	0, 2, 1, 2, 4, 4, 3, 4, 7, 5, 4, 4, 0, 1, 2, 0x77,
}
var chm_ac_symbols = []byte{
	0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21,
	0x31, 0x06, 0x12, 0x41, 0x51, 0x07, 0x61, 0x71,
	0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42, 0x91,
	0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33, 0x52, 0xf0,
	0x15, 0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24, 0x34,
	0xe1, 0x25, 0xf1, 0x17, 0x18, 0x19, 0x1a, 0x26,
	0x27, 0x28, 0x29, 0x2a, 0x35, 0x36, 0x37, 0x38,
	0x39, 0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
	0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
	0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
	0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
	0x79, 0x7a, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96,
	0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5,
	0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4,
	0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3,
	0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2,
	0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda,
	0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
	0xea, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
	0xf9, 0xfa,
}

func MakeHeaders(p []byte, t byte, w, h uint16, lqt, cqt []byte) []byte {
	// Appendix A from https://www.rfc-editor.org/rfc/rfc2435
	p = append(p, 0xFF, markerSOI)

	p = MakeQuantHeader(p, lqt, 0)
	p = MakeQuantHeader(p, cqt, 1)

	if t == 0 {
		t = 0x21 // hsamp = 2, vsamp = 1
	} else {
		t = 0x22 // hsamp = 2, vsamp = 2
	}

	p = append(p, 0xFF, markerSOF,
		0, 17, // size
		8, // bits per component
		byte(h>>8), byte(h&0xFF),
		byte(w>>8), byte(w&0xFF),
		3, // number of components
		0, // comp 0
		t,
		0,    // quant table 0
		1,    // comp 1
		0x11, // hsamp = 1, vsamp = 1
		1,    // quant table 1
		2,    // comp 2
		0x11, // hsamp = 1, vsamp = 1
		1,    // quant table 1
	)

	p = MakeHuffmanHeaders(p)

	return append(p, 0xFF, markerSOS,
		0, 12, // size
		3,    // 3 components
		0,    // comp 0
		0,    // huffman table 0
		1,    // comp 1
		0x11, // huffman table 1
		2,    // comp 2
		0x11, // huffman table 1
		0,    // first DCT coeff
		63,   // last DCT coeff
		0,    // sucessive approx
	)
}

func MakeQuantHeader(p []byte, qt []byte, tableNo byte) []byte {
	p = append(p, 0xFF, markerDQT, 0, 67, tableNo)
	return append(p, qt...)
}

func MakeHuffmanHeader(p, codelens, symbols []byte, tableNo, tableClass byte) []byte {
	p = append(p, 0xFF, markerDHT,
		0, byte(3+len(codelens)+len(symbols)), // size
		(tableClass<<4)|tableNo,
	)
	p = append(p, codelens...)
	return append(p, symbols...)
}

func MakeHuffmanHeaders(p []byte) []byte {
	p = MakeHuffmanHeader(p, lum_dc_codelens, lum_dc_symbols, 0, 0)
	p = MakeHuffmanHeader(p, lum_ac_codelens, lum_ac_symbols, 0, 1)
	p = MakeHuffmanHeader(p, chm_dc_codelens, chm_dc_symbols, 1, 0)
	p = MakeHuffmanHeader(p, chm_ac_codelens, chm_ac_symbols, 1, 1)
	return p
}
