package forge_test

import (
	"fmt"
	"testing"

	"github.com/shoenig/test/must"

	"github.com/git-town/git-town/v22/internal/config/configdomain"
	"github.com/git-town/git-town/v22/internal/forge"
	"github.com/git-town/git-town/v22/internal/forge/forgedomain"
	"github.com/git-town/git-town/v22/internal/git/gitdomain"
	. "github.com/git-town/git-town/v22/pkg/prelude"
)

type mockProposalFinder struct{}

func (self *mockProposalFinder) FindProposal(branch, target gitdomain.LocalBranchName) (Option[forgedomain.Proposal], error) {
	var prNumber int
	for _, char := range branch {
		prNumber += int(char)
	}
	return Some(forgedomain.Proposal{
		Data: forgedomain.ProposalData{
			Body:         None[string](),
			MergeWithAPI: false,
			Number:       1,
			Source:       branch,
			Target:       target,
			Title:        "Test Mocker",
			URL:          fmt.Sprintf("https://www.github.com/git-town/pull/%d", prNumber),
		},
		ForgeType: forgedomain.ForgeTypeForgejo,
	}), nil
}

func TestProposalStackLineageBuilder_CheckLineageAndProposals(t *testing.T) {
	t.Parallel()
	// arrange
	mainBranch := gitdomain.NewLocalBranchName("main")
	featureBranchA := gitdomain.NewLocalBranchName("a")
	featureBranchB := gitdomain.NewLocalBranchName("b")
	lineage := configdomain.NewLineageWith(configdomain.LineageData{
		featureBranchA: mainBranch,
		featureBranchB: featureBranchA,
	})
	var connector forgedomain.ProposalFinder = &mockProposalFinder{}
	args := forge.ProposalStackLineageArgs{
		Connector:                connector,
		CurrentBranch:            featureBranchA,
		Lineage:                  lineage,
		MainAndPerennialBranches: gitdomain.NewLocalBranchNames(mainBranch.String()),
	}
	tree, err := forge.NewProposalStackLineageTree(args)
	must.NoError(t, err)
	must.True(t, tree.BranchToProposal[mainBranch].IsNone())
	must.True(t, tree.BranchToProposal[featureBranchB].IsSome())
	must.True(t, tree.BranchToProposal[featureBranchA].IsSome())
	want := `

-------------------------
 - main
   - https://www.github.com/git-town/pull/97 :point_left:
     - https://www.github.com/git-town/pull/98

Stack generated by [Git Town](https://github.com/git-town/git-town)
`[1:]

	// act
	have := forge.NewProposalStackLineageBuilder(args, MutableSome(tree))

	// assert
	builder, hasBuilder := have.Get()
	must.True(t, hasBuilder)
	actual := builder.Build(args)
	must.EqOp(t, want, actual)
}

func TestProposalStackLineageBuilder_CurrentBranchHasMultipleDirectDescendants(t *testing.T) {
	t.Parallel()
	// arrange
	/*
				git-town branch:
					main
							a
									b
											c
									d
							e
									f
		There is a stack: main -> a -> [b -> c, d]
		There is a separate stack: main -> e -> f
	*/
	mainBranch := gitdomain.NewLocalBranchName("main")
	featureBranchA := gitdomain.NewLocalBranchName("a")
	featureBranchB := gitdomain.NewLocalBranchName("b")
	featureBranchC := gitdomain.NewLocalBranchName("c")
	featureBranchD := gitdomain.NewLocalBranchName("d")
	featureBranchE := gitdomain.NewLocalBranchName("e")
	featureBranchF := gitdomain.NewLocalBranchName("f")
	lineage := configdomain.NewLineageWith(configdomain.LineageData{
		// first stack
		featureBranchA: mainBranch,
		featureBranchB: featureBranchA,
		featureBranchC: featureBranchB,
		featureBranchD: featureBranchA,
		// second stack
		featureBranchE: mainBranch,
		featureBranchF: featureBranchE,
	})

	t.Run("Stack Proposal For Branch A", func(t *testing.T) {
		t.Parallel()
		var connector forgedomain.ProposalFinder = &mockProposalFinder{}
		args := forge.ProposalStackLineageArgs{
			Connector:                connector,
			CurrentBranch:            featureBranchA,
			Lineage:                  lineage,
			MainAndPerennialBranches: gitdomain.NewLocalBranchNames(mainBranch.String()),
		}
		want := `
-------------------------
 - main
   - https://www.github.com/git-town/pull/97 :point_left:
     - https://www.github.com/git-town/pull/98
       - https://www.github.com/git-town/pull/99
     - https://www.github.com/git-town/pull/100

Stack generated by [Git Town](https://github.com/git-town/git-town)
`
		have := forge.NewProposalStackLineageBuilder(args, MutableNone[forge.ProposalStackLineageTree]())
		builder, hasBuilder := have.Get()
		must.True(t, hasBuilder)
		// actual
		actual := builder.Build(args)
		// assert
		must.EqOp(t, want, actual)
	})

	t.Run("Stack Proposal For Branch B", func(t *testing.T) {
		t.Parallel()
		var connector forgedomain.ProposalFinder = &mockProposalFinder{}
		args := forge.ProposalStackLineageArgs{
			Connector:                connector,
			CurrentBranch:            featureBranchB,
			Lineage:                  lineage,
			MainAndPerennialBranches: gitdomain.NewLocalBranchNames(mainBranch.String()),
		}
		want := `
-------------------------
 - main
   - https://www.github.com/git-town/pull/97
     - https://www.github.com/git-town/pull/98 :point_left:
       - https://www.github.com/git-town/pull/99
     - https://www.github.com/git-town/pull/100

Stack generated by [Git Town](https://github.com/git-town/git-town)
`
		have := forge.NewProposalStackLineageBuilder(args, MutableNone[forge.ProposalStackLineageTree]())
		builder, hasBuilder := have.Get()
		must.True(t, hasBuilder)
		// actual
		actual := builder.Build(args)
		// assert
		must.EqOp(t, want, actual)
	})

	t.Run("Stack Proposal For Branch C", func(t *testing.T) {
		t.Parallel()
		var connector forgedomain.ProposalFinder = &mockProposalFinder{}
		args := forge.ProposalStackLineageArgs{
			Connector:                connector,
			CurrentBranch:            featureBranchC,
			Lineage:                  lineage,
			MainAndPerennialBranches: gitdomain.NewLocalBranchNames(mainBranch.String()),
		}
		want := `
-------------------------
 - main
   - https://www.github.com/git-town/pull/97
     - https://www.github.com/git-town/pull/98
       - https://www.github.com/git-town/pull/99 :point_left:
     - https://www.github.com/git-town/pull/100

Stack generated by [Git Town](https://github.com/git-town/git-town)
`
		have := forge.NewProposalStackLineageBuilder(args, MutableNone[forge.ProposalStackLineageTree]())
		builder, hasBuilder := have.Get()
		must.True(t, hasBuilder)
		// actual
		actual := builder.Build(args)
		// assert
		must.EqOp(t, want, actual)
	})

	t.Run("Stack Proposal For Branch D", func(t *testing.T) {
		t.Parallel()
		var connector forgedomain.ProposalFinder = &mockProposalFinder{}
		args := forge.ProposalStackLineageArgs{
			Connector:                connector,
			CurrentBranch:            featureBranchD,
			Lineage:                  lineage,
			MainAndPerennialBranches: gitdomain.NewLocalBranchNames(mainBranch.String()),
		}
		want := `
-------------------------
 - main
   - https://www.github.com/git-town/pull/97
     - https://www.github.com/git-town/pull/100 :point_left:
     - https://www.github.com/git-town/pull/98
       - https://www.github.com/git-town/pull/99

Stack generated by [Git Town](https://github.com/git-town/git-town)
`
		have := forge.NewProposalStackLineageBuilder(args, MutableNone[forge.ProposalStackLineageTree]())
		builder, hasBuilder := have.Get()
		must.True(t, hasBuilder)
		// actual
		actual := builder.Build(args)
		// assert
		must.EqOp(t, want, actual)
	})

	t.Run("Stack Proposal For Branch E", func(t *testing.T) {
		t.Parallel()
		var connector forgedomain.ProposalFinder = &mockProposalFinder{}
		args := forge.ProposalStackLineageArgs{
			Connector:                connector,
			CurrentBranch:            featureBranchE,
			Lineage:                  lineage,
			MainAndPerennialBranches: gitdomain.NewLocalBranchNames(mainBranch.String()),
		}
		want := `
-------------------------
 - main
   - https://www.github.com/git-town/pull/101 :point_left:
     - https://www.github.com/git-town/pull/102

Stack generated by [Git Town](https://github.com/git-town/git-town)
`
		have := forge.NewProposalStackLineageBuilder(args, MutableNone[forge.ProposalStackLineageTree]())
		builder, hasBuilder := have.Get()
		must.True(t, hasBuilder)
		// actual
		actual := builder.Build(args)
		// assert
		must.EqOp(t, want, actual)
	})

	t.Run("Stack Proposal For Branch F", func(t *testing.T) {
		t.Parallel()
		var connector forgedomain.ProposalFinder = &mockProposalFinder{}
		args := forge.ProposalStackLineageArgs{
			Connector:                connector,
			CurrentBranch:            featureBranchF,
			Lineage:                  lineage,
			MainAndPerennialBranches: gitdomain.NewLocalBranchNames(mainBranch.String()),
		}
		want := `
-------------------------
 - main
   - https://www.github.com/git-town/pull/101
     - https://www.github.com/git-town/pull/102 :point_left:

Stack generated by [Git Town](https://github.com/git-town/git-town)
`
		have := forge.NewProposalStackLineageBuilder(args, MutableNone[forge.ProposalStackLineageTree]())
		builder, hasBuilder := have.Get()
		must.True(t, hasBuilder)
		// actual
		actual := builder.Build(args)
		// assert
		must.EqOp(t, want, actual)
	})
}

func TestProposalStackLineageBuilder_NoLineage(t *testing.T) {
	t.Parallel()
	// arrange
	lineage := configdomain.NewLineageWith(configdomain.LineageData{})
	var connector forgedomain.ProposalFinder = &mockProposalFinder{}
	args := forge.ProposalStackLineageArgs{
		Connector:                connector,
		CurrentBranch:            gitdomain.NewLocalBranchName("main"),
		Lineage:                  lineage,
		MainAndPerennialBranches: lineage.Roots(),
	}
	expected := None[forge.ProposalStackLineageBuilder]()

	// act
	actual := forge.NewProposalStackLineageBuilder(args, MutableNone[forge.ProposalStackLineageTree]())

	// assert
	must.EqOp(t, expected, actual)
}

func TestProposalStackLineageBuilder_NoLineageForMainAndPerennialBranches(t *testing.T) {
	t.Parallel()
	// arrange
	mainBranch := gitdomain.NewLocalBranchName("main")
	featureBranchA := gitdomain.NewLocalBranchName("a")
	lineage := configdomain.NewLineageWith(configdomain.LineageData{
		featureBranchA: mainBranch,
	})
	var connector forgedomain.ProposalFinder = &mockProposalFinder{}
	args := forge.ProposalStackLineageArgs{
		Connector:                connector,
		CurrentBranch:            mainBranch,
		Lineage:                  lineage,
		MainAndPerennialBranches: lineage.Roots(),
	}
	expected := None[forge.ProposalStackLineageBuilder]()

	// act
	actual := forge.NewProposalStackLineageBuilder(args, MutableNone[forge.ProposalStackLineageTree]())

	// assert
	must.EqOp(t, expected, actual)
}

func TestProposalStackLineageBuilder_SwapBranch(t *testing.T) {
	t.Parallel()
	// arrange
	mainBranch := gitdomain.NewLocalBranchName("main")
	featureBranchA := gitdomain.NewLocalBranchName("a")
	featureBranchB := gitdomain.NewLocalBranchName("b")
	lineage := configdomain.NewLineageWith(configdomain.LineageData{
		featureBranchA: mainBranch,
		featureBranchB: featureBranchA,
	})
	var connector forgedomain.ProposalFinder = &mockProposalFinder{}
	args := forge.ProposalStackLineageArgs{
		Connector:                connector,
		CurrentBranch:            featureBranchA,
		Lineage:                  lineage,
		MainAndPerennialBranches: gitdomain.NewLocalBranchNames(mainBranch.String()),
	}

	tree, err := forge.NewProposalStackLineageTree(args)
	must.NoError(t, err)
	builder, hasBuilder := forge.NewProposalStackLineageBuilder(args, MutableSome(tree)).Get()
	must.True(t, hasBuilder)
	original := `

-------------------------
 - main
   - https://www.github.com/git-town/pull/97 :point_left:
     - https://www.github.com/git-town/pull/98

Stack generated by [Git Town](https://github.com/git-town/git-town)
`[1:]
	must.EqOp(t, original, builder.Build(args))

	/*
			*  - update lineage - swap
		  *  - update stack within the builder
	*/
	lineage.Set(featureBranchB, mainBranch)
	lineage.Set(featureBranchA, featureBranchB)
	args.Lineage = lineage
	must.NoError(t, builder.UpdateStack(args))
	want := `

-------------------------
 - main
   - https://www.github.com/git-town/pull/98
     - https://www.github.com/git-town/pull/97 :point_left:

Stack generated by [Git Town](https://github.com/git-town/git-town)
`[1:]

	// act
	actual := builder.Build(args)

	// assert
	must.EqOp(t, want, actual)
}
