// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0 OR ISC

#include <vector>

#include <openssl/err.h>
#include <openssl/mem.h>
#include <openssl/ocsp.h>
#include "../crypto/ocsp/internal.h"

static const uint8_t kOCSPRequestDER[] = {
    0x30, 0x68, 0x30, 0x66, 0x30, 0x3f, 0x30, 0x3d, 0x30, 0x3b, 0x30,
    0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04,
    0x14, 0xde, 0x79, 0x32, 0xb3, 0x21, 0x7e, 0x48, 0xfb, 0x4e, 0x47,
    0xae, 0x0b, 0x90, 0x07, 0xa5, 0x53, 0x76, 0xae, 0x44, 0xca, 0x04,
    0x14, 0x12, 0xdf, 0x81, 0x75, 0x71, 0xca, 0x92, 0xd3, 0xce, 0x1b,
    0x2c, 0x2b, 0x77, 0x3b, 0x9e, 0x33, 0x77, 0xf3, 0xf7, 0x6f, 0x02,
    0x02, 0x77, 0x78, 0xa2, 0x23, 0x30, 0x21, 0x30, 0x1f, 0x06, 0x09,
    0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x02, 0x04, 0x12,
    0x04, 0x10, 0x30, 0x3f, 0x12, 0x8c, 0xd8, 0x24, 0xa2, 0xb4, 0x65,
    0xf4, 0xc8, 0x46, 0x88, 0x2b, 0x3e, 0x1f
};

extern "C" int LLVMFuzzerTestOneInput(const uint8_t *buf, size_t len) {
  bssl::UniquePtr<BIO> bio(BIO_new(BIO_s_mem()));
  BIO_write(bio.get(), buf, len);
  // |OCSP_REQ_CTX| needs a dummy OCSP_REQUEST to be sent out, before an
  // HTTP OCSP response can be expected.
  const uint8_t *data = kOCSPRequestDER;
  bssl::UniquePtr<OCSP_REQUEST> request(
      d2i_OCSP_REQUEST(nullptr, &data, sizeof(kOCSPRequestDER)));
  OCSP_REQ_CTX *req_ctx =
      OCSP_sendreq_new(bio.get(), nullptr, request.get(), 0);

  // |OCSP_REQ_CTX_nbio| sends the dummy OCSP request in the BIO and parses the
  // contents in |bio|.
  if (OCSP_REQ_CTX_nbio(req_ctx) > 0) {
    // Check that there are contents.
    const uint8_t *contents;
    size_t outlen;
    if(!BIO_mem_contents(req_ctx->mem, &contents, &outlen)){
      // This code block shouldn't be reached. |req_ctx->mem| should always have
      // contents if |OCSP_REQ_CTX_nbio| returns 1.
      return 1;
    }
  }
  OCSP_REQ_CTX_free(req_ctx);
  ERR_clear_error();
  return 0;
}
